/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck;

import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.selfcheck.monitor.TableSizeMonitor.ThresholdType;

/**
 * セルフチェック機能の設定定義クラス
 */
public class SelfCheckConfig {

	public static final Log log = LogFactory.getLog(SelfCheckConfig.class);

	public static final String _configFilePath;
	public static final String _configFilePathDefault = System.getProperty("hinemos.manager.etc.dir") + File.separator + "selfcheck-service.properties";
	public static Properties properties = null;

	// SelfCheck Service Attribute
	public static volatile Date lastMonitorDate = null;

	public static final int startupDelaySec;
	public static final int intervalSec;
	public static final int alertThreshold;
	public static final int threadPoolSize;

	// SNMP Configuration to 127.0.0.1 (Hinemos Manager)
	public static final String snmpCommunity;
	public static final int snmpPort;
	public static final String snmpVersion;
	public static final int snmpTimeoutMSec;
	public static final int snmpRetries;

	// monitor java vm heap
	public static final boolean jvmHeapMonitoring;
	public static final int jvmHeapThresholdMByte;

	// monitor filesystem (by snmp)
	public static final boolean fsUsageMonitoring;
	public static final List<FileSystemUsageConfig> fsUsageList;

	// monitor ram memory (by snmp)
	public static final boolean swapOutMonitoring;

	// monitor database
	public static final boolean dbMonitoring;
	public static final String dbValidationQuery;

	// monitor java scheduler
	public static final boolean schedulerMonitoring;
	public static final int schedulerDelayThresholdSec;

	// monitor jaxws
	public static final boolean wsQueueMonitoring;
	public static final int wsQueueThreshold;

	// monitor syslog/snmptrap queue
	public static final boolean syslogQueueMonitoring;
	public static final int syslogQueueThreshold;
	public static final boolean snmptrapQueueMonitoring;
	public static final int snmptrapQueueThreshold;

	// monitor asynchronous task
	public static final boolean asyncTaskMonitoring;
	public static final List<AsyncTaskQueueConfig> asyncTaskList;

	// monitor slow thread
	public static final boolean threadActivityMonitoring;
	public static final int threadRunningTimeThreshold;

	// monitor log table
	public static final boolean tableSizeMonitoring;

	public static final List<TableSizeConfig> tableList;

	// monitor monitor job
	public static final boolean jobRunningSessionMonitoring;
	public static final long jobRunningSessionThreshold;

	static {
		// 設定ファイルの読み込み
		_configFilePath = System.getProperty("hinemos.selfcheck.file", _configFilePathDefault);

		properties = new Properties();
		FileInputStream fin = null;
		try {
			fin = new FileInputStream(_configFilePath);
			properties.load(fin);
		} catch (Exception e) {
			log.warn("config file loading failure. (" + _configFilePath + ")", e);
		} finally {
			if (fin != null) {
				try {
					fin.close();
				} catch (Exception e) {
					log.warn("file closing failure.", e);
				}
			}
		}

		startupDelaySec = Integer.parseInt(properties.getProperty("selfcheck.starup.delay", "90"));
		intervalSec = Integer.parseInt(properties.getProperty("selfcheck.interval", "150"));
		alertThreshold = Integer.parseInt(properties.getProperty("selfcheck.alert.threshold", "3"));
		threadPoolSize = Integer.parseInt(properties.getProperty("selfcheck.threadpool.size", "4"));

		snmpCommunity = properties.getProperty("selfcheck.snmp.community", "public");
		snmpPort = Integer.parseInt(properties.getProperty("selfcheck.snmp.port", "161"));
		snmpVersion = properties.getProperty("selfcheck.snmp.version", "2c");
		snmpTimeoutMSec = Integer.parseInt(properties.getProperty("selfcheck.snmp.timeout", "3000"));
		snmpRetries = Integer.parseInt(properties.getProperty("selfcheck.snmp.retries", "3"));

		jvmHeapMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.jvm.freeheap", "true"));
		jvmHeapThresholdMByte = Integer.parseInt(properties.getProperty("selfcheck.monitoring.jvm.freeheap.threshold", "32"));

		fsUsageMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.filesystem.usage", "false"));
		String fsUsageRaw = properties.getProperty("selfcheck.monitoring.filesystem.usage.list", "");
		List<FileSystemUsageConfig> fsUsages = new ArrayList<FileSystemUsageConfig>();
		for (String fs : fsUsageRaw.split(",")) {
			String[] pair = fs.split(":");
			if (pair.length == 2) {
				fsUsages.add(new FileSystemUsageConfig(pair[0], Integer.parseInt(pair[1])));
			}
		}
		fsUsageList = Collections.unmodifiableList(fsUsages);

		swapOutMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.swapout", "false"));

		dbMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.db", "true"));
		dbValidationQuery = properties.getProperty("selfcheck.monitoring.db.validationquery", "SELECT 1 FOR UPDATE");

		schedulerMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.scheduler.delay", "true"));
		schedulerDelayThresholdSec = Integer.parseInt(properties.getProperty("selfcheck.monitoring.scheduler.delay.threshold", "300"));

		wsQueueMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.ws.queue", "true"));
		wsQueueThreshold = Integer.parseInt(properties.getProperty("selfcheck.monitoring.ws.queue.threshold", "10000"));

		syslogQueueMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.systemlog.queue", "true"));
		syslogQueueThreshold = Integer.parseInt(properties.getProperty("selfcheck.monitoring.systemlog.queue.threshold", "10000"));

		snmptrapQueueMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.snmptrap.queue", "true"));
		snmptrapQueueThreshold = Integer.parseInt(properties.getProperty("selfcheck.monitoring.snmptrap.queue.threshold", "10000"));

		asyncTaskMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.asynctask.queue", "true"));
		String asyncTaskRaw = properties.getProperty("selfcheck.monitoring.asynctask.queue.list", "");
		List<AsyncTaskQueueConfig> asyncTasks = new ArrayList<AsyncTaskQueueConfig>();
		for (String task : asyncTaskRaw.split(",")) {
			String[] pair = task.split(":");
			if (pair.length == 2) {
				asyncTasks.add(new AsyncTaskQueueConfig(pair[0], Integer.parseInt(pair[1])));
			}
		}
		asyncTaskList = Collections.unmodifiableList(asyncTasks);

		threadActivityMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.thread.activity", "true"));
		threadRunningTimeThreshold = Integer.parseInt(properties.getProperty("selfcheck.monitoring.thread.activity.threshold", "300"));

		tableSizeMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.table.size", "true"));
		String tableSizeRaw = properties.getProperty("selfcheck.monitoring.table.size.list", "");
		List<TableSizeConfig> tableSizes = new ArrayList<TableSizeConfig>();
		for (String tableSize : tableSizeRaw.split(",")) {
			String[] pair = tableSize.split(":");
			if (pair.length == 3) {
				ThresholdType type;
				try {
					type = ThresholdType.valueOf(pair[2]);
				} catch (IllegalArgumentException e) {
					log.warn("table size monitoring - invalid type, set [MBYTE|COUNT]. : " + tableSize);
					continue;
				}
				tableSizes.add(new TableSizeConfig(pair[0], type, Long.parseLong(pair[1])));
			}
		}
		tableList = Collections.unmodifiableList(tableSizes);

		jobRunningSessionMonitoring = Boolean.parseBoolean(properties.getProperty("selfcheck.monitoring.job.runningsession", "true"));
		jobRunningSessionThreshold = Long.parseLong(properties.getProperty("selfcheck.monitoring.job.runningsession.threshold", "1000"));
	}

	/**
	 * この設定に基づく定期実行の最終実行日時を設定するメソッド
	 * @param date 最終実行日時
	 */
	public static void setLastMonitorDate(Date date) {
		lastMonitorDate = date;
	}

}
