/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.composite;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.composite.RoleIdListComposite;
import com.clustercontrol.composite.RoleIdListComposite.Mode;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.dialog.NotifyBasicCreateDialog;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.notify.NotifyInfo;

/**
 * 通知基本情報コンポジットクラス<BR>
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>「通知ID」 テキストボックス</dd>
 *  <dd>「説明」 テキストボックス</dd>
 * </dl>
 * 
 * @version 4.0.0
 * @since 3.0.0
 */

public class NotifyBasicComposite extends Composite {

	/** カラム数（タイトル）。*/
	public static final int WIDTH_TITLE = 3;

	/** カラム数（値）。*/
	public static final int WIDTH_VALUE = 8;

	/** 空白のカラム。*/
	public static final int WIDTH_BLANK = 4;

	/** カラム数（全て）。*/
	public static final int WIDTH_ALL = 15;

	/** 通知ID テキストボックス。 */
	private Text m_textNotifyId = null;

	/** 説明 テキストボックス。 */
	private Text m_textDescription = null;

	/** オーナーロールID用テキスト */
	protected RoleIdListComposite m_ownerRoleId = null;

	/** 変更対象通知ID。 */
	private String notifyId = null;

	/** ダイアログの編集モード */
	private int notifyIdType = 0;
	
	/** 親ダイアログ  */
	protected NotifyBasicCreateDialog parentDialog = null;


	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public NotifyBasicComposite(Composite parent, int style, String notifyId) {
		super(parent, style);

		this.notifyId = notifyId;
		this.notifyIdType = NotifyTypeConstant.NOTIFY_ID_TYPE_DEFAULT;
		this.initialize();
	}


	public NotifyBasicComposite(Composite parent, int style, String notifyId, int notifyIdType, NotifyBasicCreateDialog parentDialog) {
		super(parent, style);

		this.notifyId = notifyId;
		this.notifyIdType = notifyIdType;
		this.parentDialog = parentDialog;
		
		this.initialize();
	}


	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;
		this.setLayout(layout);

		/*
		 * 通知ID
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notify.id") + " : ");

		// テキスト
		this.m_textNotifyId = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textNotifyId.setLayoutData(gridData);
		this.m_textNotifyId.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * 説明
		 */
		// ラベル
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("description") + " : ");
		// テキスト
		this.m_textDescription = new Text(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textDescription.setLayoutData(gridData);
		// 空白
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_BLANK;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * オーナーロールID
		 */
		Label labelRoleId = new Label(this, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		labelRoleId.setLayoutData(gridData);
		labelRoleId.setText(Messages.getString("owner.role.id") + " : ");
		if (this.parentDialog.getUpdateFlg()) {
			this.m_ownerRoleId = new RoleIdListComposite(this, SWT.NONE, false, Mode.OWNER_ROLE);
		} else {
			this.m_ownerRoleId = new RoleIdListComposite(this, SWT.NONE, true, Mode.OWNER_ROLE);
		}
		gridData = new GridData();
		gridData.horizontalSpan = 5;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_ownerRoleId.setLayoutData(gridData);
		
		if (!this.parentDialog.getUpdateFlg()) {
			m_ownerRoleId.getComboRoleId().addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					parentDialog.setOwnerRoleId(m_ownerRoleId.getText());
				}
			});
		}
	}

	/**
	 * 更新処理
	 * 
	 */
	@Override
	public void update(){
		// 通知IDが必須項目であることを明示
		if(this.m_textNotifyId.getEnabled() && "".equals(this.m_textNotifyId.getText())){
			this.m_textNotifyId.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_textNotifyId.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知情報
	 * @param udpateFlg 更新するか否か（true：更新する）
	 */
	public void setInputData(NotifyInfo info, boolean updateFlg) {

		if(info != null){
			// 通知
			if (info.getNotifyId() != null) {
				this.m_textNotifyId.setText(info.getNotifyId());
				if (updateFlg) {
					this.m_textNotifyId.setEnabled(false);
				}
			}
			// 説明
			if (info.getDescription() != null) {
				this.m_textDescription.setText(info.getDescription());
			}
			// オーナーロールID
			if (info.getOwnerRoleId() != null) {
				this.m_ownerRoleId.setText(info.getOwnerRoleId());
				this.parentDialog.setOwnerRoleId(info.getOwnerRoleId());
			} else {
				this.parentDialog.setOwnerRoleId(m_ownerRoleId.getText());
			}
			
		}

		this.update();
	}

	/**
	 * 引数で指定された通知情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 入力値を設定する通知情報
	 * @return 検証結果
	 * 
	 */
	public ValidateResult createInputData(NotifyInfo info, String notifyId) {

		if(info != null){
			// 通知
			if (this.m_textNotifyId.getText() != null
					&& !"".equals((this.m_textNotifyId.getText()).trim())) {
				info.setNotifyId(this.m_textNotifyId.getText());
			}
			// 説明
			if (this.m_textDescription.getText() != null
					&& !"".equals((this.m_textDescription.getText()).trim())) {
				info.setDescription(this.m_textDescription.getText());
			}
			// オーナーロールID
			if (this.m_ownerRoleId.getText() != null
					&& !"".equals((this.m_ownerRoleId.getText()).trim())) {
				info.setOwnerRoleId(this.m_ownerRoleId.getText());
			}
		}
		return null;
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {

		if(this.notifyId != null && !"".equals(this.notifyId.trim())){
			this.m_textNotifyId.setEnabled(false);
		}
		else {
			this.m_textNotifyId.setEnabled(enabled);
		}
		this.m_textDescription.setEnabled(enabled);
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {

		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);

		return validateResult;
	}
	
	public RoleIdListComposite getRoleIdList(){
		return this.m_ownerRoleId;
	}
}