/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.plugin.impl;

import java.nio.charset.Charset;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.plugin.api.HinemosPlugin;
import com.clustercontrol.snmptrap.bean.SnmpTrapV1;
import com.clustercontrol.snmptrap.service.SnmpTrapMonitor;
import com.clustercontrol.snmptrap.util.SnmpTrapReceiver;

/**
 * SNMPTRAP監視の初期化・終了処理(udp:162の待ち受け開始)を制御するプラグイン.
 * 
 * @author takahatat
 */
public class SnmpTrapPlugin implements HinemosPlugin {

	public static final Log log = LogFactory.getLog(SnmpTrapPlugin.class);

	/** snmptrapの待ち受けポートを設けるかどうかを制御するパラメータ(クラスタ構成用) */
	public static final String _keyNoListen = "monitor.snmptrap.nolisten";
	public static final boolean _noListenDefault = false;
	public static final boolean _noListen;

	/** snmptrapの待ち受けアドレス */
	public static final String _keyListenAddress = "monitor.snmptrap.listen.address";
	public static final String _listenAddressDefault = "0.0.0.0";
	public static final String _listenAddress;

	/** snmptrapの待ち受けポート番号 */
	public static final String _keyListenPort = "monitor.snmptrap.listen.port";
	public static final int _listenPortDefault = 162;
	public static final int _listenPort;

	/** snmptrapのデフォルト文字コード */
	public static final String _keyCharset = "monitor.snmptrap.charset";
	public static final Charset _charsetDefault = Charset.forName("UTF-8");
	public static final Charset _charset;

	/** 受信処理とフィルタリング処理の間に存在するsnmptrap処理待ちキューの最大サイズ*/
	public static final String _keyTaskQueueSize = "monitor.snmptrap.filter.queue.size";
	public static final int _taskQueueSizeDefault = 15 * 60 * 30;	// 15[min] * 30[msg/sec]
	public static final int _taskQueueSize;

	/** フィルタリング処理のスレッド数 */
	public static final String _keyTaskThreadSize = "monitor.snmptrap.filter.thread.size";
	public static final int _taskThreadSizeDefault = 1;
	public static final int _taskThreadSize;

	/** 監視結果を通知するクラス */
	public static final String _keyNotifierClass = "monitor.snmptrap.notifier.class";
	public static final String _notifierClassDefault = "com.clustercontrol.snmptrap.util.SnmpTrapNotifierSingle";
	public static final String _notifierClass;

	/** SNMPTRAP監視の受信クラス */
	private static SnmpTrapReceiver _receiver;

	/** SNMPTRAP監視のフィルタリングクラス */
	private static SnmpTrapMonitor _handler;

	static {
		_listenAddress = HinemosProperties.getProperty(_keyListenAddress, _listenAddressDefault);

		boolean noListen = _noListenDefault;
		String noListenStr = HinemosProperties.getProperty(_keyNoListen);
		try {
			noListen = Boolean.parseBoolean(noListenStr);
		} catch (Exception e) { }
		_noListen = noListen;

		int listenPort = _listenPortDefault;
		String listenPortStr = HinemosProperties.getProperty(_keyListenPort);
		try {
			listenPort = Integer.parseInt(listenPortStr);
		} catch (Exception e) { }
		_listenPort = listenPort;

		String charsetAll = "";
		for (String c : Charset.availableCharsets().keySet()) {
			charsetAll += c + ", ";
		}
		log.info("supported charset : " + charsetAll);

		Charset charset = _charsetDefault;
		String charsetStr = HinemosProperties.getProperty(_keyCharset);
		try {
			charset = Charset.forName(charsetStr);
		} catch (Exception e) { }
		_charset = charset;

		String taskQueueSizeStr = HinemosProperties.getProperty(_keyTaskQueueSize);
		int taskQueueSize = _taskQueueSizeDefault;
		try {
			if (taskQueueSizeStr != null) {
				taskQueueSize = Integer.parseInt(taskQueueSizeStr);
			}
		} catch (Exception e) { }
		_taskQueueSize = taskQueueSize;

		String taskThreadSizeStr = HinemosProperties.getProperty(_keyTaskThreadSize);
		int taskThreadSize = _taskThreadSizeDefault;
		try {
			if (taskThreadSizeStr != null) {
				taskThreadSize = Integer.parseInt(taskThreadSizeStr);
			}
		} catch (Exception e) { }
		_taskThreadSize = taskThreadSize;

		_notifierClass = HinemosProperties.getProperty(_keyNotifierClass, _notifierClassDefault);
	}

	@Override
	public Set<String> getDependency() {
		Set<String> dependency = new HashSet<String>();
		dependency.add(AsyncWorkerPlugin.class.getName());
		dependency.add(SharedTablePlugin.class.getName());
		return dependency;
	}

	@Override
	public void create() {
		log.info(String.format("starting SnmpTrapPlugin : listenAddress = %s, listenPort = %d, charset = %s, queueSize = %d, threads = %d, _notifierClass = %s",
				_listenAddress, _listenPort, _charset.name(), _taskQueueSize, _taskThreadSize, _notifierClass));

		_handler = new SnmpTrapMonitor(_charset, _taskThreadSize, _taskQueueSize);
		_receiver = new SnmpTrapReceiver(_listenAddress, _listenPort, _handler);
	}

	@Override
	public void activate() {
		try {
			if (_noListen) {
				log.info("activating SnmpTrapPlugin without receiver.");
				_handler.start();
			} else {
				_receiver.start();
			}
		} catch (Exception e) {
			log.warn("SnmpTrapPlugin activation failure.", e);
		}
	}

	@Override
	public void deactivate() {
		if (_noListen) {
			if (_handler != null) {
				_handler.shutdown();
			}
		} else {
			if (_receiver != null) {
				_receiver.shutdown();
			}
		}

	}

	@Override
	public void destroy() {

	}

	public static void snmptrapReceived(String receiverId, SnmpTrapV1 snmptrap) {
		_handler.snmptrapReceived(receiverId, snmptrap);
	}

	public static long getReceivedCount() {
		return _handler.getReceivedCount();
	}

	public static long getNotifiedCount() {
		return _handler.getNotifiedCount();
	}

	public static long getDiscardedCount() {
		return _handler.getDiscardedCount();
	}

	public static int getQueuedCount() {
		return _handler.getQueuedCount();
	}



}
