/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.session;

import java.util.ArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarDuplicate;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.ObjectPrivilege_InvalidRole;
import com.clustercontrol.calendar.bean.CalendarDetailInfo;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.bean.CalendarPatternInfo;
import com.clustercontrol.calendar.factory.ModifyCalendar;
import com.clustercontrol.calendar.factory.SelectCalendar;
import com.clustercontrol.calendar.util.CalendarCache;
import com.clustercontrol.calendar.util.CalendarManagerUtil;
import com.clustercontrol.calendar.util.CalendarPatternCache;
import com.clustercontrol.calendar.util.CalendarValidator;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.commons.util.JpaTransactionManager;

/**
 *
 * <!-- begin-user-doc --> カレンダー情報の制御を行うsession bean <!-- end-user-doc --> *
 *
 */
public class CalendarControllerBean {

	private static Log m_log = LogFactory.getLog( CalendarControllerBean.class );

	/**
	 * カレンダ一覧を取得します。<BR>
	 * 
	 * @return カレンダ情報のリスト
	 * @throws HinemosUnknown
	 */
	public ArrayList<CalendarInfo> getAllCalendarList() throws HinemosUnknown, InvalidRole {
		JpaTransactionManager jtm = null;
		ArrayList<CalendarInfo> list = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			list = select.getAllCalendarList(null);
			jtm.commit();
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e){
			m_log.warn("getAllCalendarList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return list;
	}

	/**
	 * オーナーロールIDを指定してカレンダ一覧を取得します。<BR>
	 * 
	 * @param ownerRoleId オーナーロールID
	 * @return カレンダ情報のリスト
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public ArrayList<CalendarInfo> getCalendarList(String ownerRoleId) throws InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		ArrayList<CalendarInfo> list = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			list = select.getAllCalendarList(ownerRoleId);
			jtm.commit();
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e){
			m_log.warn("getCalendarList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return list;
	}

	/**
	 * カレンダIDのリストを取得します。<BR>
	 * 
	 * 引数のArrayListにはカレンダーidが含まれています。
	 * <p>
	 *	list.add(cal.getCalendar_id());
	 * </p>
	 * 
	 * @return カレンダIDのリスト
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public ArrayList<String> getCalendarIdList() throws InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		ArrayList<String> list = new ArrayList<String>();
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			list = select.getCalendarIdList();
			jtm.commit();
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("getCalendarIdList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return list;
	}

	/**
	 * 引数で指定したカレンダーIDに対応するカレンダ情報を取得します。<BR>
	 * 
	 * @param id
	 * @return カレンダ情報
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 */
	public CalendarInfo getCalendar(String id) throws CalendarNotFound, HinemosUnknown, InvalidRole {
		JpaTransactionManager jtm = null;
		CalendarInfo info = null;

		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			info = select.getCalendar(id);
			jtm.commit();
		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("getCalendar() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return info;
	}

	/**
	 * 引数で指定したカレンダーIDに対応するカレンダ情報を取得します。<BR>
	 * CalendarInfo.CalendarDetailInfo.etcInfoも取得します。
	 * 
	 * @param id
	 * @return カレンダ情報
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public CalendarInfo getCalendarFull(String id) throws CalendarNotFound, InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		CalendarInfo info = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			info = select.getCalendarFull(id);
			jtm.commit();
		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("getCalendar() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return info;
	}

	/**
	 * カレンダ[カレンダパターン]情報一覧を取得します。<BR>
	 * 
	 * @param ownerRoleId オーナーロールID
	 * @return カレンダ[カレンダパターン]情報のリスト
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws CalendarNotFound
	 */
	public ArrayList<CalendarPatternInfo> getCalendarPatternList(String ownerRoleId) throws HinemosUnknown, InvalidRole, CalendarNotFound {
		JpaTransactionManager jtm = null;
		ArrayList<CalendarPatternInfo> list = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			list = select.getCalendarPatternList(ownerRoleId);
			jtm.commit();
		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e){
			m_log.warn("getCalendarPatternList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return list;
	}

	/**
	 * 引数で指定したIDに対応するカレンダ[カレンダパターン]情報を取得します。<BR>
	 * 
	 * @param id
	 * @return カレンダ[カレンダパターン]情報
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public CalendarPatternInfo getCalendarPattern(String id) throws CalendarNotFound, InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		CalendarPatternInfo info = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			info = select.getCalendarPattern(id);
			jtm.commit();
		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("getCalendarPattern() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return info;
	}

	/**
	 * カレンダ（基本）情報を登録します。<BR>
	 * 
	 * 引数のDTOの内容をマネージャに登録します。
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param info　登録するカレンダー（基本）情報
	 * @return
	 * @throws CalendarDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 * @throws InvalidRole
	 */
	public void addCalendar(CalendarInfo info)
			throws CalendarDuplicate, HinemosUnknown, InvalidSetting, InvalidRole {
		JpaTransactionManager jtm = null;
		boolean commit = false;
		try{
			jtm = new JpaTransactionManager();

			// トランザクションがすでに開始されている場合は処理終了
			jtm.begin(true);

			//入力チェック
			CalendarValidator.validateCalendarInfo(info);

			String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);

			ModifyCalendar modify = new ModifyCalendar();
			modify.addCalendar(info, loginUser);

			jtm.commit();
			commit = true;
		} catch (CalendarDuplicate e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch(Exception e){
			m_log.warn("addCalendar() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (commit) {
				try {
					// read-committedのため、commit後に外部コンポーネントに通知する
					CalendarManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("addCalendar() transaction failure. : "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(), e);
				} finally {
					jtm.close();
				}
			}
			// EntityManagerクリア
			jtm.close();
		}
	}

	/**
	 * カレンダ（基本）情報を変更します。<BR>
	 * 
	 * 引数のプロパティの内容で更新します。
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param info
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 * @throws InvalidRole
	 */
	public void modifyCalendar(CalendarInfo info) throws CalendarNotFound, HinemosUnknown, InvalidSetting, InvalidRole {
		JpaTransactionManager jtm = null;
		boolean commit = false;

		try {
			jtm = new JpaTransactionManager();

			// トランザクションがすでに開始されている場合は処理終了
			jtm.begin(true);

			//入力チェック
			CalendarValidator.validateCalendarInfo(info);

			String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
			ModifyCalendar modify = new ModifyCalendar();
			modify.modifyCalendar(info, loginUser);
			jtm.commit();
			commit = true;
			// コミット後にキャッシュクリア
			CalendarCache.remove(info.getId());
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyCalendar() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
			if (commit) {
				try {
					// read-committedのため、commit後に外部コンポーネントに通知する
					CalendarManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("modifyCalendar() : transaction failure. "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ(基本）情報を 削除します。<BR>
	 * 
	 * 引数のIDに対応するカレンダー（基本）情報を削除します。
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param id カレンダーID
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 */
	public void deleteCalendar(String id) throws CalendarNotFound, HinemosUnknown, InvalidRole {
		JpaTransactionManager jtm = null;
		boolean commit = false;
		try {
			jtm = new JpaTransactionManager();

			// トランザクションがすでに開始されている場合は処理終了
			jtm.begin(true);

			CalendarValidator.valideDeleteCalendar(id);

			ModifyCalendar modify = new ModifyCalendar();
			modify.deleteCalendar(id);

			jtm.commit();
			commit = true;
			// コミット後にキャッシュクリア
			CalendarCache.remove(id);

		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteCalendar() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();

			if (commit) {
				try {
					// read-committedのため、commit後に外部コンポーネントに通知する
					CalendarManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("deleteCalendar() : transaction failure. "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ[カレンダパターン]情報を登録します。<BR>
	 * 
	 * 引数のDTOの内容をマネージャに登録します。
	 * Database commit後に実施すべきユーザ実装が存在するため、
	 * トランザクション開始はユーザが制御する。
	 * 
	 * @param info カレンダ[カレンダパターン]情報
	 * @return
	 * @throws CalendarDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 * @throws InvalidRole
	 */
	public void addCalendarPattern(CalendarPatternInfo info)
			throws CalendarDuplicate, HinemosUnknown, InvalidSetting, InvalidRole {
		JpaTransactionManager jtm = null;
		boolean commit = false;
		try{
			jtm = new JpaTransactionManager();

			// トランザクションがすでに開始されている場合は処理終了
			jtm.begin(true);

			//入力チェック
			CalendarValidator.validateCalendarPatternInfo(info);

			String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);

			ModifyCalendar modify = new ModifyCalendar();
			modify.addCalendarPattern(info, loginUser);

			jtm.commit();
			commit = true;
		} catch (CalendarDuplicate e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch(Exception e){
			m_log.warn("addCalendarPattern() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (commit) {
				try {
					// read-committedのため、commit後に外部コンポーネントに通知する
					CalendarManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("addCalendarPattern() transaction failure. : "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(), e);
				} finally {
					jtm.close();
				}
			}
			// EntityManagerクリア
			jtm.close();
		}
	}

	/**
	 * カレンダ[カレンダパターン]情報を変更します。<BR>
	 * 
	 * 引数のプロパティの内容で更新します。
	 * Database commit後に実施すべきユーザ実装が存在するため、
	 * トランザクション開始はユーザが制御する。
	 * 
	 * @param info
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 * @throws InvalidRole
	 */
	public void modifyCalendarPattern(CalendarPatternInfo info)
			throws CalendarNotFound, HinemosUnknown, InvalidSetting, InvalidRole {
		JpaTransactionManager jtm = null;
		boolean commit = false;

		try {
			jtm = new JpaTransactionManager();

			// トランザクションがすでに開始されている場合は処理終了
			jtm.begin(true);

			//入力チェック
			CalendarValidator.validateCalendarPatternInfo(info);

			String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
			ModifyCalendar modify = new ModifyCalendar();
			modify.modifyCalendarPattern(info, loginUser);
			jtm.commit();
			commit = true;

			// コミット後にキャッシュクリア
			CalendarPatternCache.remove(info.getId());

		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("modifyCalendarPattern() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
			if (commit) {
				try {
					// read-committedのため、commit後に外部コンポーネントに通知する
					CalendarManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("modifyCalendarPattern() : transaction failure. "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * カレンダ[カレンダパターン]情報を 削除します。<BR>
	 * 
	 * 引数のIDに対応するカレンダ[カレンダパターン]情報を削除します。
	 * Database commit後に実施すべきユーザ実装が存在するため、
	 * トランザクション開始はユーザが制御する。
	 * 
	 * @param id ID
	 * @return
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public void deleteCalendarPattern(String id) throws CalendarNotFound, InvalidRole, HinemosUnknown  {
		JpaTransactionManager jtm = null;
		boolean commit = false;

		try {
			jtm = new JpaTransactionManager();

			// トランザクションがすでに開始されている場合は処理終了
			jtm.begin(true);

			CalendarValidator.valideDeleteCalendarPattern(id);

			ModifyCalendar modify = new ModifyCalendar();
			modify.deleteCalendarPattern(id);
			jtm.commit();
			commit = true;

			// コミット後にキャッシュクリア
			CalendarPatternCache.remove(id);

		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("deleteCalendarPattern() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();

			if (commit) {
				try {
					// read-committedのため、commit後に外部コンポーネントに通知する
					CalendarManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("deleteCalendarPattern() : transaction failure. "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * 月間カレンダビューに表示する情報
	 * 引数で指定したカレンダーIDに対応するカレンダ詳細情報を取得します。<BR>
	 * 
	 * @param id
	 * @param orderNo
	 * @return カレンダ詳細情報
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 */
	public ArrayList<Integer> getCalendarMonth(String id, Integer year, Integer month) throws CalendarNotFound, HinemosUnknown, InvalidRole {
		JpaTransactionManager jtm = null;
		ArrayList<Integer> ret = null;

		//カレンダ一覧を指定してないときに操作した場合、空のリストを返す
		if (id == null || year <= 0 || month <= 0) {
			return new ArrayList<Integer>();
		}
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			ret = select.getCalendarMonth(id, year, month);
			jtm.commit();
		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("getCalendarMonthInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return ret;
	}
	/**
	 * 週間カレンダビューにて表示する情報
	 * 引数で指定した、カレンダID、年、月、日に対応するカレンダ詳細情報を取得します<BR>
	 * @param id
	 * @param year
	 * @param month
	 * @param day
	 * @return
	 * @throws CalendarNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 */
	public ArrayList<CalendarDetailInfo> getCalendarWeek(String id, Integer year, Integer month, Integer day) throws CalendarNotFound, HinemosUnknown, InvalidRole {
		JpaTransactionManager jtm = null;
		ArrayList<CalendarDetailInfo> ret = null;

		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			ret = select.getCalendarWeek(id, year, month, day);
			jtm.commit();
		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("getCalendarDisp() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return ret;
	}

	/**
	 * カレンダが実行可能かチェックします。<BR>
	 * 
	 * 引数のカレンダーidが引数のタイムスタンプにおいて実行可能であるかチェックします。
	 * 
	 * @param id
	 * @param checkTimestamp
	 * @return カレンダが実行可能か
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public Boolean isRun(String id, Long checkTimestamp) throws CalendarNotFound, InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		Boolean isRun = false;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			SelectCalendar select = new SelectCalendar();
			isRun = select.isRun(id, checkTimestamp);
			jtm.commit();
		} catch (CalendarNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("isRun() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return isRun;
	}


}

