/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.fault.HinemosException;

public class CommandCreator {

	private static Log log = LogFactory.getLog(CommandCreator.class);

	public static enum PlatformType { AUTO, WINDOWS, UNIX, REGACY };

	public static final String sysUser;

	public static final String osName;
	public static final PlatformType sysPlatform;

	// Windows用コマンドの分割パターン。
	// 先頭に空白文字1個以上があり、その後ろに
	// 1. ダブルクォートでくくられた（空白を含む）文字列塊
	// 2. ダブルクォートと空白文字を除く任意文字列塊
	// の、1・2が複数個（1個以上）連なった形で、1つの引数を構成する。
	private static final Pattern winCmdSplitPattern = Pattern.compile("^\\s+(((\".*?\")|([^\"\\s]+?))+)");

	static {
		sysUser = System.getProperty("user.name");
		log.info("java process user detected : " + sysUser);

		osName = System.getProperty("os.name");
		if (osName != null && (osName.startsWith("Windows") || osName.startsWith("windows"))) {
			sysPlatform = PlatformType.WINDOWS;
		} else {
			sysPlatform = PlatformType.UNIX;
		}
		log.info("os detected : " + osName);
		log.info("platform detected : " + sysPlatform);
	}

	/**
	 * convert platform string to type
	 * @param str platform string
	 * @return platform type
	 */
	public static PlatformType convertPlatform(String str) {
		// Local Variable
		PlatformType platform = null;

		// MAIN
		if ("windows".equals(str)) {
			platform = PlatformType.WINDOWS;
		} else if ("unix".equals(str)) {
			platform = PlatformType.UNIX;
		} else if ("compatible".equals(str) || "regacy".equals(str)) {
			platform = PlatformType.REGACY;
		} else {
			platform = PlatformType.AUTO;
		}
		if (log.isDebugEnabled())
			log.debug("string converted. (str = " + str + ", type = " + platform + ")");
		return platform;
	}

	public static String[] createCommand(String execUser, String execCommand, PlatformType platform) throws HinemosException {
		return createCommand(execUser, execCommand, platform, YesNoConstant.TYPE_YES);
	}

	/**
	 * create command for platform
	 * @param execUser execution user
	 * @param execCommand command string
	 * @param platform target platform
	 * @return command
	 * @throws HinemosException
	 */
	public static String[] createCommand(String execUser, String execCommand, PlatformType platform, Integer specifyUser) throws HinemosException {
		// Local Variables
		String[] command = null;

		// Main
		if (specifyUser == YesNoConstant.TYPE_YES && execUser == null) {
			throw new NullPointerException("execUser is not defined. (execUser = " + execUser + ")");
		}
		if (execCommand == null) {
			throw new NullPointerException("execCommand is not defined. (execCommand = " + execCommand + ")");
		}
		if (platform == null) {
			throw new NullPointerException("platform is not defined. (platform = " + platform + ")");
		}

		String user;
		if (specifyUser == YesNoConstant.TYPE_YES) {
			// ジョブを実行するユーザを指定する場合はexecUserをそのまま使用する
			user = execUser;
		} else {
			// エージェントの実行ユーザで実行するの場合はsysuserに置き換える
			user = sysUser;
		}

		switch (platform) {
		case WINDOWS :
			command = createWindowsCommand(user, execCommand);
			break;
		case UNIX :
			command = createUnixCommand(user, execCommand);
			break;
		case REGACY :
			command = createRegacyCommand(user, execCommand);
			break;
		case AUTO :
		default :
			command = createCommand(user, execCommand, sysPlatform, specifyUser);
			break;
		}

		return command;
	}

	/**
	 * create command for Windows like CMD /C [COMMAND]
	 * @param execUser execution user
	 * @param execCommand command string parsed by CMD
	 * @return command and arguments
	 * @throws HinemosException
	 */
	private static String[] createWindowsCommand(String execUser, String execCommand) throws HinemosException {
		// Local Variables
		String[] command = null;
		ArrayList<String> commandList = new ArrayList<String>();
		boolean isDebugEnable = log.isDebugEnabled();
		
		// MAIN
		if (execUser.equals(sysUser)) {
			
			// 引数を分割するための正規表現は、空白文字＋文字列を１つの塊として判断するため、
			// 与えられたコマンド文字列の先頭に空白を付与して分解を開始する。
			String splitTarget = " " + execCommand;
			
			while (true) {
				if (isDebugEnable) {
					log.debug("parse command for windows. split target = " + splitTarget);
				}
				Matcher match = winCmdSplitPattern.matcher(splitTarget);
				if (match.find()){
					// マッチした場合、空白文字を除いた部分を有効な文字列（コマンド or 引数）として取り出す。
					// 具体的には、正規表現 "^\\s+(((\".*?\")|([^\"\\s]+?))+)" の中の、
					// 頭の ^\\s+ を取り除いた、 (((\".*?\")|([^\"\\s]+?))+) の部分を有効な引数として取り出している。
					String arg = match.group(1);
					if (isDebugEnable) {
						log.debug("parse command for windows. find arg = " + arg);
					}
					commandList.add(arg);
					splitTarget = splitTarget.substring(match.end());
				} else {
					if (isDebugEnable) {
						log.debug("parse command for windows. can't find args");
					}
					break;
				}
			}
			
			command = commandList.toArray(new String[commandList.size()]);
			if (isDebugEnable) {
				log.debug("created command for windows. (cmd = " + Arrays.toString(command) + ")");
			}
		} else {
			throw new HinemosException("execution user and jvm user must be same on Windows. (execUser = " + execUser + ", sysUser = " + sysUser + ")");
		}
		return command;
	}

	/**
	 * create command for UNIX line (sudo -u [USER]) sh -c [COMMAND]
	 * @param execUser execution user
	 * @param execCommand command string parsed by sh
	 * @return command and arguments
	 * @throws HinemosException
	 */
	private static String[] createUnixCommand(String execUser, String execCommand) throws HinemosException {
		// Local Variables
		String[] command = null;

		// MAIN
		if (execUser.equals(sysUser)) {
			command = new String[]{ "sh", "-c", execCommand };
			if (log.isDebugEnabled()) log.debug("created command for unix. (cmd = " + Arrays.toString(command));
		} else {
			command = new String[]{ "sudo", "-u", execUser, "sh", "-c", execCommand };
			if (log.isDebugEnabled()) log.debug("created command for unix. (cmd = " + Arrays.toString(command));
		}
		return command;
	}

	/**
	 * create regacy (before 3.1) command like (su [USER] -c) execCommand
	 * @param execUser execution user
	 * @param execCommand command string
	 * @return command and arguments
	 * @throws HinemosException
	 */
	private static String[] createRegacyCommand(String execUser, String execCommand) throws HinemosException {
		// Local Variables
		String[] command = null;

		// MAIN
		if (execUser.equals(sysUser)) {
			// split by half space
			command = execCommand.split(" ");
		} else {
			if ("root".equals(sysUser)) {
				command = new String[]{ "su", execUser, "-c", execCommand };
				if (log.isDebugEnabled()) log.debug("created command for regacy. (cmd = " + Arrays.toString(command));
			} else {
				throw new HinemosException("execution user and jvm user must be same. (execUser = " + execUser + ", sysUser = " + sysUser + ")");
			}
		}
		return command;
	}
	/**
	 * プロダクト実行ユーザを取得します。
	 * @return プロダクト実行ユーザ
	 */
	public static String getSysUser() {
		return sysUser;
	}


	public static void main(String[] args) {

		String command = "hogehoge \"a b\" c , d";
		String[] commandArr = null;


		System.out.println("SpecifyUser = YES");

		// Windows環境用
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.WINDOWS, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.WINDOWS, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// UNIX/Linux環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("root", command, PlatformType.UNIX, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.UNIX, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// 下位互換性環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("Administrator", command, PlatformType.REGACY, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		// 環境自動識別用
		System.setProperty("user.name", "Administrator");
		System.setProperty("os.name", "Windows Server 2008");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.AUTO, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		System.setProperty("user.name", "root");
		System.setProperty("os.name", "Linux");
		try {
			commandArr = createCommand("root", command, PlatformType.AUTO, YesNoConstant.TYPE_YES);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		System.out.println("SpecifyUser = NO");

		// Windows環境用
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.WINDOWS, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.WINDOWS, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// UNIX/Linux環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("root", command, PlatformType.UNIX, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("hinemos", command, PlatformType.UNIX, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		// 下位互換性環境用
		System.setProperty("user.name", "root");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		System.setProperty("user.name", "Administrator");
		try {
			commandArr = createCommand("hinemos", command, PlatformType.REGACY, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		try {
			commandArr = createCommand("Administrator", command, PlatformType.REGACY, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
		// 環境自動識別用
		System.setProperty("user.name", "Administrator");
		System.setProperty("os.name", "Windows Server 2008");
		try {
			commandArr = createCommand("Administrator", command, PlatformType.AUTO, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}

		System.setProperty("user.name", "root");
		System.setProperty("os.name", "Linux");
		try {
			commandArr = createCommand("root", command, PlatformType.AUTO, YesNoConstant.TYPE_NO);
			if (commandArr != null) for (String arg : commandArr) { System.out.println("arg : " + arg); }
			System.out.println();
		} catch (Exception e) {
			System.out.println("ERROR MSG * " + e.getMessage() + "\n");
		}
	}
}
