/*
Copyright (C) 2015 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.xcloud.aws.ui.dialogs;

import static com.clustercontrol.xcloud.aws.common.AWSConstants.EPROP_AvailabilityZone;
import static com.clustercontrol.xcloud.aws.common.AWSConstants.EPROP_Subnet;
import static com.clustercontrol.xcloud.aws.common.AWSConstants.EPROP_Vpc;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.wb.swt.SWTResourceManager;

import com.clustercontrol.ws.xcloud.ExtendedProperty;
import com.clustercontrol.ws.xcloud.Instance;
import com.clustercontrol.ws.xcloud.InstanceStatus;
import com.clustercontrol.ws.xcloud.Subnet;
import com.clustercontrol.ws.xcloud.Vpc;
import com.clustercontrol.xcloud.aws.common.AWSStringConstants;
import com.clustercontrol.xcloud.common.CloudStringConstants;
import com.clustercontrol.xcloud.model.cloud.INetwork;
import com.clustercontrol.xcloud.ui.dialogs.IDialogConstants;
import com.clustercontrol.xcloud.util.CollectionComparator;

public class AttachElbDialog extends Dialog implements CloudStringConstants, AWSStringConstants {
	protected INetwork network;
	protected Table table;
	protected InstanceSelector selector;
	protected List<Instance> selected;
	protected AWSResourceProvider provider;

	/**
	 * Create the dialog.
	 * @param parentShell
	 */
	public AttachElbDialog(Shell parentShell, INetwork network) {
		super(parentShell);
		this.network = network;
		this.provider = new AWSResourceProvider(network.getLocation());
	}

	public List<Instance> getSelectedInstances(){
		return selected;
	}
	
	/**
	 * Create contents of the dialog.
	 * @param parent
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite area = (Composite) super.createDialogArea(parent);
		Composite container = new Composite(area, SWT.NONE);
		container.setBackground(SWTResourceManager.getColor(SWT.COLOR_WIDGET_BACKGROUND));
		GridLayout gl_container = new GridLayout(2, false);
		container.setLayout(gl_container);
		GridData gd_container = new GridData(GridData.FILL_BOTH);
		gd_container.widthHint = 614;
		container.setLayoutData(gd_container);
		
		Label lblNewLabel = new Label(container, SWT.RIGHT);
		lblNewLabel.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		lblNewLabel.setText(strManager + strSeparator);
		
		Label lblNewLabel_4 = new Label(container, SWT.BORDER);
		lblNewLabel_4.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		lblNewLabel_4.setText(network.getLocation().getCloudScope().getCloudScopes().getHinemosManager().getManagerName());
		
		Label lblNewLabel_1 = new Label(container, SWT.RIGHT);
		lblNewLabel_1.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		lblNewLabel_1.setText(strCloudScope + strSeparator);
		
		Label lblNewLabel_5 = new Label(container, SWT.BORDER);
		lblNewLabel_5.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		lblNewLabel_5.setText(String.format("%s (%s)", network.getLocation().getCloudScope().getName(), network.getLocation().getCloudScope().getId()));
		
		Label lblVolumeType = new Label(container, SWT.NONE);
		lblVolumeType.setAlignment(SWT.RIGHT);
		lblVolumeType.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		lblVolumeType.setText(strElb + strSeparator);
		
		Label lblNewLabel_2 = new Label(container, SWT.BORDER);
		lblNewLabel_2.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		lblNewLabel_2.setText(String.format("%s (%s)", network.getName(), network.getId()));
		
		Label lblNewLabel_3 = new Label(container, SWT.NONE);
		lblNewLabel_3.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		lblNewLabel_3.setAlignment(SWT.RIGHT);
		lblNewLabel_3.setText(strVPC + strSeparator);
		
		Label lblNewLabel_6 = new Label(container, SWT.BORDER);
		lblNewLabel_6.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		
		String vpcId = network.getExtendedProperty(EPROP_Vpc);
		lblNewLabel_6.setText(vpcId == null ? "": vpcId);
		
		Label lblInstance = new Label(container, SWT.NONE);
		lblInstance.setAlignment(SWT.RIGHT);
		lblInstance.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
		lblInstance.setText(strCompute + strSeparator);
		
		CheckboxTableViewer checkboxTableViewer = CheckboxTableViewer.newCheckList(container, SWT.BORDER | SWT.FULL_SELECTION);
		table = checkboxTableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
		checkboxTableViewer.addCheckStateListener(new ICheckStateListener() {
			@Override
			public void checkStateChanged(CheckStateChangedEvent event) {
				checkFinished();
			}
		});
		
		int count = 0;
		String countString = network.getExtendedProperty(EPROP_Subnet + "_count");
		if (countString != null)
			count = Integer.valueOf(countString);
		
		List<String> subnetIds = new ArrayList<>();
		for (int i = 0; i < count; ++i) {
			String subnetId = network.getExtendedProperty(String.format("%s[%d]", EPROP_Subnet, i));
			subnetIds.add(subnetId);
		}
		
		AWSResourceProvider provider = new AWSResourceProvider(network.getLocation());
		List<Vpc> vpcs = provider.getVpcs();
		
		final List<Subnet> subnets = new ArrayList<>();
		for (Vpc vpc: vpcs) {
			if (vpc.getId().equals(vpcId)) {
				CollectionComparator.compareCollection(subnetIds, vpc.getSubnets(), new CollectionComparator.Comparator<String, Subnet>() {
					public boolean match(String o1, Subnet o2) {return o1.equals(o2.getSubnetId());}
					public void matched(String o1, Subnet o2) {subnets.add(o2);}
				});
				break;
			}
		}
		
		List<Instance> availableInstance = new ArrayList<>();
		List<Instance> instances = provider.getInstances();
		for (Instance instance: instances) {
			if (instance.getInstanceStatus() == InstanceStatus.TERMINATED)
				continue;
			
			String availabilityZone = null;
			String vpc = null;
			for (ExtendedProperty prop: instance.getExtendedProperties()) {
				if (prop.getName().equals(EPROP_AvailabilityZone)) {
					availabilityZone = prop.getValue();
				} else if (prop.getName().equals(EPROP_Vpc)) {
					vpc = prop.getValue();
				}
			}
			
			if (availabilityZone == null)
				continue;
			
			if (vpc != null && !vpc.equals(vpcId))
				continue;
			
			for (Subnet subnet: subnets) {
				if (subnet.getAvailabilityZone().equals(availabilityZone)) {
					availableInstance.add(instance);
					break;
				}
			}
		}
		
		if (selected != null) {
			List<String> instanceIds = new ArrayList<>();
			for (Instance instance: selected) {
				instanceIds.add(instance.getId());
			}
			selector = new InstanceSelector(checkboxTableViewer, null, availableInstance, instanceIds);
		} else {
			selector = new InstanceSelector(checkboxTableViewer, null, availableInstance, network.getAttachedInstances());
		}
		
		getShell().getDisplay().asyncExec(new Runnable() {
			@Override
			public void run() {
				checkFinished();
			}
		});
		
		return area;
	}
	
	/**
	 * Create contents of the button bar.
	 * @param parent
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL,
				false);
		createButton(parent, IDialogConstants.CANCEL_ID,
				IDialogConstants.CANCEL_LABEL, false);
	}

	/**
	 * Return the initial size of the dialog.
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(657, 294);
	}

	@Override
	protected void okPressed() {
		selected = new ArrayList<>(selector.getSelectedInstances());		
		super.okPressed();
	}

	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText(String.format("%s - %s (%s)", dlgNetwork, strElbSetting, strAws));
	}
	
	protected void checkFinished() {
		Button okButton = getButton(IDialogConstants.OK_ID);
		if (okButton == null)
			return;
			
		if (selector != null) {
			CollectionComparator.Result<String, Instance> result = CollectionComparator.compareCollection(network.getAttachedInstances(), selector.getSelectedInstances(), new CollectionComparator.Comparator<String, Instance>() {
				public boolean match(String o1, Instance o2) {return o1.equals(o2.getId());}
			});
			getButton(IDialogConstants.OK_ID).setEnabled(!result.o1s.isEmpty() || !result.o2s.isEmpty());
		} else {
			getButton(IDialogConstants.OK_ID).setEnabled(false);
		}
	}
}