/*

 Copyright (C) 2013 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */
package com.clustercontrol.agent.util;

import java.util.ArrayList;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.ws.WebServiceException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.Agent;
import com.clustercontrol.agent.AgentEndPointWrapper;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.ws.agent.HinemosUnknown_Exception;
import com.clustercontrol.ws.agent.InvalidRole_Exception;
import com.clustercontrol.ws.agent.InvalidUserPass_Exception;
import com.clustercontrol.ws.agent.MessageInfo;
import com.clustercontrol.ws.monitor.MonitorInfo;
import com.clustercontrol.ws.monitor.MonitorStringValueInfo;

public class MonitorStringUtil {

	// ロガー
	private static Log m_log = LogFactory.getLog(MonitorStringUtil.class);

	//オリジナルメッセージのサイズ上限（Byte）
	public static int m_limit_logmsg = 1024;

	private static int notifyRetryCount = -1;
	private static int notifyRetrySleep = -1;

	static{
		// 1行のメッセージ上限を定める
		String limit_logmsg = AgentProperties.getProperty("limit.size.logmsg");
		if (limit_logmsg != null) {
			try {
				m_limit_logmsg = Integer.parseInt(limit_logmsg);
				m_log.info("limit.size.logmsg = " + m_limit_logmsg);
			} catch (NumberFormatException e) {
				m_log.error("limit.logmsg",e);
			}
		}
	}

	/**
	 * 監視文字列を整形する
	 * @param line
	 * @return formatted line
	 */
	public static String formatLine(String line){
		// CR-LFの場合は\rが残ってしまうので、ここで削除する。
		line = line.replace("\r", "");

		// 長さが上限値を超える場合は切り捨てる
		if (line.length() > m_limit_logmsg) {
			m_log.info("log line is too long");
			line = line.substring(0, m_limit_logmsg);
		}
		return line;
	}

	/**
	 * 監視文字列をパターンマッチし、マネージャに送信する
	 * @param line 監視文字列
	 * @param m_monitorInfoList 監視設定のリスト
	 */
	public static void patternMatch(String line, ArrayList<MonitorInfo> m_monitorInfoList, String filename) {
		for (MonitorInfo monitorInfo : m_monitorInfoList) {
			patternMatch(line, monitorInfo, null, filename);
		}
	}

	/**
	 * 監視文字列をパターンマッチし、マネージャに送信する
	 * @param line 監視文字列
	 * @param generationDate 生成日時
	 * @param monitorInfo 監視設定
	 */
	public static void patternMatch(String line, MonitorInfo monitorInfo, Date generationDate, String filename) {
		if (monitorInfo.getCalendar() != null &&
				! CalendarWSUtil.isRun(monitorInfo.getCalendar())) {
			m_log.debug("patternMatch is skipped because of calendar");
			return;
		}

		if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
			m_log.debug("patternMatch is skipped because of monitor flg");
			return;
		}

		for(MonitorStringValueInfo stringInfo : monitorInfo.getStringValueInfo()) {
			if(m_log.isDebugEnabled()){
				m_log.debug("patternMatch() line = " + line
						+ ", monitorId = " + stringInfo.getMonitorId()
						+ ", orderNo = " + stringInfo.getOrderNo()
						+ ", pattern = " + stringInfo.getPattern());
			}
			if (!stringInfo.isValidFlg()) {
				continue;
			}
			String patternText = stringInfo.getPattern();
			String message = line;
			m_log.trace("patternMatch check " + message);

			Pattern pattern = null;
			// 大文字・小文字を区別しない場合
			if(stringInfo.isCaseSensitivityFlg()){
				pattern = Pattern.compile(patternText, Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
			}
			// 大文字・小文字を区別する場合
			else{
				pattern = Pattern.compile(patternText, Pattern.DOTALL);
			}
			Matcher matcher = pattern.matcher(line);

			if (matcher.matches()) {
				m_log.debug("patternMatch match " + message);

				// 「処理する」
				if (stringInfo.getProcessType() == ProcessConstant.TYPE_YES) {
					MessageInfo logmsg = new MessageInfo();
					logmsg.setMessage(line);

					if(generationDate != null){
						m_log.debug("patternMatch set generation date : " + generationDate);
						logmsg.setGenerationDate(generationDate.getTime());
					}else{
						logmsg.setGenerationDate(new Date().getTime());
					}
					logmsg.setHostName(Agent.getAgentInfo().getHostname());
					try {
						if (filename != null) {
							monitorInfo.getLogfileCheckInfo().setLogfile(filename);
						}
						sendNotifyRetry(message, logmsg, monitorInfo, stringInfo);
						m_log.debug("patternMatch send message : " + message);
						m_log.debug("patternMatch send logmsg message : " + logmsg.getMessage());
						m_log.debug("patternMatch send logmsg generation date : " + new Date(logmsg.getGenerationDate()));
						m_log.debug("patternMatch send logmsg hostname : " + logmsg.getHostName());
					} catch (HinemosUnknown_Exception e) {
						m_log.warn("patternMatch (" + e.getClass().getSimpleName() +
								") : " + e.getMessage());
					} catch (InvalidRole_Exception e) {
						m_log.warn("patternMatch (" + e.getClass().getSimpleName() +
								") : " + e.getMessage());
					} catch (InvalidUserPass_Exception e) {
						m_log.warn("patternMatch (" + e.getClass().getSimpleName() +
								") : " + e.getMessage());
					}
				}
				break;
			}
		}
	}

	/**
	 * 通知をマネージャに送信する。
	 * 通知に失敗した場合、1分後にリトライする。
	 * @param message
	 * @param logmsg
	 * @param monitorInfo
	 * @param stringInfo
	 * @throws HinemosUnknown_Exception
	 * @throws InvalidRole_Exception
	 * @throws InvalidUserPass_Exception
	 */
	private static void sendNotifyRetry(String message, MessageInfo logmsg,
			MonitorInfo monitorInfo, MonitorStringValueInfo stringInfo)
					throws HinemosUnknown_Exception, InvalidRole_Exception, InvalidUserPass_Exception {
		boolean flag = false;

		if (notifyRetryCount < 0) {
			String str = AgentProperties.getProperty("notify.retry.count", "15");
			notifyRetryCount = Integer.parseInt(str);
			m_log.info("notify.retry.count=" + notifyRetryCount);
		}
		if (notifyRetrySleep < 0) {
			String str = AgentProperties.getProperty("notify.retry.sleep", "60000");
			notifyRetrySleep = Integer.parseInt(str);
			m_log.info("notify.retry.sleep=" + notifyRetrySleep);
		}

		for (int i = 0; i < notifyRetryCount; i++) {
			try {
				AgentEndPointWrapper.sendNotify(message, logmsg, monitorInfo, stringInfo);
				flag = true;
			} catch (WebServiceException e) {
				try {
					Thread.sleep(notifyRetrySleep);
				} catch (InterruptedException e1) {
					m_log.warn("sendNotifyRetry " + e1.getMessage());
				}
			}
			if (flag) {
				return;
			}
		}
		m_log.warn("give up notifying. Maybe, manager is down");
	}

}
