/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.JobInvalidException;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.JudgmentObjectConstant;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobManagementUserInfo;
import com.clustercontrol.jobmanagement.bean.JobNotificationsInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.util.Messages;

/**
 * ジョブユーティリティクラス
 * 
 * 以下ィを提供します。<BR>
 * <li>ジョブツリーアイテムに関するユーティリティ
 * <li>ログインユーザが参照可能なジョブユニットかどうかをチェックするユーティリティ
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class JobUtil {
	/** ログ出力のインスタンス<BR> */
	protected static Log m_log = LogFactory.getLog( JobUtil.class );
			
	/** コピーしたジョブIDの接頭語 */
	private static final String COPY_OF = "Copy Of?";
	
	/**
	 * 引数で指定されたジョブツリーアイテムのコピーを作成する
	 * 
	 * @param original コピー元ジョブツリーアイテム
	 * @return コピーとして作成されたジョブツリーアイテム
	 */
	public static JobTreeItem copy(JobTreeItem original) {
		JobTreeItem clone = null;
		if(original != null){
			clone = (JobTreeItem)original.clone();
		}
		
		return clone;
	}
	
	/**
	 * @param original コピー元ジョブツリーアイテム
	 * @param top コピー元ジョブツリーアイテム
	 * @return コピーとして作成されたジョブツリーアイテム
	 */
	public static JobTreeItem copy(JobTreeItem original, JobTreeItem top, String jobunitId) {
		JobTreeItem clone = copy(original);
		clone.getData().setJobunitId(jobunitId);
		
		//待ち条件を削除する
		deleteWaitRule(clone);
		
		//ジョブIDの変更
		changeJobId(clone, top, clone);
		
		//FIXME ジョブユニットIDを変更する
		
		return clone;
	}
	
	/**
	 * ジョブIDを一括変更する<BR>
	 * ジョブツリーアイテムのツリー階層の全てに対して一括変更する。<BR>
	 * topとcloneに指定されたジョブツリーアイテムを、ジョブIDの重複チェック対象とする。<BR>
	 * ジョブIDが重複した場合、コピーしたジョブIDの接頭語とカウンタを付加してジョブIDを決定する。
	 * 
	 * @param item ジョブID変更対象のジョブツリーアイテム
	 * @param top ジョブIDの重複チェック対象のジョブツリーアイテム
	 * @param clone ジョブIDの重複チェック対象のジョブツリーアイテム
	 */
	protected static void changeJobId(JobTreeItem item, JobTreeItem top, JobTreeItem clone) {
		if(item == null || top == null)
			return;
		
		//ジョブIDを変更
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			int count = 1;
			StringBuffer jobId = new StringBuffer();
			jobId.append(info.getId());
			while(true){              
				if(!findJobId(jobId.toString(), top) && !findJobId(jobId.toString(), clone)){
					break;
				}
				jobId.delete(0, jobId.length());
				if(count == 1){
					jobId.append(COPY_OF.replace("?", " "));
				}
				else{
					jobId.append(COPY_OF.replace("?", " (" + count + ") "));
				}
				jobId.append(info.getId());
				count++;
			}
			info.setId(jobId.toString());
			
			// ジョブユニットの場合のみジョブユニットIDを上書き
			if (info.getType() == JobConstant.TYPE_JOBUNIT) {
				info.setJobunitId(jobId.toString());
				//System.out.println("changeJobId() setJobunitId = " + jobId.toString());
			} 
			//else {
			//	info.setJobunitId(item.getParent().getData().getJobunitId());
			//	System.out.println("changeJobId() setJobunitId = " + item.getParent().getData().getJobunitId() + " from jobid = " + item.getParent().getData().getJobunitId());
			//}
			
			// 通知グループIDの変更
			ArrayList<JobNotificationsInfo> jobNotifyList = info.getNotifications();
			Iterator<JobNotificationsInfo> itrJob = jobNotifyList.iterator();
			
			Iterator<NotifyRelationInfo> itrNotify = null;
			Collection<NotifyRelationInfo> ctNotify = null;
			Collection<NotifyRelationInfo> modCtNotify = null;

			NotifyRelationInfo notify = null;
			NotifyRelationInfo modNotify = null;	//変更後格納用
			
			// 新しい通知グループIDを作成
			String newNotifyGroupId = NotifyRelationInfo.createNotifyGroupIdJob(info.getJobunitId(), info.getId(), 0);
			//System.out.println("changeJobId() newNotifyGroupId = " + newNotifyGroupId);
			// ジョブの通知の種類（開始・通知・警告・危険）分調べる
			while(itrJob.hasNext()){
				JobNotificationsInfo jobNotify = itrJob.next();
				jobNotify.setNotifyGroupId(newNotifyGroupId);
				
				// 通知関連情報を調査
				ctNotify = jobNotify.getNotifyId();
				
				if(ctNotify != null) {
					itrNotify = ctNotify.iterator();
					modCtNotify = new ArrayList<NotifyRelationInfo>();
					
					// 通知関連情報内の通知グループIDの変更
					while(itrNotify.hasNext()){
						notify = itrNotify.next();
						
						// ディープコピー（通知グループIDは新規）を作成
						modNotify = new NotifyRelationInfo(newNotifyGroupId, 
								notify.getNotifyId(),
								notify.getNotifyType(),
								notify.getNotifyFlg());
						
						modCtNotify.add(modNotify);
						
					}
					
					if(modCtNotify.size() != 0){
						jobNotify.setNotifyId(modCtNotify);
					}
				}
			}
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			childrens[i].getData().setJobunitId(info.getJobunitId());
			//System.out.println("changeJobId() set childrens[i] " + info.getJobunitId());
			changeJobId(childrens[i], top, clone);
		}
	}
	
	/**
	 * ジョブツリーアイテムからジョブ待ち条件情報を削除する<BR>
	 * ジョブツリーアイテムのツリー階層の全てが削除対象
	 * 
	 * @param item ジョブ待ち条件情報を削除するジョブツリーアイテム
	 */
	protected static void deleteWaitRule(JobTreeItem item) {
		if(item == null)
			return;
		
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			//待ち条件を削除する
			JobWaitRuleInfo waitRule = info.getWaitRule();
			if(waitRule != null && waitRule instanceof JobWaitRuleInfo){
				if(waitRule.getObject() != null && waitRule.getObject() instanceof ArrayList){
					waitRule.setObject(new ArrayList());
				}
			}
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			deleteWaitRule(childrens[i]);
		}
	}
	
	/**
	 * ジョブツリーアイテムからジョブIDが一致するインスタンスの有無を返す<BR>
	 * ジョブツリーアイテムのツリー階層の全てが検索対象
	 * 
	 * @param jobId ジョブID
	 * @param item ジョブツリーアイテム
	 * @return ジョブIDが一致するジョブツリーアイテムがあればtrue、なければfalse。
	 */
	public static boolean findJobId(String jobId, JobTreeItem item) {
		boolean find = false;
		
		//ジョブIDをチェック
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			if(jobId.compareTo(info.getId()) == 0){
				find = true;
				return find;
			}
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			find = findJobId(jobId, childrens[i]);
			if(find){
				break;
			}
		}
		
		return find;
	}
	
	/**
	 * ジョブツリーアイテム内のジョブIDが一致するインスタンスの有無を返す<BR>
	 * 
	 * @param item ジョブツリーアイテム
	 * @param isTop ツリーのトップか否か
	 * @return ジョブIDが一致するジョブ/ネット/ユニットが存在すればtrue、なければfalse。
	 * @throws JobInvalidException
	 */
	public static void findDuplicateJobId(JobTreeItem item, boolean isTop) throws JobInvalidException{
		m_log.debug("findDuplicateJobId() start : isTop = " + isTop);
		
		// 自身がtopの場合は何もしない
		if(!isTop){
			
			// 自身がジョブユニット/ジョブネット/ジョブの場合
			if(item.getData() instanceof JobInfo){
				String jobId = item.getData().getId();
				m_log.debug("findDuplicateJobId() jobId = " + jobId);
				
				JobTreeItem[] children = item.getChildrenArray();
				for (JobTreeItem child : children) {
					
					m_log.debug("findDuplicateJobId() child = " + child.getData().getId());

					// 配下のツリーにトップのジョブIDが含まれるか？
					if(findJobId(jobId, child)){
						
						// jobunitid内にjobidが重複している
						m_log.debug("findDuplicateJobId() jobId is in child " + child.getPath());
						Object[] args = {jobId, child.getData().getJobunitId()};
						throw new JobInvalidException(Messages.getString("message.job.65",args));
					} else {
						m_log.debug("findDuplicateJobId() jobId is not in child " + child.getPath());
					}
				}
			}
		}
		
		JobTreeItem[] children = item.getChildrenArray();
		for (JobTreeItem child : children) {
			m_log.debug("findDuplicateJobId() call child " + child.getData().getId());
			findDuplicateJobId(child, false);
		}
		
		m_log.debug("findDuplicateJobId() success!!");
	}
	
	/**
	 * ジョブツリーアイテム内のジョブユニットIDが一致するインスタンスの有無を返す<BR>
	 * 
	 * @param item ジョブツリーアイテム
	 * @return ジョブユニットIDが一致するユニットが存在すればtrue、なければfalse。
	 * @throws JobInvalidException
	 */
	public static void findDuplicateJobunitId(JobTreeItem item) throws JobInvalidException{
		m_log.debug("findDuplicateJobunitId() start " + item.getPath());
		
		JobTreeItem top = item.getChildren(0);
		JobTreeItem[] jobunits = top.getChildrenArray();
		
		HashSet<String> set = new HashSet<String>();
		for (JobTreeItem jobunit : jobunits) {
			// ジョブユニットID
			String jobunitId = jobunit.getData().getJobunitId();
			m_log.debug("findDuplicateJobunitId() jobunitId = " + jobunitId);

			if(set.contains(jobunitId)){
				m_log.debug("findDuplicateJobunitId() hit " + jobunitId);
				Object[] args = {jobunitId};
				throw new JobInvalidException(Messages.getString("message.job.64",args));
			} else {
				m_log.debug("findDuplicateJobunitId() add " + jobunitId + " to set");
				set.add(jobunitId);
			}
		}
		
		m_log.debug("findDuplicateJobunitId() success!!");
	}
	
	/**
	 * ジョブツリーアイテムの最上位のインスタンスを取得する
	 * 
	 * @param item ジョブツリーアイテム
	 * @return 最上位のジョブツリーアイテム
	 */
	public static JobTreeItem getTopJobTreeItem(JobTreeItem item) {
		if(item == null)
			return null;
		
		while (item.getParent() != null) {
			if(item.getParent().getData().getType() == JobConstant.TYPE_COMPOSITE){
				item = item.getParent();
				break;
			}
			else{
				item = item.getParent();
			}
		}
		
		return item;
	}
	
	/**
	 * ジョブツリーアイテムの付属するジョブユニットのインスタンスを取得する
	 * 
	 * @param item ジョブツリーアイテム
	 * @return 付属するジョブユニットのジョブツリーアイテム
	 */
	public static JobTreeItem getTopJobUnitTreeItem(JobTreeItem item) {
		if(item == null)
			return null;
		
		while (item.getParent() != null) {
			//System.out.println("getTopJobUnitTreeItem() " + item.getParent().getData().getJobunitId() + "." + item.getParent().getData().getId());
			if(item.getParent().getData().getType() == JobConstant.TYPE_JOBUNIT){
				item = item.getParent();
				break;
			}
			else if(item.getData().getType() == JobConstant.TYPE_JOBUNIT){
				break;
			}
			else{
				item = item.getParent();
			}
		}
		
		return item;
	}
	
	/**
	 * ジョブツリーアイテムのジョブ待ち条件情報をチェックする
	 * 
	 * @param item ジョブ待ち条件情報をチェックするジョブツリーアイテム
	 */
	public static boolean checkWaitRule(JobTreeItem item) throws JobInvalidException{
		boolean check = true;
		
		if(item == null)
			return check;
		
		if(item.getData() != null && item.getData() instanceof JobInfo){
			//ジョブID取得
			String jobId = item.getData().getId();
			
			//待ち条件情報を取得する
			JobWaitRuleInfo waitRule = item.getData().getWaitRule();
			if(waitRule != null && waitRule instanceof JobWaitRuleInfo && 
					waitRule.getObject() != null && waitRule.getObject().size() > 0){
				
				Iterator itr = waitRule.getObject().iterator();
				while(itr.hasNext()) {
					//判定対象を取得
					JobObjectInfo objectInfo = (JobObjectInfo)itr.next();
					if(objectInfo.getType() != JudgmentObjectConstant.TYPE_TIME){
						//判定対象のジョブIDが同一階層に存在するかチェック
						boolean find = false;
						String targetJobId = objectInfo.getJobId();
						JobTreeItem[] childrens = item.getParent().getChildrenArray();
						for(int i = 0; i < childrens.length; i++){
							//ジョブIDをチェック
							JobInfo childInfo = childrens[i].getData();
							if(childInfo != null && childInfo instanceof JobInfo && 
									!jobId.equals(childInfo.getId())){
								if(targetJobId.compareTo(childInfo.getId()) == 0){
									find = true;
									break;
								}
							}
						}
						if(!find){
							String args[] = {jobId, targetJobId};
							throw new JobInvalidException(Messages.getString("message.job.59", args));
						}
					}
				}
			}
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			check = checkWaitRule(childrens[i]);
			if(!check){
				break;
			}
		}
		
		return check;
	}
	
	/**
	 * ログインユーザで参照可能なジョブユニットかどうかチェックします。<BR>
	 * @param jobunit 選択されたジョブユニット
	 * @param loginUser ログインユーザのユーザID
	 * @return ログインユーザで参照可能なジョブユニットである場合true
	 */
	public static boolean isReferableJobunit (JobTreeItem jobunit, String loginUser) {
		
		boolean ret = false;
		
    	ArrayList<JobManagementUserInfo> userInfo = jobunit.getData().getManagementUser();
    	if (userInfo == null || userInfo.size() == 0) {
    		ret = true;
    	} else {
    		for (JobManagementUserInfo i : userInfo) {
    			if (loginUser.equals(i.getUserId())) {
    				ret = true;
    				break;
    			}
    		}
    	}
		
		return ret;
	}
}