/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  fsnd_dtq.c
 * @brief %jp{f[^L[ւ̋M}%en{Forced Send to Data Queue}
 *
 * Copyright (C) 1998-2006 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/dtqobj.h"



#if _KERNEL_SPT_FSND_DTQ


/** %jp{f[^L[ւ̋M}%en{Forced Send to Data Queue}
 * @param  dtqid    %jp{MΏۂ̃f[^L[IDԍ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{f[^L[֑Mf[^}%en{Data element tobe sent}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(dtqids邢͎gpłȂ)}%en{Invalid ID number(dtqid is invalid or unusable)}
 * @retval E_NOEXS  %jp{IuWFNg(Ώۃf[^L[o^)}%en{Non-existant object(specified data queue is not registerd)}
 */
ER fsnd_dtq(ID dtqid, VP_INT data)
{
	_KERNEL_T_DTQCB          *dtqcb;
	const _KERNEL_T_DTQCB_RO *dtqcb_ro;
	_KERNEL_T_TSKHDL         tskhdl;
	_KERNEL_T_TCB            *tcb;
	_KERNEL_DTQ_T_DTQCNT     dtqcnt;
	
	/* %jp{ID ̃`FbN} */
#if _KERNEL_SPT_FSND_DTQ_E_ID
	if ( !_KERNEL_DTQ_CHECK_DTQID(dtqid) )
	{
		return E_ID;	/* %jp{sIDԍ}%en{Invalid ID number} */
	}
#endif

	_KERNEL_ENTER_SVC();	/* %jp{T[rXR[ɓ}%en{enter service-call} */
	
	/* %jp{IuWFNg݃`FbN} */
#if _KERNEL_SPT_FSND_DTQ_E_NOEXS
	if ( !_KERNEL_DTQ_CHECK_EXS(dtqid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[I} */
		return E_NOEXS;			/* %jp{IuWFNg} */
	}
#endif
	
	/* %jp{f[^L[Rg[ubN擾} */
	dtqcb    = _KERNEL_DTQ_ID2DTQCB(dtqid);
	dtqcb_ro = _KERNEL_DTQ_GET_DTQCB_RO(dtqid, dtqcb);	/* %jp{RO擾} */

	/* %jp{f[^L[̗̈eʎ擾} */
	dtqcnt = _KERNEL_DTQ_GET_DTQCNT(dtqcb_ro);

#if _KERNEL_SPT_FSND_DTQ_E_ILUSE
	if ( dtqcnt == 0 )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[I} */
		return E_ILUSE;			/* %jp{T[rXR[sgp} */
	}
#endif
	
	/* %jp{M҂s擪^XNo} */
	tskhdl = _KERNEL_DTQ_RMH_RQUE(dtqcb);
	if ( tskhdl != _KERNEL_TSKHDL_NULL )
	{
		VP_INT *p_data;

		/* %jp{҂^XNΑ҂} */
		tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB擾} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);			/* %jp{G[R[hݒ} */
		p_data = (VP_INT *)_KERNEL_TSK_GET_DATA(tcb);
		*p_data = data;
		_KERNEL_DSP_WUP_TSK(tskhdl);				/* %jp{^XN̑҂} */
		_KERNEL_DTQ_RMV_RTOQ(tskhdl);
		
		/* %jp{^XNfBXpb`̎s} */
		_KERNEL_DSP_TSK();
	}
	else
	{
		_KERNEL_DTQ_T_DTQCNT sdtqcnt;
		_KERNEL_DTQ_T_DTQCNT head;
		VP_INT               *dtq;
		
		/* %jp{f[^L[擾} */
		sdtqcnt = _KERNEL_DTQ_GET_SDTQCNT(dtqcb);
		head    = _KERNEL_DTQ_GET_HEAD(dtqcb);
		dtq     = _KERNEL_DTQ_GET_DTQ(dtqcb_ro);

		if ( sdtqcnt < dtqcnt )		/* %jp{L[ɋ󂫂͂邩H} */
		{
			/* %jp{f[^L[ɒǉ} */
			if ( head < dtqcnt - sdtqcnt - 1 )
			{
				dtq[head + sdtqcnt] = data;
			}
			else
			{
				dtq[head + sdtqcnt - dtqcnt] = data;
			}
			sdtqcnt++;
			_KERNEL_DTQ_SET_SDTQCNT(dtqcb, sdtqcnt);
		}
		else
		{
			/* %jp{f[^L[ɋǉ} */
			dtq[head] = data;

			/* %jp{擪ʒu炷} */
			if ( head + 1 < dtqcnt )
			{
				head++;
			}
			else
			{
				head = 0;
			}
			_KERNEL_DTQ_SET_HEAD(dtqcb, head);
		}		
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
	
	return E_OK;	/* %jp{I}%en{Normal completion} */
}


#else	/* _KERNEL_SPT_FSND_DTQ */


#if _KERNEL_SPT_FSND_DTQ_E_NOSPT

/** %jp{f[^L[ւ̋M}%en{Forced Send to Data Queue}
 * @param  dtqid    %jp{MΏۂ̃f[^L[IDԍ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{f[^L[֑Mf[^}%en{Data element tobe sent}
 * @retval E_NOSPT  %jp{T|[g@\}%en{Unsupported function}
 */
ER fsnd_dtq(ID dtqid, VP_INT data)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_FSND_DTQ */



/* end of file */
