/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  tloc_mtx.c
 * @brief %jp{~[ebNX̃bNl(^CAEg)}%en{Lock Mutex(with Timeout)}
 *
 * Copyright (C) 1998-2009 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/mtxobj.h"



#if _KERNEL_SPT_TLOC_MTX


/** %jp{~[ebNX̃bNl(^CAEg)}%en{Lock Mutex(with Timeout)}
 * @param  mtxid    %jp{bNΏۂ̃~[ebNXIDԍ}%en{ID number of the mutex to be locked}
 * @param  tmout    %jp{^CAEgw}%en{Specified timeout}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(mtxids邢͎gpłȂ)}%en{Invalid ID number(mtxid is invalid or unusable)}
 * @retval E_CTX    %jp{ReLXgG[}%en{Context error}
 * @retval E_NOEXS  %jp{IuWFNg(Ώۃ~[ebNXo^)}%en{Non-existant object(specified mutex is not registerd)}
 * @retval E_ILUSE  %jp{T[rXR[sgp(~[ebNX̑dbNCDẍᔽ)}%en{Illegal service call use(multiple locking of a mutex, ceiling priority violation)}
 * @retval E_RLWAI  %jp{҂Ԃ̋(҂Ԃ̊Ԃrel_wait)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_DLT    %jp{҂IuWFNg̍폜(҂Ԃ̊ԂɑΏۃ~[ebNX폜)}%en{Waiting object deleted(mutex is deleted waiting)}
 */
ER tloc_mtx(ID mtxid, TMO tmout)
{
	return _kernel_loc_mtx(mtxid, TMO_FEVR);
}

#else	/* _KERNEL_SPT_TLOC_MTX */

#if _KERNEL_SPT_TWAI_SEM_E_NOSPT

/** %jp{~[ebNX̃bNl(^CAEg)}%en{Lock Mutex(with Timeout)}
 * @param  mtxid    %jp{bNΏۂ̃~[ebNXIDԍ}%en{ID number of the mutex to be locked}
 * @param  tmout    %jp{^CAEgw}%en{Specified timeout}
* @retval E_NOSPT  %jp{T|[g@\}%en{Unsupported function}
 */
ER tloc_mtx(ID mtxid, TMO tmout)
{
	return E_NOSPT;
}

#endif

#endif		/* _KERNEL_SPT_TLOC_MTX */


#if _KERNEL_SPT_KLOC_MTX

/** %jp{~[ebNX̃bNl(^CAEg)}%en{Lock Mutex(with Timeout)}
 * @param  mtxid    %jp{bNΏۂ̃~[ebNXIDԍ}%en{ID number of the mutex to be locked}
 * @param  tmout    %jp{^CAEgw}%en{Specified timeout}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(mtxids邢͎gpłȂ)}%en{Invalid ID number(mtxid is invalid or unusable)}
 * @retval E_CTX    %jp{ReLXgG[}%en{Context error}
 * @retval E_NOEXS  %jp{IuWFNg(Ώۃ~[ebNXo^)}%en{Non-existant object(specified mutex is not registerd)}
 * @retval E_ILUSE  %jp{T[rXR[sgp(~[ebNX̑dbNCDẍᔽ)}%en{Illegal service call use(multiple locking of a mutex, ceiling priority violation)}
 * @retval E_RLWAI  %jp{҂Ԃ̋(҂Ԃ̊Ԃrel_wait)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_TMOUT  %jp{^CAEg}%en{Timeout}
 * @retval E_DLT    %jp{҂IuWFNg̍폜(҂Ԃ̊ԂɑΏۃ~[ebNX폜)}%en{Waiting object deleted(mutex is deleted waiting)}
 */
ER _kernel_loc_mtx(ID mtxid, TMO tmout)
{
	_KERNEL_T_MTXCB_RO_PTR	mtxcb_ro;
	_KERNEL_T_MTXCB_PTR		mtxcb;
	_KERNEL_T_MTXHDL		mtxhdl;
	_KERNEL_T_TCB_PTR		tcb;
	_KERNEL_T_TSKHDL		tskhdl;
	_KERNEL_T_TSKHDL		tskhdl_lock;
	ER						ercd;
	
	/* %jp{ReLXg`FbN} */
#if _KERNEL_SPT_KLOC_MTX_E_CTX
	if ( tmout != TMO_POL && _KERNEL_SYS_SNS_DPN() )
	{
		return E_CTX;			/* %jp{ReLXgG[}%en{Context error} */
	}
#endif
	
	/* %jp{ID ̃`FbN} */
#if _KERNEL_SPT_KLOC_MTX_E_ID
	if ( !_KERNEL_MTX_CHECK_MTXID(mtxid) )
	{
		return E_ID;			/* %jp{sIDԍ}%en{Invalid ID number} */
	}
#endif
	
	_KERNEL_ENTER_SVC();		/* %jp{T[rXR[ɓ}%en{enter service-call} */
	
	/* %jp{IuWFNg݃`FbN} */
#if _KERNEL_SPT_KLOC_MTX_E_NOEXS
	if ( !_KERNEL_MTX_CHECK_EXS(mtxid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
		return E_NOEXS;			/* %jp{IuWFNg}%en{Non-existant object} */
	}
#endif

	/* %jp{~[ebNXRg[ubN擾} */
	mtxcb    = _KERNEL_MTX_ID2MTXCB(mtxid);
	mtxcb_ro = _KERNEL_MTX_GET_MTXCB_RO(mtxid, mtxcb);

	/* %jp{s̃^XNnh擾} */
	tskhdl = _KERNEL_SYS_GET_RUNTSK();
	tcb    = _KERNEL_TSK_TSKHDL2TCB(tskhdl);
	
	/* %jp{bÑ^XNnh擾} */
	tskhdl_lock = _KERNEL_MTX_GET_TSKHDL(mtxcb);
	
	if ( _KERNEL_MTX_GET_TSKHDL(mtxcb) == _KERNEL_TSKHDL_NULL )
	{
		/* %jp{L^XNȂ΃~[ebNXbN} */
		_KERNEL_MTX_SET_TSKHDL(mtxcb, tskhdl); 		/* %jp{~[ebNX̊l} */

		/* %jp{~[ebNXTCBɐڑ} */
		mtxhdl = _KERNEL_MTX_GET_MTXHDL(mtxid, mtxcb);
		_kernel_add_mtx(mtxhdl, tskhdl);
		
#if _KERNEL_SPT_MTX_TA_CEILING
		if ( _KERNEL_MTX_GET_MTXATR(mtxcb_ro) == TA_CEILING )
		{
			/* %jp{^XN̗DxV[Ol܂ňグ} */
			if ( _KERNEL_TSK_GET_TSKPRI(tcb) < _KERNEL_MTX_GET_CEILPRI(mtxcb_ro) )
			{
				_KERNEL_TSK_SET_TSKPRI(tcb, _KERNEL_MTX_GET_CEILPRI(mtxcb_ro));
			}
		}
#endif
		
		ercd = E_OK;
	}
	else
	{
#if _KERNEL_SPT_TWAI_SEM || _KERNEL_SPT_POL_SEM
		if ( tmout == TMO_POL )
		{
			ercd = E_TMOUT;  /* %jp{^CAEg}%en{Timeout} */
		}
		else
#endif
		{
			/* %jp{^XN҂Ԃɂ} */
			_KERNEL_TSK_SET_TSKWAIT(tcb, _KERNEL_TTW_MTX);
			_KERNEL_TSK_SET_WOBJID(tcb, mtxid);
			_KERNEL_DSP_WAI_TSK(tskhdl);
			_KERNEL_MTX_ADD_QUE(mtxcb, _KERNEL_MTX_GET_MTXCB_RO(mtxid, mtxcb), tskhdl);			/* %jp{҂sɒǉ} */

#if _KERNEL_SPT_TLOC_MTX
			if ( tmout != TMO_FEVR )
			{
				_KERNEL_MTX_ADD_TOQ(tskhdl, tmout);				/* %jp{^CAEgL[ɒǉ} */
			}
#endif
			
#if _KERNEL_SPT_MTX_TA_INHERIT
			/* %jp{Dxp} */
			if ( _KERNEL_MTX_GET_MTXATR(mtxcb_ro) == TA_INHERIT )
			{
				_KERNEL_T_TCB_PTR tcb_lock;

				/* %jp{bN^XNTCB擾} */
				tcb_lock = _KERNEL_TSK_TSKHDL2TCB(tskhdl_lock);

				/* %jp{Dxp} */
				if ( _KERNEL_TSK_GET_TSKPRI(tcb_lock) > _KERNEL_TSK_GET_TSKPRI(tcb) )
				{
					_KERNEL_TSK_SET_TSKPRI(tcb_lock, _KERNEL_TSK_GET_TSKPRI(tcb));
					if ( _KERNEL_TSK_GET_TSKSTAT(tcb) == TTS_RDY )
					{
						_KERNEL_SYS_RMV_RDQ(tskhdl);
						_KERNEL_SYS_ADD_RDQ(tskhdl);
					}
				}
			}

			
			/* %jp{^XNfBXpb`̎s} */
			_KERNEL_DSP_TSK();

			/* %jp{G[R[h̎擾} */
			ercd = _KERNEL_TSK_GET_ERCD(tcb);
		}
#endif
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{IuWFNg}%en{Non-existant object} */
	
	return ercd;
}


#endif	/* _KERNEL_SPT_KLOC_MTX */



/* end of file */
