/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  unl_mtx.c
 * @brief %jp{~[ebNX̃bN}%en{Unlock mutex}
 *
 * Copyright (C) 1998-2009 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/mtxobj.h"



#if _KERNEL_SPT_UNL_MTX


/** %jp{~[ebNX̃bN}%en{Unlock Mutex}
 * @param  mtxid    %jp{bNΏۂ̃~[ebNXIDԍ}%en{ID number of the mutex to be unlocked}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(mtxids邢͎gpłȂ)}%en{Invalid ID number(mtxid is invalid or unusable)}
 * @retval E_CTX    %jp{ReLXgG[}%en{Context error}
 * @retval E_NOEXS  %jp{IuWFNg(Ώۃ~[ebNXo^)}%en{Non-existant object(specified mutex is not registerd)}
 * @retval E_ILUSE  %jp{T[rXR[sgp(Ώۃ~[ebNXbNĂȂ)}%en{Illegal service call use(the invoking task does not have the specified mutex locked)}
 */
ER unl_mtx(ID mtxid)
{
	_KERNEL_T_MTXCB_PTR		mtxcb;
	_KERNEL_T_MTXHDL		mtxhdl;
	_KERNEL_T_TCB_PTR		tcb;
	_KERNEL_T_TSKHDL		tskhdl;
		
	/* %jp{ReLXg`FbN} */
#if _KERNEL_SPT_UNL_MTX_E_CTX
	if ( _KERNEL_SYS_SNS_CTX() )
	{
		return E_CTX;			/* %jp{ReLXgG[}%en{Context error} */
	}
#endif
	
	/* %jp{ID ̃`FbN} */
#if _KERNEL_SPT_UNL_MTX_E_ID
	if ( !_KERNEL_MTX_CHECK_MTXID(mtxid) )
	{
		return E_ID;			/* %jp{sIDԍ}%en{Invalid ID number} */
	}
#endif
	
	_KERNEL_ENTER_SVC();		/* %jp{T[rXR[ɓ}%en{enter service-call} */
	
	/* %jp{IuWFNg݃`FbN} */
#if _KERNEL_SPT_UNL_MTX_E_NOEXS
	if ( !_KERNEL_MTX_CHECK_EXS(mtxid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
		return E_NOEXS;			/* %jp{IuWFNg}%en{Non-existant object} */
	}
#endif
	
	/* %jp{~[ebNXRg[ubN擾} */
	mtxcb = _KERNEL_MTX_ID2MTXCB(mtxid);

	/* %jp{s̃^XNnh擾} */
	tskhdl = _KERNEL_SYS_GET_RUNTSK();
		
	/* %jp{LĂ邩`FbN} */
#if _KERNEL_SPT_UNL_MTX_E_ILUSE
	if ( tskhdl != _KERNEL_MTX_GET_TSKHDL(mtxcb) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
		return E_ILUSE;			/* %jp{T[rXR[sgp}%en{Illegal service call use} */
	}
#endif
	
	tcb    = _KERNEL_TSK_TSKHDL2TCB(tskhdl);
	mtxhdl = _KERNEL_MTX_GET_MTXHDL(mtxid, mtxcb);
	
	/* %jp{~[ebNO} */
	_kernel_rmv_mtx(mtxhdl, tskhdl);
	
	/* %jp{L~[ebNXȂ} */
	if ( _KERNEL_TSK_GET_MTXHDL(tcb) == _KERNEL_MTXHDL_NULL )
	{
		/* %jp{Dxɖ߂} */
		_KERNEL_TSK_SET_TSKPRI(tcb, _KERNEL_TSK_GET_TSKBPRI(tcb));		
		if ( _KERNEL_TSK_GET_TSKSTAT(tcb) == TTS_RDY )
		{
			_KERNEL_SYS_RMV_RDQ(tskhdl);
			_KERNEL_SYS_ADD_RDQ(tskhdl);
		}
	}
	
	/* %jp{^XNfBXpb`̎s} */
	_KERNEL_DSP_TSK();

	_KERNEL_LEAVE_SVC();	/* %jp{IuWFNg}%en{Non-existant object} */
	
	return E_OK;
}


#else	/* _KERNEL_SPT_UNL_MTX */


#if _KERNEL_SPT_UNL_MTX_E_NOSPT

/** %jp{~[ebNX̃bN}%en{Unlock Mutex}
 * @param  mtxid    %jp{bNΏۂ̃~[ebNXIDԍ}%en{ID number of the mutex to be unlocked}
 * @retval E_NOSPT  %jp{T|[g@\}%en{Unsupported function}
 */
ER unl_mtx(ID mtxid)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_LOC_MTX */



/* end of file */
