/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  sig_sem.c
 * @brief %jp{Z}tH̕ԋp}%en{Release Semaphore Resource}
 *
 * Copyright (C) 1998-2006 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/semobj.h"



#if _KERNEL_SPT_SIG_SEM


/** %jp{Z}tH̕ԋp}%en{Release Semaphore Resource}
 * @param  semid    %jp{Z}tHԋpΏۂ̃Z}tHIDԍ}%en{ID number of the semaphore to which resource is released}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(semids邢͎gpłȂ)}%en{Invalid ID number(semid is invalid or unusable)}
 * @retval E_NOEXS  %jp{IuWFNg(ΏۃZ}tHo^)}%en{Non-existant object(specified semaphore is not registerd)}
 * @retval E_QOVR   %jp{L[COI[o[t[(ő厑zԋp)}%en{Queue overflow(release will exceed maximum resource count)}
 */
ER sig_sem(ID semid)
{
	_KERNEL_T_SEMCB_PTR  semcb;
	_KERNEL_T_TSKHDL     tskhdl;
	_KERNEL_T_TCB        *tcb;
	_KERNEL_SEM_T_SEMCNT semcnt;
	
	/* %jp{ID ̃`FbN} */
#if _KERNEL_SPT_SIG_SEM_E_ID
	if ( !_KERNEL_SEM_CHECK_SEMID(semid) )
	{
		return E_ID;	/* %jp{sIDԍ}%en{Invalid ID number} */
	}
#endif
	
	_KERNEL_ENTER_SVC();	/* %jp{T[rXR[ɓ}%en{enter service-call} */
	
	/* %jp{IuWFNg݃`FbN} */
#if _KERNEL_SPT_SIG_SEM_E_NOEXS
	if ( !_KERNEL_SEM_CHECK_EXS(semid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[I} */
		return E_NOEXS;			/* %jp{IuWFNg} */
	}
#endif
	
	/* %jp{Z}tHRg[ubN擾} */
	semcb = _KERNEL_SEM_ID2SEMCB(semid);
	
	/* %jp{҂s擪^XNo} */
	tskhdl = _KERNEL_SEM_RMH_QUE(semcb);
	if ( tskhdl != _KERNEL_TSKHDL_NULL )
	{
		/* %jp{҂^XNΑ҂} */
		tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB擾} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);			/* %jp{G[R[hݒ} */
		_KERNEL_DSP_WUP_TSK(tskhdl);				/* %jp{^XN̑҂} */
		_KERNEL_SEM_RMV_TOQ(tskhdl);
		
		/* %jp{^XNfBXpb`̎s} */
		_KERNEL_DSP_TSK();
	}
	else
	{
		/* %jp{Z}tHJE^擾} */
		semcnt = _KERNEL_SEM_GET_SEMCNT(semcb);

		/* %jp{L[COI[o[t[`FbN} */
#if _KERNEL_SPT_SIG_SEM_E_QOVR
		{
			_KERNEL_T_SEMCB_RO_PTR semcb_ro;

			semcb_ro = _KERNEL_SEM_GET_SEMCB_RO(semid, semcb);
			if ( semcnt >= _KERNEL_SEM_GET_MAXSEM(semcb_ro) )
			{
				_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
				return E_QOVR;			/* %jp{L[COI[o[t[}%en{Queue overflow} */
			}
		}
#endif
		
		/* %jp{Z}tHԋp} */
		_KERNEL_SEM_SET_SEMCNT(semcb, semcnt + 1);
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
	
	return E_OK;	/* %jp{I}%en{Normal completion} */
}


#else	/* _KERNEL_SPT_SIG_SEM */


#if _KERNEL_SPT_SIG_SEM_E_NOSPT

/** %jp{Z}tH̕ԋp}%en{Release Semaphore Resource}
 * @param  semid    %jp{Z}tHԋpΏۂ̃Z}tHIDԍ}%en{ID number of the semaphore to which resource is released}
 * @retval E_NOSPT  %jp{T|[g@\}%en{Unsupported function}
 */
ER sig_sem(ID semid)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_SIG_SEM */



/* end of file */
