/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  iwup_tsk.c
 * @brief %jp{^XN̋N}%en{Wakeup Task}
 *
 * Copyright (C) 1998-2009 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"


#if _KERNEL_SPT_IWUP_TSK

#if _KERNEL_SPT_DPC


static void _kernel_dpc_iwup_tsk(ID tskid, VP_INT param);


/** %jp{^XN̋N}%en{Wakeup Task}
 * @param  tskid    %jp{NΏۂ̃^XNIDԍ}%en{ID number of the task to be woken up}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(tskids邢͎gpłȂ)}%en{Invalid ID number(tskid is invalid or unusable)}
 * @retval E_NOMEM  %jp{s}%en{Insufficient memory}
 */
ER wup_tsk(
		ID tskid)
{
#if _KERNEL_SPT_IWUP_TSK_E_ID
	if ( tskid == TSK_SELF )		/* %jp{^XNw莞̕ϊ} */
	{
		return E_ID;			/* %jp{sIDԍ}%en{Invalid ID number} */
	}
	
	if ( !_KERNEL_TSK_CHECK_TSKID(tskid) )
	{
		return E_ID;			/* %jp{sIDԍ}%en{Invalid ID number} */
	}

#endif

	/* %jp{xsv} */
	return _KERNEL_SYS_REQ_DPC(_kernel_dpc_iwup_tsk, tskid, 0);
}


void _kernel_dpc_iwup_tsk(ID tskid, VP_INT param)
{
	_KERNEL_T_TSKHDL tskhdl;
	_KERNEL_T_TCB    *tcb;
	
		
	/* %jp{IuWFNg݃`FbN} */
#if _KERNEL_SPT_IWUP_TSK_E_NOEXS
		if ( !_KERNEL_TSK_CHECK_EXS(tskid) )
		{
			return;					/* %jp{IuWFNg}%en{Non-existant object} */
		}
#endif
	
	/* %jp{IDԍw莞̕ϊ} */
	tskhdl = _KERNEL_TSK_ID2TSKHDL(tskid);
	
	/* %jp{TCB擾} */
	tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);
	
	if ( ((_KERNEL_TSK_GET_TSKSTAT(tcb) & _KERNEL_TTS_WAI)) && _KERNEL_TSK_GET_TSKWAIT(tcb) == _KERNEL_TTW_SLP )
	{
		/* %jp{҂} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);		/* %jp{G[R[hݒ} */
		_KERNEL_DSP_WUP_TSK(tskhdl);			/* %jp{^XN̑҂} */
	}
	else
	{
#if _KERNEL_TCB_WUPCNT
		{
			_KERNEL_TSK_T_WUPCNT wupcnt;
			
			wupcnt = _KERNEL_TSK_GET_WUPCNT(tcb);

#if _KERNEL_SPT_IWUP_TSK_E_QOVR
			if ( wupcnt >= _KERNEL_TMAX_WUPCNT )
			{
				_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[I} */
				return E_QOVR;
			}
#endif
			
			_KERNEL_TSK_SET_WUPCNT(tcb, wupcnt + 1);
		}
#endif
	}

	_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[I} */

	return E_OK;
}

#else

/** %jp{^XN̋N}%en{Wakeup Task}
 * @param  tskid    %jp{NΏۂ̃^XNIDԍ}%en{ID number of the task to be woken up}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(tskids邢͎gpłȂ)}%en{Invalid ID number(tskid is invalid or unusable)}
 * @retval E_NOEXS  %jp{IuWFNg(Ώۃ^XNo^)}%en{Non-existant object(specified task is not registered)}
 * @retval E_OBJ    %jp{IuWFNgԃG[(Ώۃ^XNx~)}%en{Object state error(specified task is in DORMANT state)}
 * @retval E_QOVR   %jp{L[COI[o[t[(NvL[CÕI[o[t[)}%en{Queue overflow(overflow of wakeup request count)}
 */
ER iwup_tsk(
		ID tskid)
{
	return wup_tsk(tskid);
}

#endif


#else	/* _KERNEL_SPT_IWUP_TSK */


/** %jp{^XN̋N}%en{Wakeup Task}
 * @param  tskid    %jp{NΏۂ̃^XNIDԍ}%en{ID number of the task to be woken up}
 * @retval E_NOSPT  %jp{T|[g@\}%en{Unsupported function}
 */
ER iwup_tsk(
		ID tskid)
{
	return E_NOSPT;
}


#endif	/* _KERNEL_SPT_IWUP_TSK */


/* end of file */
