#include "kernel.h"
#include "task.h"
#include "intmgr.h"
#include "screen.h"
#include "gdt.h"

TaskScheduler SystemScheduler;
TCB* TaskScheduler::CurrentTcb;

extern "C" void task_dispatch(TCB* tcb);

extern "C" void TaskEntryPoint(Task* task) {
    task->run();
    while(true) {
        __asm__ __volatile__("hlt");
    }
}

//-------------------------------------------------------------------
// Task Class Method
//-------------------------------------------------------------------
Task::Task(size_t stack_size) :
    StackSize(stack_size), Priority(NORMAL_PRIORITY), isStackInMemPool(true) {
    Stack = (uint8*)MemoryManager.alloc(StackSize);
    *((uint32*)&Stack[StackSize - 8]) = 0; // return address(dummy)
    *((uint32*)&Stack[StackSize - 4]) = (uint32)this;
    Tcb.Context = (TaskContext*)&Stack[StackSize - sizeof(TaskContext) - 8];
    Tcb.Context->eflags = 0x0202;
    Tcb.Context->cs  = Gdt::KERNEL_CS * sizeof(GdtEntry);
    Tcb.Context->eip = (uint32)TaskEntryPoint;
    Tcb.Context->ds  = Gdt::KERNEL_DS * sizeof(GdtEntry);
    Tcb.Context->es  = Tcb.Context->ds;
    Tcb.StackSegment = Tcb.Context->ds;
    Status = READY;
}

#if 0 /* Under immplementation */
Event& Task::waitForEvent(void) {
    if (EventQueue.isEmpty()) {
        if (Scheduler != 0) {
            Scheduler->makeWait(*this);
        } else {
            /* should throw any exception! */
        }
    }
    return (Event&)*EventQueue.dequeue();
}

void Task::sendEvent(Event& evt) {
    EventQueue.enqueue(evt);
    if (Scheduler != 0) {
        Scheduler->makeReady(*this);
    }
}
#endif

//-------------------------------------------------------------------
// TaskScheduler Class Method
//-------------------------------------------------------------------
void TaskScheduler::dispatch(bool dispatch_immd) {
    Task* next_task;
    if (((next_task = (Task*)ReadyQueue.dequeue()) != 0) &&
        (CurrentTask->getPriority() <= next_task->getPriority())) {
        addReadyQueue(*CurrentTask);
        CurrentTask = next_task;
        CurrentTask->setStatus(Task::RUNNING);
        if (dispatch_immd) {
            task_dispatch(&CurrentTask->getTcb());
        } else {
            CurrentTcb = &CurrentTask->getTcb();
        }
    }
}

void TaskScheduler::init(Task& cur_task) {
    CurrentTask = &cur_task;
    CurrentTcb  = &cur_task.getTcb();
}

TCB* TaskScheduler::getCurrentTcb(void) {
    return &CurrentTask->getTcb();
}

