#ifndef __IOPORT_H__
#define __IOPORT_H__

#include <types.h>

class IoPortLib
{
public:
    enum IoPort {
        // DMA Controller
        DMAC0_CH0_ADR = 0x00,
        DMAC0_CH0_CNT = 0x01,
        DMAC0_CH1_ADR = 0x02,
        DMAC0_CH1_CNT = 0x03,
        DMAC0_CH2_ADR = 0x04,
        DMAC0_CH2_CNT = 0x05,
        DMAC0_CH3_ADR = 0x06,
        DMAC0_CH3_CNT = 0x07,
        DMAC0_CMD     = 0x08,
        DMAC0_REQUEST = 0x09,
        DMAC0_SMASK   = 0x0a,
        DMAC0_MODE    = 0x0b,
        DMAC0_CBPFF   = 0x0c, /* clear byte pointer flip flop */
        DMAC0_CLEAR   = 0x0d,
        DMAC0_CLRMASK = 0x0e,
        DMAC0_ALLMASK = 0x0f,

        DMAC1_CMD     = 0xd0,
        DMAC1_SMASK   = 0xd4,
        DMAC1_MODE    = 0xd6,
        DMAC1_CLEAR   = 0xda,

        DMAPG_RSVD0   = 0x80,
        DMAPG_CH2_ADR = 0x81,
        DMAPG_CH3_ADR = 0x82,
        DMAPG_CH1_ADR = 0x83,
        DMAPG_CH0_ADR = 0x87,

        // Interrupt Controller
        PIC0_ICW1 = 0x20,
        PIC0_ICW2 = 0x21,
        PIC0_ICW3 = 0x21,
        PIC0_ICW4 = 0x21,
        PIC0_OCW1 = 0x21,
        PIC0_OCW2 = 0x20,
        PIC0_OCW3 = 0x20,
        PIC0_IMR  = 0x21,
        PIC1_ICW1 = 0xa0,
        PIC1_ICW2 = 0xa1,
        PIC1_ICW3 = 0xa1,
        PIC1_ICW4 = 0xa1,
        PIC1_OCW1 = 0xa1,
        PIC1_OCW2 = 0xa0,
        PIC1_OCW3 = 0xa0,
        PIC1_IMR  = 0xa1,

        // Timer
        PIT_CNT0  = 0x40,
        PIT_CNT1  = 0x41,
        PIT_CNT2  = 0x42,
        PIT_CTRL  = 0x43,

        // Keyboard
        KBD_DATA  = 0x60,
        KBD_STS   = 0x61,

        // VGA
        VGA_CTLADR  = 0x3d4,
        VGA_CTLDATA = 0x3d5,

        // Floppy Disk Controller
        FDC_P_DOR   = 0x3f2,
        FDC_P_MSR   = 0x3f4,
        FDC_P_DR    = 0x3f5,
        FDC_P_CCR   = 0x3f7,
        FDC_S_DOR   = 0x372,
        FDC_S_MSR   = 0x374,
        FDC_S_DR    = 0x375,
        FDC_S_CCR   = 0x377,
    };

    static __inline void out8(uint16 port, uint8 value) {
        __asm__ __volatile__ ("outb %%al, %%dx" :: "d" (port), "a" (value));
    }

    static void out16(IoPort port, uint16 value) {
        __asm__ __volatile__ ("outw %%ax, %%dx" :: "d" (port), "a" (value));
    }

    static void out32(IoPort port, uint32 value) {
        __asm__ __volatile__ ("outl %%eax, %%dx" :: "d" (port), "a" (value));
    }

    static uint8 in8(IoPort port) {
        uint8 value;
        __asm__ __volatile__ ("inb %%dx, %%al" : "=a" (value) : "d" (port));
        return value; 
    }

    static uint8 in16(IoPort port) {
        uint16 value;
        __asm__ __volatile__ ("inw %%dx, %%ax" : "=a" (value) : "d" (port));
        return value; 
    }

    static uint8 in32(IoPort port) {
        uint32 value;
        __asm__ __volatile__ ("inl %%dx, %%eax" : "=a" (value) : "d" (port));
        return value; 
    }
};

#endif /* __IOPORT_H__ */
