/*************************************************************************************************/
/*!
   	@file		EdgemapCell.h
	@author 	Fanzo
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	"iFace/iEdgemapCell.h"

#pragma pack( push , 8 )		//set align

namespace icubic
{

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define

/**************************************************************************************************
"EdgemapCell" class 
**************************************************************************************************/
class EdgemapCell_bs : public IEdgemapCell
{
	cb_copy_impossible( EdgemapCell_bs );
	
// variable member
protected:
	IMemAllocLump*				m_allocator;
	Array< CellEdge* >			m_line;
	irect						m_area;
	
// private functions
private:
//=================================================================================================
//!	create CellEdge
//!	@retval			---
//-------------------------------------------------------------------------------------------------
CellEdge* CreateCellEdge()
{
	return ( CellEdge* )m_allocator->Allocate( sizeof( CellEdge ) );
}
//=================================================================================================
//!	subpixelunit
//!	@retval			---
//-------------------------------------------------------------------------------------------------
fvector2 ToFx
		(
		const fvector2 &pos
		)const
{
	fvector2	r;
	r.x = floor( pos.x * cell_anti_scale() + 0.5f );
	r.y = floor( pos.y * cell_anti_scale() + 0.5f );
	return r;
}
//=================================================================================================
//!	subpixelunit
//!	@retval			---
//-------------------------------------------------------------------------------------------------
float ToFx
		(
		float v
		)const
{
	return floor( v * cell_anti_scale() + 0.5f );
}
//=================================================================================================
//!	add edge
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void InsertEdge
		(
		int		sx , 
		int		sy , 
		int		tx , 
		int		ty , 
		int		dir
		)
{
	CellEdge*	e = CreateCellEdge();//m_edgefactory.Create();
	e->m_x		= sx >> cell_anti_shift();
	e->m_dy		= ( sy < ty ? ( ty - sy ) : ( sy - ty )  ) * dir;
	e->m_area	= ( ( sx & cell_anti_mask() ) + tx - ( sx & ~cell_anti_mask() ) ) * e->m_dy;
	e->m_next	= 0;
	
	CellEdgePtr*	ce = &m_line[( min( sy , ty ) >> cell_anti_shift() ) - m_area.ymin];
	while( true )
	{
		if( *ce == 0 || e->m_x <= ( *ce )->m_x )
		{
			e->m_next = *ce;
			*ce	= e;
			break;
		}
		ce	= &( *ce )->m_next;
	}
}
//=================================================================================================
//!	set segment x
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void AddSegmentX
		(
		int		sx , 
		int		csy , 
		int		tx , 
		int		cty , 
		int		dir
		)
{
	int		sy	= csy;
	int		ty	= cty;
	if( sx > tx )
	{
		swap( &sx , &tx );
		swap( &sy , &ty );
	}
	if( sx == tx )
	{
		InsertEdge( sx , sy , tx , ty , dir );
		return;
	}
	
	//prev	
	int		ssx	= sx;
	int		ttx	= min( tx , ( ssx + cell_anti_scale() ) & ( ~cell_anti_mask() ) );
	float	ssy	= ( float )sy;
	float	tty	= ( ty - sy ) * ( ttx - sx ) / ( float )( tx - sx ) + ( float )sy;
	InsertEdge( ssx , (int)ssy , ttx , (int)tty , dir );
	ssx	= ttx;
	ssy	= tty;
	
	// center
	float	dy = ( ty - sy ) * cell_anti_scale() / ( float )( tx - sx );
	for( ttx = ssx + cell_anti_scale() , tty = ssy + dy ; ttx <= tx ; ttx += cell_anti_scale() , tty += dy )
	{
		InsertEdge( ssx , clip( (int)ssy , csy , cty ) , ttx , clip( (int)tty , csy , cty ) , dir );
		ssx = ttx;
		ssy	= tty;
	}
	
	// post
	if( ssx < tx )
	{
		InsertEdge( ssx , (int)ssy , tx , ty , dir );
	}
}
//=================================================================================================
//!	set segment
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void AddSegment
		(
		const fvector2	&start , 
		const fvector2	&target
		)
{
	fvector2	s	= ToFx( start );
	fvector2	t	= ToFx( target );
	int			sx	= ( int )s.x;
	int			tx	= ( int )t.x;
	int			sy	= ( int )s.y;
	int			ty	= ( int )t.y;
	int			dir	= 1;
	if( sy == ty )
		return;
	if( sy > ty )
	{
		swap( &sx , &tx );
		swap( &sy , &ty );
		swap( &s , &t );
		dir = -1;
	}
	//prev	
	int		ssy	= sy;
	int		tty	= min( ty , ( ssy + cell_anti_scale() ) & ( ~cell_anti_mask() ) );
	float	ssx	= ( float )sx;
	float	ttx	= ( tx - sx ) * ( tty - sy ) / ( float )( ty - sy ) + ( float )sx;
	AddSegmentX( (int)ssx , ssy , (int)ttx , tty , dir );
	ssx	= ttx;
	ssy	= tty;
	
	// center
	float	dx = ( tx - sx ) * cell_anti_scale() / ( float )( ty - sy );
	for( tty = ssy + cell_anti_scale() , ttx = ssx + dx ; tty <= ty ; tty += cell_anti_scale() , ttx += dx )
	{
		AddSegmentX( (int)ssx , ssy , (int)ttx , tty , dir );
		ssx = ttx;
		ssy	= tty;
	}
	
	// post
	if( ssy < ty )
	{
		AddSegmentX( (int)ssx , ssy , tx , ty , dir );
	}
}
//=================================================================================================
//!	set node
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void AddOutline
		(
		OutlineNode*	pnode , 
		const faffine&	affine
		)
{
	if( pnode == 0)
		return;
	fvector2	start	= affine.Transform( fvector2( pnode->x , pnode->y ) );
	fvector2	t_start = start;
	pnode	= pnode->m_next;
	while( pnode != 0 )
	{
		fvector2	target	= affine.Transform( fvector2( pnode->x , pnode->y ) );
		AddSegment( start , target );
		
		// update
		start	= target;
		pnode	= pnode->m_next;
	}
	if( start != t_start )
		AddSegment( start , t_start );
}
//=================================================================================================
//!	get area
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool GetArea
		(
		irect*			rect , 
		const frect&	fbb , 
		const faffine&	affine
		)
{
	fvector2	pnt[4] = 
	{
		fvector2( fbb.xmin , fbb.ymin ) , 
		fvector2( fbb.xmax , fbb.ymin ) , 
		fvector2( fbb.xmin , fbb.ymax ) , 
		fvector2( fbb.xmax , fbb.ymax ) , 
	};
	int		i;
	for( i = 0 ; i < _countof( pnt ) ; i++ )
		pnt[ i ]	= affine.Transform( pnt[i] );
	frect	fr( pnt[0].x , pnt[0].y , pnt[0].x , pnt[0].y );
	for( i = 1 ; i < _countof( pnt ) ; i++ )
	{
		fr.xmin		= min( fr.xmin , pnt[i].x );
		fr.ymin		= min( fr.ymin , pnt[i].y );
		fr.xmax	= max( fr.xmax , pnt[i].x );
		fr.ymax	= max( fr.ymax , pnt[i].y );
	}
	irect	area;
	area.xmin	= ( int )floorf( fr.xmin );
	area.ymin	= ( int )floorf( fr.ymin );
	area.xmax	= ( int )ceilf( fr.xmax );
	area.ymax	= ( int )ceilf( fr.ymax );
	if( area.IsExist() == false )
		return false;
	*rect	= area;
	return true;
}

// "IEdgemapCellInfo" interface functions
public:
//=================================================================================================
bool cb_call GetEdgemapCellInfo
		(
		EdgemapCellInfo*	info
		)
{
	if( m_area.IsExist() == false )
		return false;
	info->m_area	= m_area;
	info->m_line	= m_line.GetPtr();
	return true;
}
// "IEdgemapCell" interface functions
public:
//=================================================================================================
void cb_call SetOutline
		(
		IOutline*		outline , 
		const faffine&	affine
		)
{
	// initialize
	m_allocator->DeallocateAll();

	// area
	OutlineInfo		oi;
	if( false == outline->GetOutlineInfo( &oi ) )
		return;
	if( false == GetArea( &m_area , oi.m_boundbox , affine ) )
		return;
	
	// line
	int		off , num = m_area.Height() + 1;
	m_line.Resize( num );
	CellEdge**	p = &m_line[0];
	for( off = 0 ; off < num ; off++ )
		p[ off ]	= 0;

	// edge
	const OutlineSeg*	os	= oi.m_first;
	while( os != 0 )
	{
		AddOutline( os->m_node , affine );
		os	= os->m_next;
	}
}
// public functions
public:
//=================================================================================================
EdgemapCell_bs
		(
		IMemAllocLump*	alloc
		) :	
		m_allocator( alloc ) , 
		m_line( Expand_ArrayCashType , 128 )
{
	cb_assert( m_allocator != 0 , L"Allocator isn't exist." );
}
//=================================================================================================
~EdgemapCell_bs()
{
	ReleaseAllocator();
}
//=================================================================================================
void ReleaseAllocator()
{
	if( m_allocator == 0 )
		return;
	m_allocator->DeallocateAll();
	m_allocator = 0;
	m_line.Resize( 0 );
	m_area		= irect();
}
};
/**************************************************************************************************
"EdgemapCell" class 
**************************************************************************************************/
class EdgemapCell : 
		public EdgemapCell_bs , 
		virtual public object_base
{
	query_begin();
	iface_hook( IEdgemapCellInfo , IEdgemapCellInfo_IID )
	iface_hook( IEdgemapCell , IEdgemapCell_IID )
	query_end( object_base );

private:
	MemAllocLump_inc		m_allocator;
	
public:
//=================================================================================================
EdgemapCell() : EdgemapCell_bs( &m_allocator ) , m_allocator( sizeof( CellEdge ) * 512 )
{
}
//=================================================================================================
~EdgemapCell()
{
	ReleaseAllocator();
}
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// global variable define

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions define

};	//namespace

//using namespace icubic;		

#pragma pack( pop )			//release align
