/*************************************************************************************************/
/*!
   	@file		imagefileloader_bmp.h
	@author 	Fanzo
 	@date 		2008/3/25
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	"iFace/iLoaderImagefile.h"

#pragma pack( push , 8 )		//set align

namespace icubic
{

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define

#pragma pack( push , 1 )		// set align 
struct BITMAPFILEHEADER 
{
	uint16	bfType; 
	uint32	bfSize; 
	uint16	bfReserved1; 
	uint16	bfReserved2; 
	uint32	bfOffBits; 
}; 
struct BITMAPINFOHEADER
{
	uint32  biSize; 
	int32   biWidth; 
	int32   biHeight; 
	uint16  biPlanes; 
	uint16  biBitCount; 
	uint32  biCompression; 
	uint32  biSizeImage; 
	int32   biXPelsPerMeter; 
	int32   biYPelsPerMeter; 
	uint32  biClrUsed; 
	uint32  biClrImportant; 
};
#pragma pack( pop )			// release align


/**************************************************************************************************
"LoaderImagefile_bmp" class 
**************************************************************************************************/
class LoaderImagefile_bmp : 
	virtual public object_base , 
	public ILoaderImagefile
{
// query
	query_begin();
	iface_hook( ILoaderImagefile , ILoaderImagefile_IID )
	query_end( object_base );
	
// variable member
private:
	DPI					m_dpi;
	isize				m_size;
	int					m_colorbit;
	Array< rgba >		m_colortbl;
	Array< uint8 >		m_image;

// private functions
private:
//=======================================================================
//!	free
//!	@retval			---
//-----------------------------------------------------------------------
void Free()
{
	m_size		= isize();
	m_colorbit	= 0;
	m_colortbl.Resize( 0 );
	m_image.Resize( 0 );
}
//=======================================================================
//!	read color table.
//!	@retval			---
//-----------------------------------------------------------------------
bool LoadColorTable
		(
		iFileStreamRead			&reader , 
		int						palettenum
		)
{
	m_colortbl.Resize( palettenum );
	int		i;
	for( i = 0 ; i < palettenum ; i++ )
	{
		uint8	bytedata[ 4 ];
		if( 4 != reader->Read( &bytedata , sizeof( uint8 ) , 4 , Little_EndianType ) )
			return false;
		m_colortbl[ i ].b	= bytedata[ 0 ];
		m_colortbl[ i ].g	= bytedata[ 1 ];
		m_colortbl[ i ].r	= bytedata[ 2 ];
		m_colortbl[ i ].a	= 255;
	}
	return true;
}
//=======================================================================
//!	load image.
//!	@retval			---
//-----------------------------------------------------------------------
bool LoadImage
		(
		iFileStreamRead			&reader
		)
{
	switch( m_colorbit )
	{
	case 1:
	case 2:
	case 4:
	case 8:
		return LoadImage1to8( reader );

	case 16:
	case 24:
	case 32:
		return LoadImage16to32( reader );
	}
	return false;	
}
//=======================================================================
//!	read 1 , 2 , 4 , 8 bit image
//!	@retval			---
//-----------------------------------------------------------------------
bool LoadImage1to8
		(
		iFileStreamRead			&reader
		)
{
	// allocate memory( even if 1 , 2 , 4 bit , store 1byte. )
	m_image.Resize( m_size.width * m_size.height * 1 );

	//conv width into byte unit.( include padding.)
	int		bytewidth 	= m_size.width * m_colorbit;
	bytewidth	= ( ( bytewidth & 0x1F ) == 0 ) ? bytewidth >> 3 : ( ( bytewidth >> 5 ) + 1 ) << 2;
	
	// set filter and set mod 4
	int		pixperbyte	= 8 / m_colorbit;
	int		filter		= ( 1 << m_colorbit ) - 1;

	//read and store bitdata.
	int		y;
	for( y = 0 ; y < m_size.height ; y++)
	{
		int		off 	= m_size.width*( m_size.height - y - 1 );
		int		countX	= 0;
		int		x;
		for( x = 0 ; x < bytewidth ; x++ )
		{
			// read byte unit.
			uint8	byte;
			if( 1 != reader->Read( &byte , sizeof( byte ) , 1 , Little_EndianType ) )
				return false;

			// store pixel unit by byte.
			int		bitpos	= 8 - m_colorbit;
			int		i;
			for( i = 0 ; i < pixperbyte ; i++ )
			{
				// if not padding , store.
				if( countX < m_size.width )
				{
					m_image[ off ] = ( unsigned char )( ( byte >> bitpos ) & filter );
					bitpos	-= m_colorbit;
					countX++;
					off++;
				}
			}
		}
	}
	return true;
}
//=======================================================================
//!	read image of 16 , 24 , 32 bitperpixel
//!	@retval			---
//-----------------------------------------------------------------------
bool LoadImage16to32
		(
		iFileStreamRead			&reader
		)
{
	int		byteperpix	= 0;
	{
		if( m_colorbit == 16 )
			byteperpix	= 2;
		else if( m_colorbit == 24 )
			byteperpix	= 3;
		else
			byteperpix	= 4;
	}

	//allocate memory.
	m_image.Resize( m_size.width * m_size.height * byteperpix );
	
	//byte of padding.
	int		padbyte	= ( m_size.width * byteperpix % 4 == 0 ) ? 0 : 4 - ( m_size.width * byteperpix % 4 );

	//read bitdata.
	int		y;
	for( y = 0 ; y < m_size.height ; y++ )
	{
		int		off = m_size.width * ( m_size.height - y - 1 ) * byteperpix;
		int		x;
		for( x = 0 ; x < m_size.width ; x++)
		{
			uint8	byte[ 4 ];
			if( byteperpix != reader->Read( byte , sizeof( byte[ 0 ] ) , byteperpix , Little_EndianType ) )
				return false;

			if( byteperpix == 2 )
			{
				m_image[ off ]	= byte[ 0 ];
				m_image[ off + 1 ]= byte[ 1 ];
				off	+= 2;
			}
			else if( byteperpix == 3 )
			{
				m_image[ off ]	= byte[ 2 ];
				m_image[ off + 1 ]= byte[ 1 ];
				m_image[ off + 2 ]= byte[ 0 ];
				off	+= 3;
			}
			else
			{
				m_image[ off ]	= byte[ 3 ];
				m_image[ off + 1 ]= byte[ 2 ];
				m_image[ off + 2 ]= byte[ 1 ];
				m_image[ off + 3 ]= byte[ 0 ];
				off	+= 4;
			}
		}
		// read padding.
		if( false == reader->Seek( reader->GetPosition() + padbyte ) )
			return false;
	}
	return true;
}
/*===========================================================================
get pixel color.
                                               --- 11/10/99 ---
@retval			---
---------------------------------------------------------------------------*/
void GetPixelColor
		(
		pixelformat	format , 
		uint8		*dest , 
		int			pitchbyte
		)
{
	switch( m_colorbit )
	{
	case 1:      
	case 2:      
	case 4:      
	case 8:
		{
			int		off = 0;
			int		y;
			for( y = 0 ; y < m_size.height ; y++ )
			{
				int		x;
				uint8	*p	= dest;
				for( x = 0 ; x < m_size.width ; x++ )
				{
					to_pixel( format , p , m_colortbl[ m_image[ off ] ] );
					p += get_pixel_byte( format );
					off++;
				}
				dest	+= pitchbyte;
			}
		}
		break;
	case 16:
		{
			int		off = 0;
			int		y;
			for( y = 0 ; y < m_size.height ; y++ )
			{
				int		x;
				uint8	*p	= dest;
				for( x = 0 ; x < m_size.width ; x++ )
				{
					uint16	color	= ( uint16 )m_image[ off * 2 ] + ( ( ( uint16 )m_image[ off * 2 + 1 ] ) << 8 );
					rgba	pix;
					pix.b	= ( uint8 )( ( color & 0x1F ) << 3 );
					pix.g	= ( uint8 )( ( ( color >> 5 ) & 0x1F ) << 3 );
					pix.r	= ( uint8 )( ( ( color >> 10 ) & 0x1F ) << 3 );
					pix.a	= 255;
					to_pixel( format , p , pix );
					p += get_pixel_byte( format );
					off++;
				}
				dest	+= pitchbyte;
			}
		}	
		break;
	case 24:
		{
			int		off = 0;
			int		y;
			for( y = 0 ; y < m_size.height ; y++ )
			{
				int		x;
				uint8	*p	= dest;
				for( x = 0 ; x < m_size.width ; x++ )
				{
					rgba	pix;
					pix.r	= m_image[ off * 3 ];
					pix.g	= m_image[ off * 3 + 1 ];
					pix.b	= m_image[ off * 3 + 2 ];
					pix.a	= 255;
					to_pixel( format , p , pix );
					p += get_pixel_byte( format );
					off++;
				}
				dest	+= pitchbyte;
			}
		}
		break;
	case 32:
		{
			int		off = 0;
			int		y;
			for( y = 0 ; y < m_size.height ; y++ )
			{
				int		x;
				uint8	*p	= dest;
				for( x = 0 ; x < m_size.width ; x++ )
				{
					rgba	pix;
					pix.r	= m_image[ off * 4 + 1 ];
					pix.g	= m_image[ off * 4 + 2 ];
					pix.b	= m_image[ off * 4 + 3 ];
					pix.a	= m_image[ off * 4 ];
					to_pixel( format , p , pix );
					p += get_pixel_byte( format );
					off++;
				}
				dest	+= pitchbyte;
			}
		}
		break;
	}
}

	
// "ILoaderImagefile" interface functions
public:
//=================================================================================================
//!	check image format
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool cb_call CheckImagefileSign
		(
		iFileStreamRead			&reader
		)
{
	uint64		save	= reader->GetPosition();
	uint16	sig		= 0;
	if( 1 != reader->Read( &sig , sizeof( sig ) , 1 , Little_EndianType ) )
		return false;
	reader->Seek( save );
	
	if( sig != 19778 )
		return false;
	return true;
}
//=================================================================================================
//!	load
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool cb_call Load
		(
		iFileStreamRead			&reader
		)
{
	//free.
	Free();
	if( reader == false )
		return false;
		
	//save file pointer.
	int64	save	= reader->GetPosition();
	
	//read bitmapfile header.
	BITMAPFILEHEADER	bmfh;
	if( 1 != reader->Read( &bmfh , sizeof( bmfh ) , 1, Little_EndianType ) )
	{
		reader->Seek( save );
		Free();
		return false;
	}
	BITMAPINFOHEADER	bmih;
	if( 1 != reader->Read( &bmih , sizeof( bmih ) , 1 , Little_EndianType ) )
	{
		reader->Seek( save );
		Free();
		return false;
	}
	//check bmp file.4
	if( bmfh.bfType != 19778 )
	{
		reader->Seek( save );
		Free();
		return false;
	}
	// check compress
	if( bmih.biCompression != 0 )
	{
		reader->Seek( save );
		Free();
		return false;
	}
	//store data.
	m_size		= isize( bmih.biWidth , bmih.biHeight );
	m_colorbit	= bmih.biBitCount;
	m_dpi		= DPI( (float)bmih.biXPelsPerMeter * 0.0254f , (float)bmih.biYPelsPerMeter * 0.0254f );
	
	//set palette
	if( bmih.biBitCount != 32 && bmih.biBitCount != 24 && bmih.biBitCount != 16 )
	{
		int		palnum = ( bmih.biClrUsed == 0 ) ? ( 1 << bmih.biBitCount ) : bmih.biClrUsed;
		
		if( false == LoadColorTable( reader , palnum ) )
		{
			reader->Seek( save );
			Free();
			return false;
		}
	}		
	//load image.
	if( false == reader->Seek( save + bmfh.bfOffBits ) )
	{
		reader->Seek( save );
		Free();
		return false;
	}
	if( false == LoadImage( reader ) )
	{
		reader->Seek( save );
		Free();
		return false;
	}
	return true;
}
//=================================================================================================
//!	get number of image.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
int32 cb_call GetImageNum()
{
	if( m_colorbit == 0 )
		return 0;	
	return 1;
}
//=================================================================================================
//!	get dpi
//!	@retval			---
//-------------------------------------------------------------------------------------------------
DPI cb_call GetImageDPI
		(
		int32		imageoff
		)const
{
	return m_dpi;
}
//=================================================================================================
//!	get size of image.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
isize cb_call GetImageSize
		(
		int32		imageoff
		)
{
	if( m_colorbit == 0 )
		return isize();
	if( imageoff != 0 )
		return isize();
	return m_size;
}
//=================================================================================================
//!	get surface
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool cb_call GetImage
		(
		int32		imageoff , 
		pixelformat	destformat , 
		void*		dest , 
		int			pitchbyte
		)
{
	if( m_colorbit == 0 )
		return false;
	if( imageoff != 0 )
		return false;

	GetPixelColor( destformat , ( uint8* )dest , pitchbyte );
	return true;
}
// public functions
public:
//=================================================================================================
//!	construct
//-------------------------------------------------------------------------------------------------
LoaderImagefile_bmp() : m_colorbit( 0 )
{
}
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// global variable define

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions define

};	//namespace

//using namespace icubic;		

#pragma pack( pop )			//release align
