/*************************************************************************************************/
/*!
   	@file		surface.h
	@author 	Fanzo
 	@date 		2008/3/4
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	"iFace/iSurface.h"

#pragma pack( push , 8 )		//set align

namespace icubic
{

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define

/**************************************************************************************************
"SurfaceBitmap" class
**************************************************************************************************/
class SurfaceBitmap
{
	cb_copy_impossible( SurfaceBitmap );
	
// variable member
private:
	pixelformat		m_format;
	uint8			*m_pixel;
	uint8			*m_memory;
	isize			m_size;
	int				m_pitchbyte;
	
#ifdef cb_windows
	HBITMAP			m_hbitmap;
	HDC				m_hdc;
#endif

// public functions
public:
//=================================================================================================
//!	construct
//-------------------------------------------------------------------------------------------------
SurfaceBitmap() : 
		m_format( rgb_pixelformat ) , 
		m_pixel( 0 ) , 
		m_memory( 0 ) , 
		m_pitchbyte( 0 )
#ifdef cb_windows
		, m_hbitmap( NULL )
		, m_hdc( NULL )
#endif		
{

}
//=================================================================================================
//!	destruct
//-------------------------------------------------------------------------------------------------
~SurfaceBitmap()
{
	Destroy();
}
//=================================================================================================
//!	create
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool Create
		(
		const isize	&size , 
		pixelformat	format
		)
{
	Destroy();	
	if( size.width < 0 || size.height < 0 )
		return false;
	m_size				= size;
	m_format			= format;
	m_pitchbyte			= m_size.width * get_pixel_byte( m_format );
	m_pitchbyte			+= ( m_pitchbyte % 4 ) == 0 ? 0 : ( 4 - ( m_pitchbyte % 4 ) );
	if( size.width == 0 || size.height == 0 )
		return true;
		
	int		byte = m_size.height * m_pitchbyte;
	m_memory	= new uint8[ byte + 3 ];
	m_pixel		= ( uint8* )cb_addr_align4( m_memory );
	return true;
}
#ifdef cb_windows
//=================================================================================================
//!	create windows surface
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool Create_rgb
		(
		const isize&	size
		)
{
	if( false == Create_rgba( size ) )
		return false;
	m_format = rgb_pixelformat;
	
	return true;
}
//=================================================================================================
//!	create windows surface
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool Create_rgba
		(
		const isize&	size
		)
{
	Destroy();	
	if( size.width < 0 || size.height < 0 )
		return false;
	m_size				= size;
		
	// create hdc
	HDC	hdc	= ::GetDC( NULL );
	m_hdc	= ::CreateCompatibleDC( hdc );
	ReleaseDC( NULL , hdc );
	if( m_hdc == NULL )
	{
		Destroy();
		return false;
	}

	// create bitmap
	if( size.width != 0 && size.height !=0 )
	{
		BITMAPV4HEADER		info;
		memset( &info , 0 , sizeof( info ) );
		info.bV4Size			= sizeof(BITMAPV4HEADER);
		info.bV4Width			= m_size.width;
		info.bV4Height			= -m_size.height;			// to reverse top and bottom , set minus.
		info.bV4Planes			= 1;
		info.bV4BitCount		= 32;
		info.bV4V4Compression	= BI_BITFIELDS;
		info.bV4SizeImage		= 0;
		info.bV4XPelsPerMeter	= 0;
		info.bV4YPelsPerMeter	= 0;
		info.bV4ClrUsed			= 0;
		info.bV4ClrImportant	= 0;

		info.bV4RedMask			= 0x0000FF00;
		info.bV4GreenMask		= 0x00FF0000;
		info.bV4BlueMask		= 0xFF000000;
		info.bV4AlphaMask		= 0x000000FF;
/*
		info.bV4RedMask			= 0x0000FF00;
		info.bV4GreenMask		= 0x00FF0000;
		info.bV4BlueMask		= 0xFF000000;
		info.bV4AlphaMask		= 0x000000FF;
*/
		//create bitmap
		m_hbitmap = CreateDIBSection( m_hdc , ( BITMAPINFO* )&info , DIB_RGB_COLORS , ( void** )&m_pixel , NULL , 0 );
		if( m_hbitmap == NULL )
		{
			Destroy();
			return false;
		}
		::SelectObject( m_hdc , m_hbitmap );
	}
	// param
	m_format	= rgba_pixelformat;
	m_pitchbyte	= size.width * get_pixel_byte( m_format );
	
	return true;
}
#endif
//=================================================================================================
//!	destroy
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void Destroy()
{
	if( m_memory != 0 )
		delete []m_memory;
	m_memory		= 0;
	m_pixel			= 0;
	m_pitchbyte		= 0;
	m_size			= isize();
	
#ifdef cb_windows
	if( m_hdc != NULL )
		cb_verify( TRUE == ::DeleteDC( m_hdc ) );
	m_hdc	= NULL;
	
	if( m_hbitmap != NULL )
		cb_verify( TRUE == ::DeleteObject( m_hbitmap ) );
	m_hbitmap	= NULL;
#endif
}
//=================================================================================================
//!	get size
//!	@retval			---
//-------------------------------------------------------------------------------------------------
isize GetSize()const
{
	return m_size;
}
//=================================================================================================
//!	get format
//!	@retval			---
//-------------------------------------------------------------------------------------------------
pixelformat GetFormat()const
{
	return m_format;
}
//=================================================================================================
//!	get pixel pointer
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void* GetPixelPtr
		(
		int		*pitch_byte
		)const
{
	*pitch_byte	= m_pitchbyte;
	
	return m_pixel;
}
//=================================================================================================
//!	get pixel pointer
//!	@retval			---
//-------------------------------------------------------------------------------------------------
const void* GetConstPixelPtr
		(
		int		*pitch_byte
		)const
{
	return GetPixelPtr( pitch_byte );
}
#ifdef cb_windows
//=================================================================================================
//!	get hdc
//!	@retval			---
//-------------------------------------------------------------------------------------------------
HDC cb_call GetHDC()
{
	return m_hdc;
}
//=================================================================================================
//!	get hbitmap
//!	@retval			---
//-------------------------------------------------------------------------------------------------
HBITMAP cb_call GetHBITMAP()
{
	return m_hbitmap;
}
#endif
};
/**************************************************************************************************
"Surface" class 
**************************************************************************************************/
class Surface : 
	virtual public object_base , 
	public ISurface
{
// query
	query_begin();
	iface_hook( ISurface , ISurface_IID )
	iface_hook( ISurfaceDest , ISurfaceDest_IID )
	iface_hook( ISurfaceSource , ISurfaceSource_IID )
	query_end( object_base );

// friend
	friend		iSurface CreateSurface( const isize& , pixelformat , const DPI& );
#ifdef cb_windows
	friend		iSurface CreateSurface_win_rgb( const isize& , const DPI& );
	friend		iSurface CreateSurface_win_rgba( const isize& , const DPI& );
#endif

// variable member
private:
	SurfaceBitmap	m_bitmap;
	DPI				m_dpi;
	
// "ISurfaceDest" interface functions
public:
//=================================================================================================
DPI cb_call GetDestDPI()const
{
	return m_dpi;
}
//=================================================================================================
isize cb_call GetDestSize()const
{
	return m_bitmap.GetSize();
}
//=================================================================================================
irect cb_call GetDestAvailableArea()const
{
	return irect( m_bitmap.GetSize() );
}
//=================================================================================================
pixelformat cb_call GetDestFormat()const
{
	return m_bitmap.GetFormat();
}
//=================================================================================================
void* cb_call GetDestPixelPtr
		(
		int		*pitchbyte
		)const
{
	return m_bitmap.GetPixelPtr( pitchbyte );
}
#ifdef cb_windows
//=================================================================================================
HDC cb_call GetDestHDC
		(
		irect*		view	= 0 , 
		irect*		update	= 0
		)
{
	HDC		hdc = m_bitmap.GetHDC();
	if( hdc == NULL )
		return NULL;
	store( view , irect( m_bitmap.GetSize() ) );
	store( update , GetDestAvailableArea() );
	return hdc;
}
#endif

// "ISurfaceSource" interface functions
public:
//=================================================================================================
DPI cb_call GetSourceDPI()const
{
	return m_dpi;
}
//=================================================================================================
irect cb_call GetSourceAvailableArea()const
{
	return irect( m_bitmap.GetSize() );
}
//=================================================================================================
isize cb_call GetSourceSize()const
{
	return m_bitmap.GetSize();
}
//=================================================================================================
pixelformat cb_call GetSourceFormat()const
{
	return m_bitmap.GetFormat();
}
//=================================================================================================
const void* cb_call GetSourcePixelPtr
		(
		int		*pitchbyte
		)const
{
	return m_bitmap.GetPixelPtr( pitchbyte );
}
#ifdef cb_windows
//=================================================================================================
HDC cb_call GetSourceHDC
		(
		irect*		view		= 0 , 
		irect*		available	= 0
		)
{
	HDC		hdc = m_bitmap.GetHDC();
	if( hdc == NULL )
		return NULL;
	irect	v = irect( m_bitmap.GetSize() );
	store( view , v );
	store( available , v );
	return hdc;
}
#endif

// "ISurface" interface functions
public:
//=================================================================================================
DPI cb_call GetDPI()const
{
	return m_dpi;
}
//=================================================================================================
isize cb_call GetSurfaceSize()const
{
	return m_bitmap.GetSize();
}
#ifdef cb_windows
//=================================================================================================
HDC cb_call GetSurfaceHDC()
{
	return m_bitmap.GetHDC();
}
#endif
// public functions
public:
//=================================================================================================
Surface()
{
}
//=================================================================================================
//!	create
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool Create
		(
		const isize		&size , 
		pixelformat		format , 
		const DPI&		dpi
		)
{
	if( false == m_bitmap.Create( size , format ) )
		return false;
	m_dpi	= dpi;
	return true;
}
#ifdef cb_windows
//=================================================================================================
//!	create
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool Create_rgb
		(
		const isize		&size , 
		const DPI&		dpi
		)
{
	if( false == m_bitmap.Create_rgb( size ) )
		return false;
	m_dpi	= dpi;
	return true;
}
//=================================================================================================
//!	create
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool Create_rgba
		(
		const isize		&size , 
		const DPI&		dpi
		)
{
	if( false == m_bitmap.Create_rgba( size ) )
		return false;
	m_dpi	= dpi;
	return true;
}
#endif
};

/**************************************************************************************************
"SurfaceViewport" class 
**************************************************************************************************/
class SurfaceViewport : 
	virtual public object_base , 
	public ISurfaceViewport
{
// query
	query_begin();
	iface_hook( ISurfaceDest , ISurfaceDest_IID )
	iface_hook( ISurfaceSource , ISurfaceSource_IID )
	iface_hook( ISurface , ISurface_IID )
	iface_hook( ISurfaceViewport , ISurfaceViewport_IID )
	query_end( object_base );
	
// variable member
private:
	iSurface		m_surface;
	
	bool			m_viewport_f;
	irect			m_viewport;
	bool			m_clip_f;
	irect			m_clip;

// "ISurfaceDest" interface functions
public:
//=================================================================================================
DPI cb_call GetDestDPI()const
{
	if( m_surface == false )
		return DPI();
	return m_surface->GetDestDPI();
}
//=================================================================================================
isize cb_call GetDestSize()const
{
	if( m_surface == false )
		return isize();
	else if( m_viewport_f == false )
		return m_surface->GetDestSize();
	else
		return m_viewport.Size();
}
//=================================================================================================
irect cb_call GetDestAvailableArea()const
{
	if( m_surface == false )
		return irect();
	else if( m_viewport_f == false )
		return m_surface->GetDestAvailableArea();
	else
	{
		irect	v	= m_viewport.And( irect( m_surface->GetDestAvailableArea() ) );
		v = v.Move( ivector2( -m_viewport.xmin , -m_viewport.ymin ) );
		if( m_clip_f == false )
			return v;
		else
			return v.And( m_clip );
	}
}
//=================================================================================================
pixelformat cb_call GetDestFormat()const
{
	if( m_surface == false )
		return rgb_pixelformat;
	return m_surface->GetDestFormat();
}
//=================================================================================================
void* cb_call GetDestPixelPtr
		(
		int		*pitchbyte
		)const
{
	if( m_surface == false )
	{
		*pitchbyte	= 0;
		return 0;
	}
	else if( m_viewport_f == false )
	{
		return m_surface->GetDestPixelPtr( pitchbyte );
	}
	else
	{
		uint8*	p = ( uint8* )m_surface->GetDestPixelPtr( pitchbyte );
		return ( void* )( p + m_viewport.ymin * ( *pitchbyte ) + m_viewport.xmin * get_pixel_byte( m_surface->GetDestFormat() ) );
	}
}
#ifdef cb_windows
//=================================================================================================
HDC cb_call GetDestHDC
		(
		irect*		view		= 0 , 
		irect*		available	= 0
		)
{
	if( m_surface == false )
		return NULL;
	else if( m_viewport_f == false )
		return m_surface->GetDestHDC( view , available );
	else
	{
		irect	sv;
		HDC		hdc = m_surface->GetDestHDC( &sv , 0 );
		if( hdc == NULL )
			return NULL;
		store( view , m_viewport.Move( ivector2( sv.xmin , sv.ymin ) ) );
		store( available , GetDestAvailableArea() );
		return hdc;
	}
}
#endif

// "ISurfaceSource" interface functions
public:
//=================================================================================================
DPI cb_call GetSourceDPI()const
{
	if( m_surface == false )
		return DPI();
	return m_surface->GetSourceDPI();
}
//=================================================================================================
irect cb_call GetSourceAvailableArea()const
{
	if( m_surface == false )
		return irect();
	else if( m_viewport_f == false )
		return m_surface->GetSourceAvailableArea();
	else
	{
		irect	v	= m_viewport.And( m_surface->GetSourceAvailableArea() );
		v = v.Move( ivector2( -m_viewport.xmin , -m_viewport.ymin ) );
		return v;
	}
}
//=================================================================================================
isize cb_call GetSourceSize()const
{
	return GetSourceAvailableArea().Size();
}
//=================================================================================================
pixelformat cb_call GetSourceFormat()const
{
	if( m_surface == false )
		return rgb_pixelformat;
	return m_surface->GetSourceFormat();
}
//=================================================================================================
const void* cb_call GetSourcePixelPtr
		(
		int		*pitchbyte
		)const
{
	if( m_surface == false )
	{
		*pitchbyte	= 0;
		return 0;
	}
	else if( m_viewport_f == false )
		return m_surface->GetSourcePixelPtr( pitchbyte );
	else
	{
		irect	sa	= m_surface->GetSourceAvailableArea();
		irect	tv	= m_viewport.And( sa );
		tv = tv.Move( ivector2( -sa.xmin , -sa.ymin ) );
		if( tv.IsExist() == false )
		{
			*pitchbyte	= 0;
			return 0;
		}
		else
		{
			uint8*	p = ( uint8* )m_surface->GetSourcePixelPtr( pitchbyte );
			return ( void* )( p + tv.ymin * ( *pitchbyte ) + tv.xmin * get_pixel_byte( m_surface->GetSourceFormat() ) );
		}
	}
}
#ifdef cb_windows
//=================================================================================================
HDC cb_call GetSourceHDC
		(
		irect*		view		= 0 , 
		irect*		available	= 0
		)
{
	if( m_surface == false )
		return NULL;
	else if( m_viewport_f == false )
		return m_surface->GetSourceHDC( view , available );
	else
	{
		irect	sv;
		HDC		hdc = m_surface->GetSourceHDC( &sv , 0 );
		if( hdc == NULL )
			return NULL;
		store( view , m_viewport.Move( ivector2( sv.xmin , sv.ymin ) ) );
		store( available , GetDestAvailableArea() );
		return hdc;
	}
}
#endif

// "ISurface" interface functions
public:
//=================================================================================================
DPI cb_call GetDPI()const
{
	if( m_surface == false )
		return DPI();
	return m_surface->GetDPI();
}
//=================================================================================================
isize cb_call GetSurfaceSize()const
{
	if( m_surface == false )
		return isize();
	else if( m_viewport_f == false )
		return m_surface->GetSurfaceSize();
	else
		return m_viewport.Size();
}
#ifdef cb_windows
//=================================================================================================
HDC cb_call GetSurfaceHDC()
{
	if( m_surface == false )
		return NULL;
	return m_surface->GetSurfaceHDC();
}
#endif

// "ISurfaceViewport" interface functions
public:
//=================================================================================================
void cb_call SetClip
		(
		const irect&	rect		//!< [in]  this is viewport coordinate
		)
{
	m_clip		= rect.And( irect( m_viewport.Size() ) );
	m_clip_f	= true;
}
//=================================================================================================
void cb_call ReleaseClip()
{
	m_clip_f	= false;
}
//=================================================================================================
void cb_call SetViewport
		(
		const irect&	rect
		)
{
	if( rect.IsExist() == false )
	{
		m_viewport		= irect();
		m_clip	= irect();
	}
	else
	{		
		m_viewport		= rect;
		m_clip	= irect( m_viewport.Size() );
	}
	m_viewport_f	= true;
}
//=================================================================================================
void cb_call ReleaseViewport()
{
	m_viewport_f	= false;
}
//=================================================================================================
irect cb_call GetViewport()
{
	if( m_surface == false )
		return irect();
	else if( m_viewport_f == false )
		return irect( m_surface->GetSurfaceSize() );
	else
		return m_viewport;
}
//=================================================================================================
isize cb_call GetViewportMax()
{
	if( m_surface == false )
		return isize();
	return m_surface->GetSurfaceSize();
}

// public functions
public:
//=================================================================================================
SurfaceViewport() : m_viewport_f( false ) , m_clip_f( false )
{
}
//=================================================================================================
void SetSurface
		(
		iSurface&	surface
		)
{
	m_surface		= surface;
	m_viewport_f	= false;
	m_clip_f	= false;
}
//=================================================================================================
void ReleaseSurface()
{
	m_surface.release();
	m_viewport_f	= false;
	m_clip_f	= false;
	m_viewport		= irect();
	m_clip	= irect();
}
};

/**************************************************************************************************
"SurfaceDestViewport" class 
**************************************************************************************************/
class SurfaceDestViewport : 
	virtual public object_base , 
	public ISurfaceDestViewport
{
// query
	query_begin();
	iface_hook( ISurfaceDest , ISurfaceDest_IID )
	iface_hook( ISurfaceDestViewport , ISurfaceDestViewport_IID )
	query_end( object_base );
	
// variable member
private:
	iSurfaceDest	m_surface;
	
	bool			m_viewport_f;
	irect			m_viewport;
	bool			m_clip_f;
	irect			m_clip;

// "ISurfaceDest" interface functions
public:
//=================================================================================================
DPI cb_call GetDestDPI()const
{
	if( m_surface == false )
		return DPI();
	return m_surface->GetDestDPI();
}
//=================================================================================================
isize cb_call GetDestSize()const
{
	if( m_surface == false )
		return isize();
	else if( m_viewport_f == false )
		return m_surface->GetDestSize();
	else
		return m_viewport.Size();
}
//=================================================================================================
irect cb_call GetDestAvailableArea()const
{
	if( m_surface == false )
		return irect();
	else if( m_viewport_f == false )
	{
		if( m_clip_f == false )
			return m_surface->GetDestAvailableArea();
		else
			return m_clip.And( m_surface->GetDestAvailableArea() );
	}
	else
	{
		irect	v	= m_viewport.And( irect( m_surface->GetDestAvailableArea() ) );
		v = v.Move( ivector2( -m_viewport.xmin , -m_viewport.ymin ) );
		if( m_clip_f == false )
			return v;
		else
			return v.And( m_clip );
	}
}
//=================================================================================================
pixelformat cb_call GetDestFormat()const
{
	if( m_surface == false )
		return rgb_pixelformat;
	return m_surface->GetDestFormat();
}
//=================================================================================================
void* cb_call GetDestPixelPtr
		(
		int		*pitchbyte
		)const
{
	if( m_surface == false )
	{
		*pitchbyte	= 0;
		return 0;
	}
	else if( m_viewport_f == false )
	{
		return m_surface->GetDestPixelPtr( pitchbyte );
	}
	else
	{
		uint8*	p = ( uint8* )m_surface->GetDestPixelPtr( pitchbyte );
		return ( void* )( p + m_viewport.ymin * ( *pitchbyte ) + m_viewport.xmin * get_pixel_byte( m_surface->GetDestFormat() ) );
	}
}
#ifdef cb_windows
//=================================================================================================
HDC cb_call GetDestHDC
		(
		irect*		view		= 0 , 
		irect*		available	= 0
		)
{
	if( m_surface == false )
		return NULL;
	else if( m_viewport_f == false )
		return m_surface->GetDestHDC( view , available );
	else
	{
		irect	sv;
		HDC		hdc = m_surface->GetDestHDC( &sv , 0 );
		if( hdc == NULL )
			return NULL;
		store( view , m_viewport.Move( ivector2( sv.xmin , sv.ymin ) ) );
		store( available , GetDestAvailableArea() );
		return hdc;
	}
}
#endif

// "ISurfaceDestViewport" interface functions
public:
//=================================================================================================
void cb_call SetViewport
		(
		const irect&	rect
		)
{
	if( rect.IsExist() == false )
	{
		m_viewport		= irect();
		m_clip	= irect();
	}
	else
	{		
		m_viewport		= rect;
		m_clip	= irect( m_viewport.Size() );
	}
	m_viewport_f	= true;
}
//=================================================================================================
void cb_call ReleaseViewport()
{
	m_viewport_f	= false;
}
//=================================================================================================
irect cb_call GetViewport()
{
	if( m_surface == false )
		return irect();
	else if( m_viewport_f == false )
		return irect( m_surface->GetDestSize() );
	else
		return m_viewport;
}
//=================================================================================================
void cb_call SetClip
		(
		const irect&	rect		//!< [in]  this is viewport coordinate
		)
{
	m_clip		= rect;//rect.And( irect( m_viewport.Size() ) );
	m_clip_f	= true;
}
//=================================================================================================
void cb_call ReleaseClip()
{
	m_clip_f	= false;
}
//=================================================================================================
isize cb_call GetViewportMax()
{
	if( m_surface == false )
		return isize();
	return m_surface->GetDestSize();
}
// public functions
public:
//=================================================================================================
SurfaceDestViewport() : m_viewport_f( false ) , m_clip_f( false )
{
}
//=================================================================================================
void SetSurface
		(
		iSurfaceDest&	surface
		)
{
	m_surface		= surface;
	m_viewport_f	= false;
	m_clip_f		= false;
}
//=================================================================================================
void ReleaseSurface()
{
	m_surface.release();
	m_viewport_f	= false;
	m_clip_f		= false;
	m_viewport		= irect();
	m_clip	= irect();
}
};
///////////////////////////////////////////////////////////////////////////////////////////////////
// global variable define

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions define

//=================================================================================================
//!	create surface
//!	@retval			---
//-------------------------------------------------------------------------------------------------
cb_inline
iSurface CreateSurface
		(
		const isize&	size , 
		pixelformat		format , 
		const DPI&		dpi = DPI()
		)
{
	Surface*	p = new Surface();
	if( p == 0 )
		return iSurface();
	if( false == p->Create( size , format , dpi ) )
	{
		delete p;
		return iSurface();
	}
	object	obj( static_cast<iobject*>( p ) );
	if( obj == false )
	{
		delete p;
		return iSurface();
	}
	iSurface surface	= obj;
	return surface;
}
#ifdef cb_windows
//=================================================================================================
//!	create surface
//!	@retval			---
//-------------------------------------------------------------------------------------------------
cb_inline
iSurface CreateSurface_win_rgb
		(
		const isize&	size , 
		const DPI&		dpi = DPI()
		)
{
	Surface*	p = new Surface();
	if( p == 0 )
		return iSurface();
	if( false == p->Create_rgb( size , dpi ) )
	{
		delete p;
		return iSurface();
	}
	object	obj( static_cast<iobject*>( p ) );
	if( obj == false )
	{
		delete p;
		return iSurface();
	}
	iSurface surface	= obj;
	return surface;
}
//=================================================================================================
//!	create surface
//!	@retval			---
//-------------------------------------------------------------------------------------------------
cb_inline
iSurface CreateSurface_win_rgba
		(
		const isize&	size , 
		const DPI&		dpi = DPI()
		)
{
	Surface*	p = new Surface();
	if( p == 0 )
		return iSurface();
	if( false == p->Create_rgba( size , dpi ) )
	{
		delete p;
		return iSurface();
	}
	object	obj( static_cast<iobject*>( p ) );
	if( obj == false )
	{
		delete p;
		return iSurface();
	}
	iSurface surface	= obj;
	return surface;
}
#endif
};	//namespace

//using namespace icubic;		

#pragma pack( pop )			//release align
