/*************************************************************************************************/
/*!
   	@file		Path.h
	@author 	Fanzo
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	"iFace/iPath.h"
#include	"iFace/iEdgemapOutline.h"
#include	"PathLine.h"
#include	"PathQuadratic.h"
#include	"PathCubic.h"
#include	"PathArc.h"

#pragma pack( push , 8 )		//set align

namespace icubic
{

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define

/**************************************************************************************************
"t_Path_bs" class 
**************************************************************************************************/
template
		<
		class t_value , 
		class t_point , 
		class t_seg , 
		class t_move , 
		class t_line , 
		class t_bezierq , 
		class t_bezierc , 
		class t_arcangle , 
		class t_arcsweep , 
		class t_node , 
		class t_expr , 
		class t_factor , 
		class t_info , 
		class t_logicinfo
 		>
class t_Path_bs
{
// member class
protected:
	class Allocator : public IMemAllocLump
	{
	private:
		MemAllocLump_inc	m_allocator;
	public:
		Allocator
				(
				uint32		pool
				) : m_allocator( pool ){}
		void* cb_call Allocate
			(
			uint32		size
			)
		{
			return m_allocator.Allocate( size );
		}
		~Allocator()
		{
			m_allocator.DeallocateAll();
		}
		void cb_call DeallocateAll()
		{
			// none
		}
		void Reset()
		{
			m_allocator.DeallocateAll();
		}
	};
// variable member
private:
	Allocator					m_allocator;
	InstanceFactory<t_move>		m_move_fc;
	InstanceFactory<t_line>		m_line_fc;
	InstanceFactory<t_bezierc>	m_bezierc_fc;
	InstanceFactory<t_bezierq>	m_bezierq_fc;
	InstanceFactory<t_arcangle>	m_arc_angle_fc;
	InstanceFactory<t_arcsweep>	m_arc_sweep_fc;
	InstanceFactory<t_expr>		m_expr_fc;
	InstanceFactory<t_factor>	m_factor_fc;

protected:
	t_logicinfo			m_info;
	
// protected functions
protected:
//=================================================================================================
t_Path_bs() : 
		m_allocator( sizeof( t_bezierc ) * 128 ) , 
		m_move_fc( &m_allocator ) , 
		m_line_fc( &m_allocator ) , 
		m_bezierc_fc( &m_allocator ) , 
		m_bezierq_fc( &m_allocator ) , 
		m_arc_angle_fc( &m_allocator ) , 
		m_arc_sweep_fc( &m_allocator ) , 
		m_expr_fc( &m_allocator ) , 
		m_factor_fc( &m_allocator )
{
	m_info.m_node	= 0;
}
//=================================================================================================
~t_Path_bs()
{
	Reset();
}
//=================================================================================================
t_seg* CreatePathSeg
		(
		PathSegType		type
		)
{
	if( type == Move_PathSegType )
		return	m_move_fc.Create();
	else if( type == Line_PathSegType )
		return	m_line_fc.Create();
	else if( type == BezierQ_PathSegType )
		return	m_bezierq_fc.Create();
	else if( type == BezierC_PathSegType )
		return	m_bezierc_fc.Create();
	else if( type == ArcAngle_PathSegType )
		return m_arc_angle_fc.Create();
	else if( type == ArcSweep_PathSegType )
		return m_arc_sweep_fc.Create();
	cb_assert( false , L"invalid PathSegType" );
	return 0;
}
//=================================================================================================
t_node* CreatePathNode
		(
		PathNodeType		type
		)
{
	if( type == Factor_PathNodeType )
		return	m_factor_fc.Create();
	else if( type == Expr_PathNodeType )
		return	m_expr_fc.Create();
	cb_assert( false , L"invalid PathNode" );
	return 0;
}
//=================================================================================================
t_factor* UpdateFactor()
{
	if( m_info.m_node == 0 || ( m_info.m_node != 0 && m_info.m_node->m_type != Factor_PathNodeType ) )
	{
		Reset();
		t_factor*	node = ( t_factor* )CreatePathNode( Factor_PathNodeType );
		m_info.m_node = node;
		return node;
	}
	else
		return (t_factor*)m_info.m_node;
}
//=================================================================================================
t_seg* AddPathSeg
		(
		PathSegType		type
		)
{
	// path element
	t_factor*	factor	= UpdateFactor();
	t_seg*		seg		= CreatePathSeg( type );

	// link
	( ( factor->m_info.m_last == 0 ) ? factor->m_info.m_first : factor->m_info.m_last->m_next ) = seg;
	factor->m_info.m_last	= seg;
	return seg;
}
//=================================================================================================
t_seg* Copy
		(
		const t_seg*	s_seg
		)
{
	t_seg*	d_seg	= CreatePathSeg( s_seg->m_type );
	if( s_seg->m_type == Move_PathSegType )
		*(t_move*)d_seg		= *(t_move*)s_seg;
	else if( s_seg->m_type == Line_PathSegType )
		*(t_line*)d_seg		= *(t_line*)s_seg;
	else if( s_seg->m_type == BezierQ_PathSegType )
		*(t_bezierq*)d_seg	= *(t_bezierq*)s_seg;
	else if( s_seg->m_type == BezierC_PathSegType )
		*(t_bezierc*)d_seg	= *(t_bezierc*)s_seg;
	else if( s_seg->m_type == ArcAngle_PathSegType )
		*(t_arcangle*)d_seg	= *(t_arcangle*)s_seg;
	else if( s_seg->m_type == ArcSweep_PathSegType )
		*(t_arcsweep*)d_seg	= *(t_arcsweep*)s_seg;
	else
	{
		cb_assert( false , L"invalid PathSegType" );
	}
	d_seg->m_next		= 0;
	return d_seg;
}
//=================================================================================================
t_info Copy
		(
		const t_info&	s_info
		)
{
	t_info	d_info;
	d_info.m_transform	= s_info.m_transform;
	
	t_seg**	d_seg	= &d_info.m_first;
	t_seg*	s_seg	= s_info.m_first;
	while( s_seg != 0 )
	{
		*d_seg			= Copy( s_seg );
		d_info.m_last	= *d_seg;
		d_seg			= &(*d_seg)->m_next;
		s_seg			= s_seg->m_next;
	}	
	return d_info;
}
//=================================================================================================
t_factor* CopyFactor
		(
		const t_factor*	s_factor
		)
{
	t_factor*	d_factor	= m_factor_fc.Create();
	d_factor->m_type	= s_factor->m_type;
	d_factor->m_info	= Copy( s_factor->m_info );
	return d_factor;
}
//=================================================================================================
t_node* Copy
		(
		const t_node*	s_node
		)
{
	if( s_node == 0 )
		return 0;
	if( s_node->m_type == Factor_PathNodeType )
		return ( t_node* )CopyFactor( ( const t_factor* )s_node );
	else if( s_node->m_type == Expr_PathNodeType )
	{
		t_expr*	s	= ( t_expr* )s_node;
		t_expr*	d	= ( t_expr* )CreatePathNode( Expr_PathNodeType );
		*d	= *s;
		d->m_left	= Copy( s->m_left );
		d->m_right	= Copy( s->m_right );
		return d;
	}
	cb_assert( false , L"invalid PathNode" );
	return 0;
}
//=================================================================================================
void Reset()
{
	m_info.m_node	= 0;
	m_move_fc.Reset();
	m_line_fc.Reset();
	m_bezierc_fc.Reset();
	m_bezierq_fc.Reset();
	m_arc_angle_fc.Reset();
	m_arc_sweep_fc.Reset();
	m_expr_fc.Reset();
	m_factor_fc.Reset();
	m_allocator.Reset();
}
//=================================================================================================
bool cb_call Operate
		(
		PathOperator			ope , 
		const t_logicinfo&		src , 
		const faffine&			affine
		)
{
	if( src.m_node == 0 )
		return false;
	if( m_info.m_node == 0 )
		return false;
	t_expr*	node	= ( t_expr* )CreatePathNode( Expr_PathNodeType );
	node->m_ope		= ope;	
	node->m_left	= m_info.m_node;
	node->m_right	= Copy( src.m_node );

	if( node->m_right->m_type == Expr_PathNodeType )
		( (t_expr*)node->m_right )->m_transform				= affine * ( (t_expr*)node->m_right )->m_transform;
	else if( node->m_right->m_type == Factor_PathNodeType )
		( (t_factor*)node->m_right )->m_info.m_transform	= affine * ( (t_factor*)node->m_right )->m_info.m_transform;

	m_info.m_node	= node;
	return true;
}
//=================================================================================================
void Move
		(
		const t_point&	pos , 
		const faffine&	transform , 
		bool			close
		)
{
	t_move* seg = (t_move*)AddPathSeg( Move_PathSegType );
	seg->m_sx			= pos.x;
	seg->m_sy			= pos.y;
	seg->m_transform	= transform;
	seg->m_close		= close;
}
//=================================================================================================
void Move
		(
		int					pntnum , 
		const t_value*	pnt , 
		const faffine&		transform , 
		bool				close
		)
{
	cb_assert( pntnum == 2 , L"pntnum is invalid." );
	t_move* seg = (t_move*)AddPathSeg( Move_PathSegType );
	seg->m_sx			= pnt[0];
	seg->m_sy			= pnt[1];
	seg->m_transform	= transform;
	seg->m_close		= close;
}
//=================================================================================================
void Line
		(
		const t_point&	pnt
		)
{
	t_line*	e = (t_line*)AddPathSeg( Line_PathSegType );
	e->m_tx		= pnt.x;
	e->m_ty		= pnt.y;
}
//=================================================================================================
void Line
		(
		int				pntnum , 
		const t_point*	pnt			//!< [in] tp,tp,...
		)
{
	int		off , num = pntnum;
	for( off = 0 ; off < num ; off++ )
	{
		t_line*	e = (t_line*)AddPathSeg( Line_PathSegType );
		e->m_tx		= pnt->x;
		e->m_ty		= pnt->y;
		pnt++;
	}
}
//=================================================================================================
void Line
		(
		int				pntnum , 
		const t_value*	pnt			//!< [in] tx , ty ...
		)
{
	cb_assert( pntnum % 2 == 0 , L"pntnum is invalid." );
	int		off , num = pntnum / 2;
	for( off = 0 ; off < num ; off++ )
	{
		t_line*	e = (t_line*)AddPathSeg( Line_PathSegType );
		e->m_tx		= *pnt++;
		e->m_ty		= *pnt++;
	}
}
//=================================================================================================
void BezierQ
		(
		const t_point&	hp , 
		const t_point&	tp
		)
{
	t_bezierq*	e = (t_bezierq*)AddPathSeg( BezierQ_PathSegType );
	e->m_hx		= hp.x;
	e->m_hy		= hp.y;
	e->m_tx		= tp.x;
	e->m_ty		= tp.y;
}
//=================================================================================================
void cb_call BezierQ
		(
		int				pntnum , 
		const t_point*	pnt			//!< [in] hp , tp , ...
		)
{
	cb_assert( pntnum % 2 == 0 , L"pntnum is invalid." );
	int		off , num = pntnum / 2;
	for( off = 0 ; off < num ; off++ )
	{
		t_bezierq*	e = (t_bezierq*)AddPathSeg( BezierQ_PathSegType );
		e->m_hx		= pnt->x;
		e->m_hy		= pnt->y;
		pnt++;
		e->m_tx		= pnt->x;
		e->m_ty		= pnt->y;
		pnt++;
	}
}
//=================================================================================================
void BezierQ
		(
		int				pntnum , 
		const t_value*	pnt			//!< [in] hx , hy , tx , ty ...
		)
{
	cb_assert( pntnum % 4 == 0 , L"pntnum is invalid." );
	int		off , num = pntnum / 4;
	for( off = 0 ; off < num ; off++ )
	{
		t_bezierq*	e = (t_bezierq*)AddPathSeg( BezierQ_PathSegType );
		e->m_hx		= *pnt++;
		e->m_hy		= *pnt++;
		e->m_tx		= *pnt++;
		e->m_ty		= *pnt++;
	}
}
//=================================================================================================
void BezierC
		(
		const t_point&	h0p , 
		const t_point&	h1p , 
		const t_point&	tp
		)
{
	t_bezierc*	e = (t_bezierc*)AddPathSeg( BezierC_PathSegType );
	e->m_h0x	= h0p.x;
	e->m_h0y	= h0p.y;
	e->m_h1x	= h1p.x;
	e->m_h1y	= h1p.y;
	e->m_tx		= tp.x;
	e->m_ty		= tp.y;
}
//=================================================================================================
void cb_call BezierC
		(
		int				pntnum , 
		const t_point*	pnt			//!< [in] h0p , h1p , tp , ...
		)
{
	cb_assert( pntnum % 3 == 0 , L"pntnum is invalid." );
	int		off , num = pntnum / 3;
	for( off = 0 ; off < num ; off++ )
	{
		t_bezierc*	e = (t_bezierc*)AddPathSeg( BezierC_PathSegType );
		e->m_h0x	= pnt->x;
		e->m_h0y	= pnt->y;
		pnt++;
		e->m_h1x	= pnt->x;
		e->m_h1y	= pnt->y;
		pnt++;
		e->m_tx		= pnt->x;
		e->m_ty		= pnt->y;
		pnt++;
	}
}
//=================================================================================================
void BezierC
		(
		int				pntnum , 
		const t_value*	pnt			//!< [in] h0x , h0y , h1x , h1y , tx , ty ...
		)
{
	cb_assert( pntnum % 6 == 0 , L"pntnum is invalid." );
	int		off , num = pntnum / 6;
	for( off = 0 ; off < num ; off++ )
	{
		t_bezierc*	e = (t_bezierc*)AddPathSeg( BezierC_PathSegType );
		e->m_h0x	= *pnt++;
		e->m_h0y	= *pnt++;
		e->m_h1x	= *pnt++;
		e->m_h1y	= *pnt++;
		e->m_tx		= *pnt++;
		e->m_ty		= *pnt++;
	}
}
//=================================================================================================
void Arc
		(
		const t_point&	cp , 
		float			dy , 
		float			angle
		)
{
	t_arcangle*	e = (t_arcangle*)AddPathSeg( ArcAngle_PathSegType );
	e->m_cx		= cp.x;
	e->m_cy		= cp.y;
	e->m_dy		= dy;
	e->m_angle	= angle;
}
//=================================================================================================
void Arc
		(
		const t_value&	rx , 
		const t_value&	ry , 
		float			x_axis_rot , 
		bool			large , 
		bool			sweep , 
		const t_point&	tp
		)
{
	t_arcsweep*	e	= (t_arcsweep*)AddPathSeg( ArcSweep_PathSegType );
	e->m_rx			= rx;
	e->m_ry			= ry;
	e->m_x_axis_rot	= x_axis_rot;
	e->m_large		= large;
	e->m_sweep		= sweep;
	e->m_tx			= tp.x;
	e->m_ty			= tp.y;
}
//=================================================================================================
void MulTransform
		(
		const faffine&	affine
		)
{
	if( m_info.m_node == 0 )
		UpdateFactor()->m_info.m_transform	= affine;
	else if( m_info.m_node->m_type == Expr_PathNodeType )
		( (t_expr*)m_info.m_node )->m_transform	= affine * ( (t_expr*)m_info.m_node )->m_transform;
	else if( m_info.m_node->m_type == Factor_PathNodeType )
		( (t_factor*)m_info.m_node )->m_info.m_transform	= affine * ( (t_factor*)m_info.m_node )->m_info.m_transform;
}
//=================================================================================================
faffine GetTransform()const
{
	if( m_info.m_node == 0 )
		return faffine();
	else if( m_info.m_node->m_type == Expr_PathNodeType )
		return ( (t_expr*)m_info.m_node )->m_transform;
	else if( m_info.m_node->m_type == Factor_PathNodeType )
		return ( (t_factor*)m_info.m_node )->m_info.m_transform;
	return faffine();
}
//=================================================================================================
void ResetTransform()
{
	if( m_info.m_node == 0 && m_info.m_node->m_type == Expr_PathNodeType )
		( (t_expr*)m_info.m_node )->m_transform	= faffine();
	else if( m_info.m_node == 0 && m_info.m_node->m_type == Factor_PathNodeType )
		( (t_factor*)m_info.m_node )->m_info.m_transform	= faffine();
}
};
/**************************************************************************************************
"Path" class 
**************************************************************************************************/
class Path : 
	virtual public object_base , 
	public IPath , 
	public t_Path_bs<float , fvector2 , PathSeg , PathSegMove , PathSegLine , PathSegBezierQ , PathSegBezierC , PathSegArcAngle , PathSegArcSweep , PathNode , PathNodeExpr , PathNodeFactor , PathInfo , PathLogicInfo>
{
// query
	query_begin();
	iface_hook( IPath , IPath_IID )
	iface_hook( IPathInfo , IPathInfo_IID )
	iface_hook( IPathLogicInfo , IPathLogicInfo_IID )
	query_end( object_base );

// variable member
private:
	
// "IPathInfo" interface functions
public:
//=================================================================================================
const PathInfo* cb_call GetPathInfo()
{
	return &(UpdateFactor()->m_info);
}
// "IPathLogicInfo" interface functions
public:
//=================================================================================================
const PathLogicInfo* cb_call GetPathLogicInfo()
{
	return &m_info;
}

// "IPathTo" interface functions
//=================================================================================================
void cb_call Move
		(
		const fvector2&		pos , 
		const faffine&		transform , 
		bool				close
		)
{
	t_Path_bs::Move( pos , transform , close );
}
//=================================================================================================
void cb_call Move
		(
		int				pntnum , 
		const float*	pnt , 
		const faffine&	transform , 
		bool			close
		)
{
	t_Path_bs::Move( pntnum , pnt , transform , close );
}
//=================================================================================================
void cb_call Line
		(
		const fvector2&		tp
		)
{
	t_Path_bs::Line( tp );
}
//=================================================================================================
void cb_call Line
		(
		int					pntnum , 
		const fvector2*		pnt			//!< [in] tp,tp,...
		)
{
	t_Path_bs::Line( pntnum , pnt );
}
//=================================================================================================
void cb_call Line
		(
		int				pntnum , 
		const float*	pnt
		)
{
	t_Path_bs::Line( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierQ
		(
		const fvector2&		hp , 
		const fvector2&		tp
		)
{
	t_Path_bs::BezierQ( hp , tp );
}
//=================================================================================================
void cb_call BezierQ
		(
		int				pntnum , 
		const fvector2*	pnt			//!< [in] hp , tp , ...
		)
{
	t_Path_bs::BezierQ( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierQ
		(
		int				pntnum , 
		const float*	pnt
		)
{
	t_Path_bs::BezierQ( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierC
		(
		const fvector2&		h0p , 
		const fvector2&		h1p , 
		const fvector2&		tp
		)
{
	t_Path_bs::BezierC( h0p , h1p , tp );
}
//=================================================================================================
void cb_call BezierC
		(
		int					pntnum , 
		const fvector2*		pnt			//!< [in] h0p , h1p , tp , ...
		)
{
	t_Path_bs::BezierC( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierC
		(
		int				pntnum , 
		const float*	pnt
		)
{
	t_Path_bs::BezierC( pntnum , pnt );
}
//=================================================================================================
void cb_call Arc
		(
		const fvector2&		cp , 
		float				dy , 
		float				angle
		)
{
	t_Path_bs::Arc( cp , dy , angle );
}
//=================================================================================================
void cb_call Arc
		(
		float				rx , 
		float				ry , 
		float				x_axis_rot , 
		bool				large , 
		bool				sweep , 
		const fvector2&		tp
		)
{
	t_Path_bs::Arc( rx , ry , x_axis_rot , large , sweep , tp );
}

// "IPath" interface functions
public:
//=================================================================================================
void cb_call Reset()
{
	t_Path_bs::Reset();
}
//=================================================================================================
void cb_call Copy
		(
		IPathInfo*	src
		)
{
	Reset();
	if( src == 0 )
		return;
	UpdateFactor()->m_info	= t_Path_bs::Copy( *src->GetPathInfo() );
}
//=================================================================================================
void cb_call MulTransform
		(
		const faffine&	affine
		)
{
	t_Path_bs::MulTransform( affine );
}
};
/**************************************************************************************************
"PathLogic" class 
**************************************************************************************************/
class PathLogic : 
	virtual public object_base , 
	public IPathLogic , 
	public t_Path_bs<float , fvector2 , PathSeg , PathSegMove , PathSegLine , PathSegBezierQ , PathSegBezierC , PathSegArcAngle , PathSegArcSweep , PathNode , PathNodeExpr , PathNodeFactor , PathInfo , PathLogicInfo>
{
// query
	query_begin();
	iface_hook( IPathLogic , IPathLogic_IID )
	iface_hook( IPathLogicInfo , IPathLogicInfo_IID )
	query_end( object_base );

// "IPathLogicInfo" interface functions
public:
//=================================================================================================
//!	get path info
//!	@retval			---
//-------------------------------------------------------------------------------------------------
const PathLogicInfo* cb_call GetPathLogicInfo()
{
	return &m_info;
}

// "IPathTo" interface functions
//=================================================================================================
void cb_call Move
		(
		const fvector2&		pos , 
		const faffine&		transform , 
		bool				close
		)
{
	t_Path_bs::Move( pos , transform , close );
}
//=================================================================================================
void cb_call Move
		(
		int				pntnum , 
		const float*	pnt , 
		const faffine&	transform , 
		bool			close
		)
{
	t_Path_bs::Move( pntnum , pnt , transform , close );
}
//=================================================================================================
void cb_call Line
		(
		const fvector2&		tp
		)
{
	t_Path_bs::Line( tp );
}
//=================================================================================================
void cb_call Line
		(
		int					pntnum , 
		const fvector2*		pnt			//!< [in] tp,tp,...
		)
{
	t_Path_bs::Line( pntnum , pnt );
}
//=================================================================================================
void cb_call Line
		(
		int				pntnum , 
		const float*	pnt
		)
{
	t_Path_bs::Line( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierQ
		(
		const fvector2&		hp , 
		const fvector2&		tp
		)
{
	t_Path_bs::BezierQ( hp , tp );
}
//=================================================================================================
void cb_call BezierQ
		(
		int				pntnum , 
		const fvector2*	pnt			//!< [in] hp , tp , ...
		)
{
	t_Path_bs::BezierQ( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierQ
		(
		int				pntnum , 
		const float*	pnt
		)
{
	t_Path_bs::BezierQ( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierC
		(
		const fvector2&		h0p , 
		const fvector2&		h1p , 
		const fvector2&		tp
		)
{
	t_Path_bs::BezierC( h0p , h1p , tp );
}
//=================================================================================================
void cb_call BezierC
		(
		int					pntnum , 
		const fvector2*		pnt			//!< [in] h0p , h1p , tp , ...
		)
{
	t_Path_bs::BezierC( pntnum , pnt );
}
//=================================================================================================
void cb_call BezierC
		(
		int				pntnum , 
		const float*	pnt
		)
{
	t_Path_bs::BezierC( pntnum , pnt );
}
//=================================================================================================
void cb_call Arc
		(
		const fvector2&		cp , 
		float				dy , 
		float				angle
		)
{
	t_Path_bs::Arc( cp , dy , angle );
}
//=================================================================================================
void cb_call Arc
		(
		float				rx , 
		float				ry , 
		float				x_axis_rot , 
		bool				large , 
		bool				sweep , 
		const fvector2&		tp
		)
{
	t_Path_bs::Arc( rx , ry , x_axis_rot , large , sweep , tp );
}

// "IPathLogic" interface functions
public:
//=================================================================================================
void cb_call Reset()
{
	t_Path_bs::Reset();
}
//=================================================================================================
void cb_call Copy
		(
		IPathLogicInfo*		src
		)
{
	Reset();
	if( src == 0 )
		return;
	m_info.m_node = t_Path_bs::Copy( src->GetPathLogicInfo()->m_node );
}
//=================================================================================================
bool cb_call Operate
		(
		PathOperator		ope , 
		IPathLogicInfo*		src , 
		const faffine&		affine
		)
{
	return t_Path_bs::Operate( ope , *src->GetPathLogicInfo() , affine );
}
//=================================================================================================
void cb_call MulTransform
		(
		const faffine&	affine
		)
{
	t_Path_bs::MulTransform( affine );
}
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// global variable define

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions define

};	//namespace

//using namespace icubic;		

#pragma pack( pop )			//release align
