/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkconv_jisx0201_kana.c,v 1.4 2002/04/11 10:06:39 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "skkutils.h"
#include "skkconv_jisx0201_kana.h"
#include "skkconv_jisx0201_kana_private.h"

static const gchar *
translate_to_jisx0201_kana (const gchar *str)
{
	gint i;
	if (!str)
		return NULL;
	for (i = 0; i < SKKCONV_JISX0201_SIZE ; i++) {
		if (!strncmp (jisx0201_list[i].katakana , str, strlen (jisx0201_list[i].katakana))) {
			return (jisx0201_list[i]).jisx0201_kana;
		}
	}
	return NULL;
}

static gboolean
is_jisx0201_kana (const gchar *str)
{
	gint len;
	if (!str)
		return FALSE;
	len = strlen (str);
	if (len < 2)
		return FALSE;
	if ( ((str[0] & 0xff) == 0x8e)) { /* dirty hack */
		return TRUE;
	}
	return FALSE;
}

gchar *
skkconv_katakana_to_jisx0201_kana (const gchar *katakana)
{
	gchar *copy,*copy2;
	gchar *ret = NULL;
	gchar *tmp_ret;
	const gchar *trans;
	if (!katakana)
		return NULL;
	copy = copy2 = g_strdup (katakana);
	for (; copy2 && *copy2; copy2 += skk_utils_charbytes (copy2)) {
		trans = translate_to_jisx0201_kana (copy2);
		if (trans) {
			if (!ret) {
				ret = g_strdup (trans);
			} else {
				tmp_ret = ret;
				ret = g_strconcat (ret, trans, NULL);
				g_free (tmp_ret);
			}
		}
	}
	g_free (copy);
	return ret;
}

gchar*
skkconv_hiragana_to_jisx0201_kana (const gchar *hiragana)
{
	/* TODO */
	gchar *copy,*copy2;
	gchar *ret = NULL;
	gchar *tmp_ret;
	const gchar *trans = NULL;
	if (!hiragana)
		return NULL;
	copy = copy2 = g_strdup (hiragana);
	for (;  copy2 && *copy2; copy2 += skk_utils_charbytes (copy2)) {
		if ((*copy2 & 0xff) == 0xa4) {
			*copy2 += 0x1;
		}
		trans = translate_to_jisx0201_kana (copy2);
		if (trans) {
			if (!ret) {
				ret = g_strdup (trans);
			} else {
				tmp_ret = ret;
				ret = g_strconcat (ret, trans, NULL);
				g_free (tmp_ret);
			}
		}
	}
	g_free (copy);
	return ret;
}

static gint
find_kana (gconstpointer p1, gconstpointer p2)
{
	return strncmp ((const gchar *)p1, ((jisx0201_rule_list *)p2)->jisx0201_kana, 
			strlen (((jisx0201_rule_list *)p2)->jisx0201_kana));
}

gchar*
skkconv_jisx0201_kana_to_katakana (const gchar *jisx0201_kana)
{
	jisx0201_rule_list *list;
	gchar *copy,*copy2;
	gchar *ret = NULL;
	gchar *tmp_ret;
	if (!jisx0201_kana)
		return NULL;
	copy = copy2 = g_strdup (jisx0201_kana);
	for (; copy2 && *copy2; copy2 += skk_utils_charbytes (copy2)) {
		list = bsearch (copy2, jisx0201_list, SKKCONV_JISX0201_SIZE, 
				sizeof (jisx0201_rule_list), find_kana);
		if (list) {
			if (ret) {
				tmp_ret = ret;
				ret = g_strconcat (ret, list->katakana, NULL);
				g_free (tmp_ret);
			} else {
				ret = g_strdup (list->katakana);
			}
			/* FIXME DIRTYHACK */
			if (strlen (list->jisx0201_kana) == 4) {
				/* increment index */
				copy2 += skk_utils_charbytes (copy2);
			}
		}  else {
			/* FIXME dirty hack */
			/* cannot assume 1 byte */
			if (ret) {
				tmp_ret = ret;
				ret = g_strdup_printf ("%s%c", ret, *copy2);
				g_free (tmp_ret);
			} else {
				ret = g_strdup_printf ("%c", *copy2);
			}
		}
	}
	return ret;
}

#ifdef SKKCONV_JISX0201_KANA_DEBUG
int
main (void)
{
	printf ("from Hiragana %s\n", skkconv_hiragana_to_jisx0201_kana ("1"));
	printf ("from Katakana %s\n", skkconv_katakana_to_jisx0201_kana (""));
	printf ("from Katakana %s\n", skkconv_katakana_to_jisx0201_kana ("Хӥ֥٥ܥѥԥץڥ"));
	printf ("from Katakana %s\n", skkconv_katakana_to_jisx0201_kana (""));
	printf ("a %s\n", skkconv_jisx0201_kana_to_katakana ("1"));
	printf ("a %s\n", skkconv_jisx0201_kana_to_katakana ("ގގގގގʎߎˎߎ̎ߎ͎ߎΎ"));
	printf ("a %s\n", skkconv_jisx0201_kana_to_katakana ("ގގގގގ"));
	printf ("from Katakana %s\n", skkconv_katakana_to_jisx0201_kana (""));
	return 0;
}
#endif
