/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkconv_kana.c,v 1.1.1.1 2002/04/06 02:23:30 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <string.h>
#include <glib.h>
#include <fcntl.h>
#include <db.h>
#include "skkconv_kana_private.h"
#include "skkutils.h"

static DB* rule_db = NULL;
static gboolean initialized = FALSE;

G_LOCK_DEFINE_STATIC(search);
G_LOCK_DEFINE_STATIC(conv_kana);
G_LOCK_DEFINE_STATIC(conv_kata);

static gboolean
is_hiragana (const gchar *str)
{
	gint len;
	if (!str)
		return FALSE;
	len = strlen (str);
	if (len < 2)
		return FALSE;
	if ( ((str[0] & 0xff) == 0xa4)) { /* dirty check */
		return TRUE;
	}
	return FALSE;
}

static gboolean
is_katakana (const gchar *str)
{
	gint len;
	if (!str)
		return FALSE;
	len = strlen (str);
	if (len < 2)
		return FALSE;
	if (((str[0] & 0xff) == 0xa5)) {
		return TRUE;
	}
	return FALSE;
}

static base_rule_list*
do_search (const gchar *key)
{
	DBT db_key,db_data;
	int err;
	memset (&db_key,0,sizeof (DBT));
	memset (&db_data,0,sizeof (DBT));
	db_key.size = strlen (key) + 1;
	db_key.data = (void*)key;
	if ((err = rule_db->get (rule_db,NULL,&db_key,&db_data,0)) != 0) {
		if (err != DB_NOTFOUND) 
			fprintf (stderr,"iiimf-skk: do_search %s\n",db_strerror (err));
		return NULL;
	}
	/* FOUND */
	return (base_rule_list*)(db_data.data);
}

gint
skkconv_is_exist (const gchar *key)
{
	DBT db_key,db_data;
	DBC *db_cursor;
	int err;
	int exist_count = 0;
	G_LOCK (search);
	if (!initialized)
		skkconv_rule_init ();
	memset (&db_key,0,sizeof (DBT));
	memset (&db_data,0,sizeof (DBT));
	rule_db->cursor (rule_db,NULL,&db_cursor,0);
	db_cursor->c_get (db_cursor,&db_key,&db_data,DB_FIRST);
	if (!strncmp (key,(const char*)db_key.data,strlen(key))) {
		exist_count++;
	}
	while (TRUE) {
		err = db_cursor->c_get (db_cursor,&db_key,&db_data,DB_NEXT);
		if (err == DB_NOTFOUND) {
			break;
		}
		if (!strncmp (key,(const char*)db_key.data,strlen(key))) {
#if 0
			db_cursor->c_close (db_cursor);
			return TRUE;
#else
			exist_count++;
#endif
		}
	}
	db_cursor->c_close (db_cursor);
#if 1
	G_UNLOCK (search);
	if (exist_count > 0)
		return exist_count;
	else
		return 0;
#else
	return FALSE;
#endif
}

gboolean
skkconv_rule_init (void)
{
	int i;
	int err;
	DBT key,data;

	/* Create Database */
	if ((err = db_create (&rule_db,NULL,0)) != 0) {
		fprintf (stderr,
				"db_create: %s\n",db_strerror (err));
	}
	rule_db->set_errfile(rule_db,stderr);
	rule_db->set_errpfx(rule_db,"iiimf-skk");
	rule_db->set_flags (rule_db,DB_DUPSORT);
	rule_db->open (rule_db, NULL, 0,DB_BTREE, DB_CREATE, 0644);
	for ( i = 0; i < SKKCONV_BASE_RULE_SIZE ; i++) {
		memset (&key,0,sizeof (DBT));
		memset (&data, 0, sizeof (DBT));
		key.data = (void*)baselist[i].key;
		key.size = strlen (baselist[i].key) + 1;
		data.data = &(baselist[i]);
		data.size = sizeof (baselist[0]);
		rule_db->put (rule_db,NULL,&key,&data,0);
	}
	initialized = TRUE;
	return TRUE;
}

gchar*
skkconv_get_hiragana (const gchar *key, gchar **append)
{
	base_rule_list* rule;
	gchar *ret;
	G_LOCK (conv_kana);
	if (!initialized)
		skkconv_rule_init ();
	rule = do_search (key);
	if (rule) {
		switch (rule->type)
		{
			case SKKCONV_TYPE_NORMAL:
			case SKKCONV_TYPE_USER:
				ret = g_strdup (rule->hira);
				if (!append) return ret;
				if (rule->append) {
					*append  = g_strdup (rule->append);
				} else {
					*append  = NULL;
				}
				G_UNLOCK (conv_kana);
				return ret;
				break;
			default:
				G_UNLOCK (conv_kana);
				return NULL;
				break;
		}
	} else {
		G_UNLOCK (conv_kana);
		return NULL;
	}
}

gchar*
skkconv_get_katakana (const gchar *key, gchar **append)
{
	base_rule_list* rule;
	gchar *ret;
	if (!initialized) 
		skkconv_rule_init ();
	G_LOCK (conv_kata);
	rule = do_search (key);
	if (rule) {
		switch (rule->type)
		{
			case SKKCONV_TYPE_NORMAL:
			case SKKCONV_TYPE_USER:
				ret = g_strdup (rule->kata);
				if (!append) return ret;
				if (rule->append) {
					*append = g_strdup (rule->append);
				} else {
					*append = NULL;
				}
				G_UNLOCK (conv_kata);
				return ret;
				break;
			default:
				G_UNLOCK (conv_kata);
				return NULL;
				break;
		}
	}
	else {
		G_UNLOCK (conv_kata);
		return NULL;
	}
}

gchar*
skkconv_get_func (const gchar *key)
{
	base_rule_list* rule;
	if (!initialized)
		skkconv_rule_init ();
	rule = do_search (key);
	if (rule) {
		switch (rule->type) {
			case SKKCONV_TYPE_FUNC:
			case SKKCONV_TYPE_USER_FUNC:
				if (rule->kata) {
					/* Dirty Hack */
					return g_strdup (rule->kata);
				} else {
					return NULL;
				}
				break;
			default:
				return NULL;
				break;
		}
	} 
	else
		return NULL;
}

SkkConvRuleType
skkconv_get_type (const gchar *key)
{
	base_rule_list *rule;
	if (!initialized)
		skkconv_rule_init ();
	rule = do_search (key);
	if (rule)
		return rule->type;
	else
		return SKKCONV_TYPE_NONE;
}

/**
 * skkconv_hiragana_to_katakana :
 * @hiragana: Hiragana
 *
 * translate hiragana to katakana
 *
 * Return: If success, translated katakana string, otherwise NULL. It's up to caller to free
 **/
gchar*
skkconv_hiragana_to_katakana (const gchar *hiragana)
{
	gchar *copy,*copy2;
	if (!hiragana)
		return NULL;
	copy = copy2 = g_strdup (hiragana);
	for (; copy2 && *copy2 ; copy2 += skk_utils_charbytes (copy2)) {
		if (is_hiragana (copy2)) {
			*copy2 += 0x01;
		}
	}
	return copy;
}

/**
 * skkconv_katakana_to_hiragana :
 * @katakana: Katakana
 *
 * translate katakana to hiragana
 *
 * Return: If success, translated hiragana string, otherwise NULL. It's up to caller to free
 **/
gchar*
skkconv_katakana_to_hiragana (const gchar *katakana)
{
	/* TODO */
	/* katakana has more characters than hiragana */
	gchar *copy,*copy2;
	if (!katakana)
		return NULL;
	copy = copy2 = g_strdup (katakana);
	for (; copy2 && *copy2; copy2 += skk_utils_charbytes (copy2)) {
		if (is_katakana (copy2)) {
			*copy2 -= 0x01;
		}
	}
	return copy;
}

#ifdef SKKCONV_KANA_DEBUG
int
main (void)
{
	gchar *result;
	skkconv_rule_init ();
	printf ("%s\n",skkconv_get_hiragana ("a",NULL));
	printf ("%s\n",skkconv_get_katakana ("te",NULL));
	result = skkconv_get_katakana ("fuga",NULL);
	if (result)
		printf ("%s\n",result);
	else
		printf ("fuga is not found\n");
	printf ("%s\n",skkconv_get_katakana ("bb",NULL));
	printf ("%d\n",skkconv_get_type ("a"));
	printf ("%d\n",skkconv_is_exist ("f"));
	printf ("%d\n",skkconv_is_exist ("n"));
	printf ("%d\n",skkconv_is_exist ("fuga"));
	printf ("%d\n",skkconv_is_exist ("fuga"));
	printf ("%s\n",skkconv_hiragana_to_katakana ("礦1󤭤Ǥ"));
	printf ("%s\n",skkconv_katakana_to_hiragana ("1󥭥ǥ"));
	return 0;
}
#endif
