/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkldict.c,v 1.1.1.1 2002/04/06 02:23:29 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <string.h>
#include <glib.h>
#include <fcntl.h>
#include <db.h>
#include "skkldict.h"
#include "skkutils.h"
#include "skkdebug.h"

static DB* ldict_okuri = NULL;
static DB* ldict = NULL;
static gboolean initialized = FALSE;

G_LOCK_DEFINE_STATIC(ldict);
G_LOCK_DEFINE_STATIC(ldict_okuri);

static void add_word (const gchar *key, const gchar *value);
static void add_word_okuri (const gchar *key, const gchar *okuri, const gchar *value);
static gchar* get_word (const gchar *key);
static gchar* get_word_okuri (const gchar *key);

static gchar*
get_word (const gchar *key)
{
	DBT db_key, db_data;
	int err;
	if (!initialized)
		return NULL;
	if (!key)
		return NULL;
	memset (&db_key, 0, sizeof (DBT));
	memset (&db_data, 0, sizeof (DBT));
	db_key.size = strlen (key) + 1;
	db_key.data = (void *)key;
	if ((err = ldict->get (ldict, NULL, &db_key, &db_data, 0)) != 0 ) {
		if (err != DB_NOTFOUND)
			fprintf (stderr, "iiimf-skk: get_word %s\n",
				db_strerror (err));
		return NULL;
	}
	return (gchar *)db_data.data;
}

static gchar*
get_word_okuri (const gchar *key)
{
	DBT db_key, db_data;
	int err;
	if (!initialized)
		return NULL;
	if (!key)
		return NULL;
	memset (&db_key, 0, sizeof (DBT));
	memset (&db_data, 0, sizeof (DBT));
	db_key.size = strlen (key) + 1;
	db_key.data = (void *)key;
	if ((err = ldict_okuri->get (ldict_okuri, NULL, &db_key, &db_data, 0)) != 0) {
		if (err != DB_NOTFOUND)
			fprintf (stderr, "iiimf-skk: get_word_okuri %s\n",
				db_strerror (err));
			return NULL;
	}
	return (gchar *)db_data.data;
}

static void
add_word (const gchar *key, const gchar *value)
{
	DBT db_key, db_data;
	GList *list;
	GList *result_list = NULL;
	int err;
	gchar *result;
	if (!key || !value)
		return;
	G_LOCK (ldict);
	/* when not initialized, use memory */
	if (!initialized)
		skk_ldict_init (NULL, NULL);
	memset (&db_key, 0, sizeof (DBT));
	memset (&db_data, 0, sizeof (DBT));
	db_key.size = strlen (key) + 1;
	db_key.data =(void*)key;
	result_list = g_list_append (result_list, g_strdup (value));
	list = skk_ldict_query_string (key, NULL, NULL);
	if (list) {
		result_list = skk_utils_list_merge_string (result_list, list);
	}
	result = skk_utils_list_to_string_all (result_list, "/");
	db_data.size = strlen (result) + 1;
	db_data.data = (void *)result;
	if ((err = ldict->put (ldict, NULL, &db_key, &db_data, 0) != 0)) {
		fprintf (stderr, "iiimf-skk: skk_ldict_add_word %s\n",
					db_strerror (err));
	}
	skk_utils_list_free (result_list, TRUE);
	g_free (result);
	G_UNLOCK (ldict);
	return;
}

static void
add_word_okuri (const gchar *key, const gchar *okuri, const gchar *value)
{
	DBT db_key, db_data;
	GList *list;
	GList *result_list = NULL;
	gchar *result;
	gchar *real_key;
	int err;
	if (!key || !value)
		return;
	G_LOCK (ldict_okuri);
	if (!initialized)
		skk_ldict_init (NULL, NULL);
	memset (&db_key, 0, sizeof (DBT));
	memset (&db_data, 0, sizeof (DBT));
	result_list = g_list_append (result_list, g_strdup (value));
	list = skk_ldict_query_string (key, okuri, NULL);
	if (list) {
		result_list = skk_utils_list_merge_string (result_list, list);
	}
	result = skk_utils_list_to_string_all (result_list, "/");
	/* TODO store db */
	real_key = g_strdup_printf ("%s%c", key, okuri[0]);
	db_key.size = strlen (real_key) + 1;
	db_key.data = (void *)real_key;
	db_data.size = strlen (result) + 1;
	db_data.data = (void *)result;
	if ((err = ldict_okuri->put (ldict_okuri, NULL, &db_key, &db_data, 0) != 0)) {
		fprintf (stderr, "iiimf-skk: add_buffer_okuri %s\n",
				db_strerror (err));
	}	
	skk_utils_list_free (result_list, TRUE);
	g_free (real_key);
	g_free (result);
	G_UNLOCK (ldict_okuri);
	return;
}

gboolean
skk_ldict_init (const gchar *okurinasi_filename, const gchar *okuriari_filename)
{
	/* FIXME */
	int err;
	gchar *home_dir;
	gchar *autosave_file;
	home_dir = g_get_home_dir ();
	autosave_file = g_strconcat (home_dir,"/.iiimf-skk-autosave",NULL);
	/* Create Local Dict Database */
	if ((err = db_create (&ldict,NULL,0) != 0)) {
	}
	ldict->set_errfile(ldict,stderr);
	ldict->set_errpfx(ldict,"iiimf-skk");
	ldict->open (ldict, okurinasi_filename, 0, DB_HASH, DB_CREATE, 0644);

	if ((err = db_create (&ldict_okuri, NULL, 0) != 0)) {
	}
	ldict_okuri->set_errfile (ldict_okuri, stderr);
	ldict_okuri->set_errpfx (ldict_okuri, "iiimf-skk");
	ldict_okuri->open (ldict_okuri, okuriari_filename, 0, DB_HASH, DB_CREATE, 0644);

	initialized = TRUE;
	DEBUG_DO (printf ("homedir %s\n",home_dir));
	g_free (autosave_file);
	return TRUE;
}

void
skk_ldict_add_word (const gchar *key, const gchar *okuri, const gchar *value)
{
	if (!key)
		return;
	if (!value)
		return;
	if (!okuri)
		add_word (key, value);
	else
		add_word_okuri (key, okuri, value);
	return;
}

gchar*
skk_ldict_get_word (const gchar *key, SkkAddDictInfo info)
{
	switch (info) {
		case SKKDICT_OKURIARI:
			return get_word_okuri (key);
			break;
		case SKKDICT_OKURINASI:
			return get_word (key);
			break;
		default:
			return NULL;
	}
	return NULL;
}

GList*
skk_ldict_get_completion (const gchar *key)
{
	DBT db_key,db_data;
	DBC *db_cursor;
	int err;
	int key_len;
	GList *ret = NULL;
	if (!initialized)
		return NULL;
	if (!key)
		return NULL;
	key_len = strlen (key);
	memset (&db_key,0,sizeof (DBT));
	memset (&db_data,0,sizeof (DBT));
	ldict->cursor (ldict,NULL,&db_cursor,0);
	if ((err = db_cursor->c_get (db_cursor,&db_key,&db_data,DB_LAST) != 0)) {
		if (err != DB_NOTFOUND) {
			fprintf (stderr,"iiimf-skk: skk_ldict_get_completion %s\n",db_strerror (err));
		}
		db_cursor->c_close (db_cursor);
		return NULL;
	}
	if (!strncmp (key,(const char*)db_key.data, strlen (key))) {
		ret = g_list_append (ret, g_strdup (db_key.data));
	}
	while (TRUE) {
		if ((err = db_cursor->c_get (db_cursor,&db_key,&db_data,DB_PREV) != 0)) {
			if (err != DB_NOTFOUND)
				fprintf (stderr,"iiimf-skk: skk_ldict_get_completion %s\n",db_strerror (err));
			break;
		}
		if (!strncmp (key,(const char*)db_key.data, strlen (key))) {
			ret = g_list_append (ret, g_strdup (db_key.data));
		}
	}
	db_cursor->c_close (db_cursor);
	return ret;
}

GList *
skk_ldict_query_string (const gchar *buf, const gchar *okuri, gint *found_num)
{
	GList *ret = NULL;
	gchar *query_str;
	gchar *query_result;
	gchar **list;
	gchar *list_count;
	gint index;
	gint found;
	if (!buf)
		return NULL;
	if (!initialized)
		skk_ldict_init (NULL, NULL);
	if (okuri) {
		query_str = g_strdup_printf ("%s%c", buf, okuri[0]);
		query_result = get_word_okuri (query_str);
	} else {
		query_str = g_strdup (buf);
		query_result = get_word (query_str);
	}
	if (query_result) {
		list = g_strsplit (query_result, "/", strlen (query_result));
	} else {
		g_free (query_str);
		if (found_num)
			*found_num = 0;
		return NULL;
	}
	for (index = 0, list_count = list[0], found = 0;
			list_count;
			index++, list_count = list[index]) {
		if (list_count && *list_count == '\0') {
			g_free (list_count);
			continue;
		}
		ret = g_list_append (ret, list_count);
		found++;
	}
	if (found_num)
		*found_num = found;
	g_free (query_str);
	return ret;
}

#ifdef SKKLDICT_DEBUG
int
main (void)
{
	skk_ldict_init (NULL, NULL);
	skk_ldict_add_word ("", NULL, "դۤ");
	skk_ldict_get_word ("",SKKDICT_OKURINASI);
	skk_ldict_add_word ("", NULL, "դդ") ;
	skk_ldict_add_word ("", NULL, "դդ" );
	skk_ldict_add_word ("", NULL, "դۤ");
	skk_ldict_add_word ("", NULL, "դۤ");
	skk_ldict_add_word ("",NULL, "դۤ");
	printf ("%s\n",skk_ldict_get_word ("", SKKDICT_OKURINASI));
	return 0;
}
#endif
