/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 *
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkquery.c,v 1.1.1.1 2002/04/06 02:23:29 famao Exp $ */

/* vi:set ts=4 sw=4: */

#include <string.h>
#include "skkquery.h"
#include "skkutils.h"
#include "skkldict.h"
#include "skkdict.h"

SkkQuery *
skk_query_new (void)
{
	SkkQuery *ret;
	ret = g_new (SkkQuery, 1);
	memset (ret, 0, sizeof (SkkQuery));
	ret->query_list = NULL;
	ret->ref_count++;
	ret->cur_item = NULL;
	return ret;
}

SkkQueryItem *
skk_query_item_new (gchar *filename, SkkQueryType type)
{
	SkkQueryItem *item;
	item = g_new (SkkQueryItem, 1);
	memset (item, 0, sizeof (SkkQueryItem));
	item->file = filename;
	item->type = type;
	return item;
}

void
skk_query_destroy (SkkQuery *query)
{
	if (!query)
		return;
	query->ref_count--;
	if (query->ref_count > 0)
		return;
	if (query->query_list) {
		/* FIXME maybe memory leaks */
		skk_utils_list_free (query->query_list, TRUE);
		query->query_list = NULL;
	}
	g_free (query);
	return;
}

void
skk_query_add_item (SkkQuery *query, SkkQueryItem *item)
{
	if (!query)
		return;
	if (!item)
		return;
	query->query_list = g_list_append (query->query_list, (gpointer)item);
	return;
}

gboolean
skk_query_set_next (SkkQuery *query)
{
	if (!query)
		return FALSE;
	query->cur_list_num++;
	query->cur_item = g_list_nth_data (query->query_list, query->cur_list_num);
	if (query->cur_item)
		return TRUE;
	return FALSE;
}

gboolean
skk_query_set_nth (SkkQuery *query, gint nth)
{
	SkkQueryItem *tmp_item;
	if (!query)
		return FALSE;
	tmp_item = query->cur_item;	
	query->cur_item = (SkkQueryItem *)g_list_nth_data (query->query_list, nth);
	if (query->cur_item) {
		query->cur_list_num = nth;
		return TRUE;
	}
	query->cur_item = tmp_item;
	return FALSE;
}	

gboolean
skk_query_set_prev (SkkQuery *query)
{
	if (!query)
		return FALSE;
	if (query->cur_list_num == 0)
		return FALSE;
	query->cur_list_num--;
	query->cur_item = g_list_nth_data (query->query_list, query->cur_list_num);
	if (query->cur_item)
		return TRUE;
	return FALSE;
}

GList *
skk_query_do_query (SkkQuery *query, gchar *buf, gchar *okuri, gint *found_num)
{
	if (!query)
		return NULL;
	if (!query->cur_item) {
		query->cur_item = (SkkQueryItem *)g_list_nth_data (query->query_list, 0);
	}
	if (!query->cur_item)
		return NULL;
	switch (query->cur_item->type) {
		case SKKQUERY_LOCAL:
			return skk_ldict_query_string (buf, okuri, found_num);
			break;
		case SKKQUERY_SERVER:
			return skk_dict_query_string (buf, okuri, found_num);
			break;
		default:
			break;
	}
	return NULL;
}
