/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 * gtkimmoduleime
 * Copyright (C) 2003 Takuro Ashie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: gtkimcontextime.c,v 1.4 2003/05/06 10:23:49 makeinu Exp $
 */

#include "gtkimcontextime.h"

#include <windows.h>
#include <imm.h>

#include <gdk/gdkwin32.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkwidget.h>

/*
#warning FIXME!!
#ifdef STRICT
#undef STRICT
*/
#include <pango/pangowin32.h>

static void     gtk_im_context_ime_class_init          (GtkIMContextIMEClass *class);
static void     gtk_im_context_ime_init                (GtkIMContextIME *im_context_ime);
static void     gtk_im_context_ime_finalize            (GObject *obj);

/* virtual functions */
static void     gtk_im_context_ime_set_client_window   (GtkIMContext   *context,
                                                        GdkWindow      *client_window);
static gboolean gtk_im_context_ime_filter_keypress     (GtkIMContext   *context,
                                                        GdkEventKey    *event);
static void     gtk_im_context_ime_focus_in            (GtkIMContext   *context);
static void     gtk_im_context_ime_focus_out           (GtkIMContext   *context);
static void     gtk_im_context_ime_set_cursor_location (GtkIMContext   *context,
                                                        GdkRectangle   *area);
static void     gtk_im_context_ime_set_use_preedit     (GtkIMContext   *context,
                                                        gboolean        use_preedit);

/* private */
static void     gtk_im_context_ime_set_preedit_font    (GtkIMContext   *context);


GType gtk_type_im_context_ime = 0;
static GObjectClass *parent_class;
GtkIMContextIME *current_context = NULL;


void
gtk_im_context_ime_register_type (GTypeModule *type_module)
{
    static const GTypeInfo im_context_ime_info = {
        sizeof (GtkIMContextIMEClass),
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) gtk_im_context_ime_class_init,
        NULL,           /* class_finalize */    
        NULL,           /* class_data */
        sizeof (GtkIMContextIME),
        0,
        (GInstanceInitFunc) gtk_im_context_ime_init,
    };

    gtk_type_im_context_ime = 
        g_type_module_register_type (type_module,
                                     GTK_TYPE_IM_CONTEXT,
                                     "GtkIMContextIME",
                                     &im_context_ime_info, 0);
}

static void
gtk_im_context_ime_class_init (GtkIMContextIMEClass *class)
{
    GtkIMContextClass *im_context_class = GTK_IM_CONTEXT_CLASS (class);
    GObjectClass *gobject_class = G_OBJECT_CLASS (class);

    parent_class = g_type_class_peek_parent (class);

    im_context_class->set_client_window   = gtk_im_context_ime_set_client_window;
    im_context_class->filter_keypress     = gtk_im_context_ime_filter_keypress;
    /*
    im_context_class->reset               = NULL;
    im_context_class->get_preedit_string  = gtk_im_context_ime_get_preedit_string;
    */
    im_context_class->focus_in            = gtk_im_context_ime_focus_in;
    im_context_class->focus_out           = gtk_im_context_ime_focus_out;
    im_context_class->set_cursor_location = gtk_im_context_ime_set_cursor_location;
    im_context_class->set_use_preedit     = gtk_im_context_ime_set_use_preedit;
    gobject_class->finalize               = gtk_im_context_ime_finalize;
}


static void
gtk_im_context_ime_init (GtkIMContextIME *context_ime)
{
    GdkRectangle *area;

    context_ime->client_window = NULL;
    context_ime->preediting = FALSE;
    area = &context_ime->cursor_location;
    area->x = area->y = area->width = area->height = 0;
}


static void
gtk_im_context_ime_finalize (GObject *obj)
{
    GtkIMContext *context = GTK_IM_CONTEXT (obj);
    GtkIMContextIME *context_ime = GTK_IM_CONTEXT_IME (obj);

    gtk_im_context_ime_set_client_window (context, NULL);
}


GtkIMContext *
gtk_im_context_ime_new (void)
{
  return g_object_new (GTK_TYPE_IM_CONTEXT_IME, NULL);
}


GdkFilterReturn 
ime_message_filter(GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
    GtkIMContext *context;
    GtkIMContextIME *context_ime;
    HWND hwnd;
    HIMC himc;
    MSG *msg = (MSG *) xevent;

    g_return_val_if_fail(GTK_IS_IM_CONTEXT_IME (data), GDK_FILTER_CONTINUE);

    context     = GTK_IM_CONTEXT (data);
    context_ime = GTK_IM_CONTEXT_IME (data);
    if (current_context != context_ime)
       return GDK_FILTER_CONTINUE;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    switch(msg->message) {
    case WM_IME_COMPOSITION:
        if (msg->lParam & GCS_RESULTSTR) {
            gchar *utf8str, buf[4096];
            glong len, pos;
            gsize bytes_read, bytes_written;
            GError *error = NULL;

            if (!himc) break;
            if (!ImmGetOpenStatus(himc)) break;

            len = ImmGetCompositionString (himc, GCS_RESULTSTR,
                                           buf, G_N_ELEMENTS (buf) - 1);
            pos = len < G_N_ELEMENTS (buf) ? len : G_N_ELEMENTS (buf) - 1;
            buf[pos] = '\0';

            utf8str = g_locale_to_utf8(buf, -1, &bytes_read, &bytes_written,
                                       &error);
            if (error) {
                g_warning("%s", error->message);
                g_error_free (error);
            }
            if (utf8str) {
                g_signal_emit_by_name(G_OBJECT(context_ime), "commit", utf8str);
                g_free (utf8str);
            }
        }
        break;

    case WM_IME_STARTCOMPOSITION:
        context_ime->preediting = TRUE;
        gtk_im_context_ime_set_cursor_location (context, NULL);
        g_signal_emit_by_name (context, "preedit_start");
        break;

    case WM_IME_ENDCOMPOSITION:
        context_ime->preediting = FALSE;
        g_signal_emit_by_name (context, "preedit_end");
        break;

    case WM_IME_NOTIFY:
        switch (msg->wParam) {
        case IMN_SETOPENSTATUS:
        {
            if (ImmGetOpenStatus (himc))
                gtk_im_context_ime_set_preedit_font(context);
            break;
        }
        break;

        default:
            break;
        }
    default:
        break;
    }

    ImmReleaseContext (hwnd, himc);

    return GDK_FILTER_CONTINUE;
}


static void
gtk_im_context_ime_set_client_window (GtkIMContext *context,
                                      GdkWindow *client_window)
{
    GtkIMContextIME *context_ime;
    GdkWindow *toplevel;
    HWND hwnd = NULL;
    HIMC himc;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));
    context_ime = GTK_IM_CONTEXT_IME(context);

    if (client_window) {
        hwnd = GDK_WINDOW_HWND (client_window);
        toplevel = gdk_window_get_toplevel(client_window);
        gdk_window_add_filter(toplevel, ime_message_filter, context_ime);
    } else if (context_ime->client_window) {
        hwnd = GDK_WINDOW_HWND (context_ime->client_window);
        toplevel = gdk_window_get_toplevel(context_ime->client_window);
        gdk_window_remove_filter(toplevel, ime_message_filter, context_ime);
    }

    context_ime->client_window = client_window;
}


static gboolean
gtk_im_context_ime_filter_keypress (GtkIMContext *context,
                                    GdkEventKey  *event)
{
    GtkIMContextIME *context_ime;
    HWND hwnd;
    HIMC himc;

    g_return_val_if_fail(GTK_IS_IM_CONTEXT_IME(context), FALSE);
    g_return_val_if_fail(event, FALSE);

    context_ime = GTK_IM_CONTEXT_IME(context);
    if (current_context != context_ime) return FALSE;
    if (!GDK_IS_WINDOW(context_ime->client_window)) return FALSE;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (himc && ImmGetOpenStatus(himc)) return FALSE;

    if (event->string && *event->string) {
        g_signal_emit_by_name(G_OBJECT(context_ime), "commit", event->string);
        return TRUE;
    }

    return FALSE;
}


static void
gtk_im_context_ime_focus_in (GtkIMContext   *context)
{
    current_context = GTK_IM_CONTEXT_IME (context);

    /* FIXME!! restore composition context */
}


static void
gtk_im_context_ime_focus_out (GtkIMContext   *context)
{
    if (current_context == GTK_IM_CONTEXT_IME (context))
        current_context = NULL;

    /* FIXME!! save compositioin context */
}


/*
 * x and y must be initialized to 0.
 */
static void
get_window_position (GdkWindow *win, gint *x, gint *y)
{
    GdkWindow *parent, *toplevel;
    gint wx, wy;

    g_return_if_fail (GDK_IS_WINDOW (win));
    g_return_if_fail (x && y);

    gdk_window_get_position(win, &wx, &wy);
    *x += wx;
    *y += wy;
    parent = gdk_window_get_parent (win);
    toplevel = gdk_window_get_toplevel (win);

    if (parent && parent != toplevel)
        get_window_position (parent, x, y);
}


static void
gtk_im_context_ime_set_cursor_location (GtkIMContext *context,
                                        GdkRectangle *area)
{
    GtkIMContextIME *context_ime;
    POINT pt;
    COMPOSITIONFORM cf;
    GtkWidget *widget;
    HWND hwnd;
    HIMC himc;
    gint wx = 0, wy = 0;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));

    context_ime = GTK_IM_CONTEXT_IME(context);
    if (!area)
        area = &context_ime->cursor_location;
    else
        context_ime->cursor_location = *area;

    if (!context_ime->client_window) return;

    gdk_window_get_user_data (context_ime->client_window,
                              (gpointer *) &widget);
    if (!GTK_IS_WIDGET (widget)) return;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;
    if (!ImmGetOpenStatus(himc)) return;

    get_window_position(context_ime->client_window, &wx, &wy);
    pt.x = wx + area->x;
    pt.y = wy + area->y;
    cf.dwStyle = CFS_POINT;
    cf.ptCurrentPos = pt;
    ImmSetCompositionWindow(himc, &cf);

    ImmReleaseContext (hwnd, himc);

    return;
}


static void
gtk_im_context_ime_set_use_preedit (GtkIMContext *context,
                                    gboolean      use_preedit)
{
    GtkIMContextIME *context_ime;
    HWND hwnd;
    HIMC himc;
    BOOL success;

    g_return_if_fail(GTK_IS_IM_CONTEXT_IME (context));
    context_ime = GTK_IM_CONTEXT_IME (context);

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    success = ImmSetOpenStatus (himc, use_preedit);
    if (!success) {
        const gchar *status_str = use_preedit ? "open" : "close";
        g_warning ("Faild to %s IME!", status_str);
    }
    ImmReleaseContext (hwnd, himc);
}


static void
gtk_im_context_ime_set_preedit_font (GtkIMContext *context)
{
    GtkIMContextIME *context_ime;
    GtkWidget *widget;
    HWND hwnd;
    HIMC himc;
    PangoContext *pango_context;
    PangoFont *pango_font;
    LOGFONT *logfont;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));

    context_ime = GTK_IM_CONTEXT_IME(context);
    if (!context_ime->client_window) return;

    gdk_window_get_user_data (context_ime->client_window,
                              (gpointer *) &widget);
    if (!GTK_IS_WIDGET (widget)) return;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    /* set font */
    pango_context = gtk_widget_get_pango_context (widget);
    if (!pango_context) return;
    pango_font = pango_context_load_font (pango_context,
                                          widget->style->font_desc);
    logfont = pango_win32_font_logfont (pango_font);
    ImmSetCompositionFont (himc, logfont);

    /* clean */
    ImmReleaseContext (hwnd, himc);
}
