/******************************************************************************/
/*! @file apolloron.h
    @brief Header file of libapolloron.
    @author Tachibanamasashi, Apolloron Project.
    @date 2009-09-21
 ******************************************************************************/

#ifndef _APOLLORON_H_
#define _APOLLORON_H_

namespace apolloron {

class String;
class List;
class Keys;

/*! @brief Version of libapolloron.
 */
#define LIBAPOLLORON_VERSION "0.43.9"


/*----------------------------------------------------------------------------*/
/* String class                                                               */
/*----------------------------------------------------------------------------*/
/*! @brief Class of variable length character sequence.
 */ 
class String {
protected:
  // nBinaryLength <= nFixedLength <= nCapacity
  char *pText; // String
  long nLength; // Size of string
  long nCapacity; // Size of all allocated memory
  long nFixedLength; // Memory size which carries out minimum allocation
  long nBinaryLength; // Size in the case of treating as binary data
  bool nLengthRenewRecommended; // this is true if the length of string is uncertain
  String ***pTmp; // Temporary String
  long *pTmpLen; // Number of array of temporary String
  virtual String *tmpStr() const; // Reference of temporary String
  virtual bool clearTmpStr() const; // Deletion of all temporary String
  virtual bool pTextReplace(char *str, long text_len, long binary_len, long capacity); // Replacement pText
  virtual bool resize(long); // Change the size of allocated memory
  virtual bool nLengthRenew(); // renew nLength
  virtual bool nLengthNeedRenew(); // It calls, when length may become unfixed
  virtual bool mresize(long = 0); // Change size of minimum memory allocation
public:
  String();
  String(const String &value);
  String(const char *value);
  String(long value);
  String(int value);
  String(char value);
  String(double value);
  virtual ~String();

  // Deletion of String instance
  virtual bool clear();

  // Deletion of all temporary String
  virtual bool gc() const;

  // Reference of length of string
  virtual long len() const;

  // Reference of width of string
  virtual long width(const char * src_charset = "UTF-8") const;

  // Memory size which carries out minimum allocation
  virtual long fixedLength() const;
  virtual bool setFixedLength(long size);

  // Reference of size as binary String
  virtual long binaryLength() const;

  // Change to binary mode (or change binary size)
  virtual bool useAsBinary(long size);

  // Change to text mode
  virtual bool useAsText();

  // Reference of current mode
  virtual bool isText() const;
  virtual bool isBinary() const;

  // Cast
  virtual const char* c_str() const;
  virtual long toLong() const;
  virtual int toInt() const;
  virtual char toChar() const;
  virtual double toDouble() const;

  virtual operator const char*() const;
  virtual operator long() const;
  virtual operator int() const;
  virtual operator const char() const;
  virtual operator double() const;

  // Reference of array
  virtual char& operator [] (long index);
  virtual char& operator [] (int index);

  // Setup
  virtual bool set(const String &value);
  virtual bool set(const char *value, long length=-1L);
  virtual bool set(long value);
  virtual bool set(int value);
  virtual bool set(char value);
  virtual bool set(double value);
  virtual bool setBinary(char value);
  virtual bool setBinary(const char *value, long length);

  virtual const String& operator = (const String &value);
  virtual const String& operator = (const char *value);
  virtual const String& operator = (long value);
  virtual const String& operator = (int value);
  virtual const String& operator = (char value);
  virtual const String& operator = (double value);

  // Comparison
  virtual int compare(const String &value) const;
  virtual int compare(const char *value) const;
  virtual int compare(long value) const;
  virtual int compare(int value) const;
  virtual int compare(char value) const;
  virtual int compare(double value) const;

  virtual int operator == (const String &value) const;
  virtual int operator == (const char *value) const;
  virtual int operator == (long value) const;
  virtual int operator == (int value) const;
  virtual int operator == (char value) const;
  virtual int operator == (double value) const;

  virtual int operator != (const String &value) const;
  virtual int operator != (const char *value) const;
  virtual int operator != (long value) const;
  virtual int operator != (int value) const;
  virtual int operator != (char value) const;
  virtual int operator != (double value) const;

  virtual int operator <= (const String &value) const;
  virtual int operator <= (const char *value) const;
  virtual int operator <= (long value) const;
  virtual int operator <= (int value) const;
  virtual int operator <= (char value) const;
  virtual int operator <= (double value) const;

  virtual int operator >= (const String &value) const;
  virtual int operator >= (const char *value) const;
  virtual int operator >= (long value) const;
  virtual int operator >= (int value) const;
  virtual int operator >= (char value) const;
  virtual int operator >= (double value) const;

  virtual int operator < (const String &value) const;
  virtual int operator < (const char *value) const;
  virtual int operator < (long value) const;
  virtual int operator < (int value) const;
  virtual int operator < (char value) const;
  virtual int operator < (double value) const;

  virtual int operator > (const String & value) const;
  virtual int operator > (const char *value) const;
  virtual int operator > (long value) const;
  virtual int operator > (int value) const;
  virtual int operator > (char value) const;
  virtual int operator > (double value) const;

  // Addition
  virtual bool add(const String &value);
  virtual bool add(const char *value);
  virtual bool add(long value);
  virtual bool add(int value);
  virtual bool add(char value);
  virtual bool add(double value);
  virtual bool addBinary(char value);
  virtual bool addBinary(const char *value, long index);

  virtual const String& operator += (const String &value);
  virtual const String& operator += (const char *value);
  virtual const String& operator += (long value);
  virtual const String& operator += (int value);
  virtual const String& operator += (char value);
  virtual const String& operator += (double value);

  virtual String& operator + (const String &value) const;
  virtual String& operator + (const char *value) const;
  virtual String& operator + (long value) const;
  virtual String& operator + (int value) const;
  virtual String& operator + (char value) const;
  virtual String& operator + (double value) const;

  // Insertion
  virtual String& insert(long pos, const String &value) const;
  virtual String& insert(long pos, const char *value) const;
  virtual String& insert(long pos, long value) const;
  virtual String& insert(long pos, int value) const;
  virtual String& insert(long pos, char value) const;
  virtual String& insert(long pos, double value) const;

  // Trim
  virtual String& left(long length) const;
  virtual String& right(long length) const;
  virtual String& mid(long pos, long length = -1L) const;
  virtual String& trimR() const;
  virtual String& trimL() const;
  virtual String& trim() const;

  // Upper / lower
  virtual String& upper() const;
  virtual String& lower() const;

  // Search
  virtual long search(const String &key, long pos = 0L) const;
  virtual long searchCase(const String &key, long pos = 0L) const;
  virtual long searchChar(const char c, long pos = 0L) const;
  virtual String& replace(const String &key, const String &replace_str, long pos = 0L) const;
  virtual String& replaceCase(const String &key, const String &replace_str, long pos = 0L) const;

  // Format
  virtual int sprintf(const char *format, ...);

  // URL encoding
  virtual String& encodeURL() const;
  virtual String& decodeURL(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

  // Escape HTML tag
  virtual String& escapeHTML(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", const char *options="") const;
  virtual String& unescapeHTML(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

  // Escape Back-slash, Double-quote, Single-quote
  virtual String& escapeQuote(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;
  virtual String& unescapeQuote(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

  // HEX encoding
  virtual String& encodeHEX() const;
  virtual String& decodeHEX() const;

  // UUencoding
  virtual String& decodeUU(String * filename = (String *)0) const;

  // MIME encoding
  virtual String& encodeMIME(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", long max_width = 72, const char * return_str = "\n", const char enctype = 'B') const;
  virtual String& decodeMIME(const char * src_charset = "AUTODETECT", const char * dest_charset = "UTF-8") const;
  virtual String& encodeBASE64(long max_width = 72, const char * return_str = "\n", const char * prefix_str = (const char *)0, const char * suffix_str = (const char *)0) const;
  virtual String& decodeBASE64() const;
  virtual String& encodeQuotedPrintable(long max_width = 72, const char * return_str = "\n", const char * prefix_str = (const char *)0, const char * suffix_str = (const char *)0) const;
  virtual String& decodeQuotedPrintable() const;

  // Crypt / Decrypt
  virtual String& crypt() const;
  virtual String& decrypt() const;

  // MD5 digest key
  virtual String& md5() const;
  virtual String& hmacMd5(const String & key) const;
  virtual String& cramMd5(const String & user, const String & challenge) const;

  // Plain text or HTML template convertion
  virtual String& evalText(Keys replace_keys) const;

  // HTML to plain text
  virtual String& convertHTMLToPlain(const char *src_charset = "UTF-8", const char *dest_charset = "UTF-8") const;

  // Character code conversion
  virtual String& strconv(const char *src_charset, const char *dest_charset) const;

  // Character code detection
  virtual const char* detectCharSet() const;
  virtual const char* detectCharSetJP() const;

  // return code (CR/LF/CRLF)
  virtual String& changeReturnCode(const char * return_str = "\n") const;

  // Character width converting
  virtual String& changeWidth(const char *option, const char *src_charset = "UTF-8", const char *dest_charset = "UTF-8") const;

  // Regular expression
  virtual String& reMatch(const char *regex) const;
  virtual String& reSubst(const char *regex1, const char *regex2) const;
  virtual String& reTrans(const char *regex1, const char *regex2) const;

  // Wild Card
  virtual String& wcMatch(const char *wildcard) const;
  virtual bool isWcMatchAll(const char *wildcard) const;

  // Multi-line string processing
  virtual String& uniq() const;

  // Files
  virtual long loadFile(const char *filename);
  virtual long loadFilePos(const char *filename, long start_pos, long size);
  virtual bool saveFile(const char *filename) const;
  virtual bool saveFilePos(const char *filename, long start_pos, long size) const;

  // MIDI data conversion
  virtual String& toMIDI() const;
};


/*----------------------------------------------------------------------------*/
/* List class                                                                 */
/*----------------------------------------------------------------------------*/
/*! @brief Class of array.
 */ 
class List {
protected:
  String **pList; // content of array
  int pListSize; // Size of array
  virtual bool addString(const String &); // Addition of string
  virtual bool delString(long index); // Deletion of string
  virtual bool insertString(long index, const String & value); // Insertion of string
public:
  List();
  List(const List & list);
  virtual ~List();

  // Deletion of object instance
  virtual bool clear();

  // Reference of element of array
  virtual String& operator [] (long index);
  virtual String& operator [] (int index);

  // Number of elements of array
  virtual long max() const;

  // Setup of array
  virtual bool set(const List & list);
  virtual const List& operator = (const List &list);

  // Addition of element of array
  virtual const List& operator += (const String &value);
  virtual bool add(const String &value);

  // Insertion of element of array
  virtual bool insert(long index, const String &value);

  // Deletion of array
  virtual bool remove(long index);
};


// Binary tree element structure
typedef struct {
  char *key;
  String *value;
  long small;
  long big;
} Element;

/*----------------------------------------------------------------------------*/
/* Keys class                                                                 */
/*----------------------------------------------------------------------------*/
/*! @brief Class of associative array.
 */
class Keys {
protected:
  long nElementStart[64]; // start point (BASE64 key)
  long nElementTotal[64];
  long nElementCache[64];
  Element ***pElement; // (pointers of array) x 64
  virtual String& getValue(const char *key);
public:
  Keys();
  Keys(const Keys & keys);
  virtual ~Keys();

  // Deletion of object instance
  virtual bool clear();

  // Reference of value of element of key
  virtual String& operator [] (const String &key);
  virtual String& operator [] (const char *key);

  // Key's existence check
  virtual bool isKey(const String &key) const;
  virtual bool isKey(const char *key) const;

  // Get number of key
  virtual long max() const;

  // Setup of keys
  virtual bool set(const Keys & keys);
  virtual const Keys& operator = (const Keys &keys);

  // Looking up key
  virtual const char *key(long index) const;
  virtual String& value(long index) const;
  virtual Element* array(long index) const;

  // Addition of key
  virtual bool addKey(const char *key, const String &value);
  virtual bool addKey(const char *key, const char *value = "");

  // Deletion of key
  virtual bool delKey(const String &key);
  virtual bool delKey(const char *key);

  // Files
  virtual bool loadFile(const String &filename);
  virtual bool loadFile(const char *filename);
  virtual bool saveFile(const String &filename) const;
  virtual bool saveFile(const char *filename) const;
};


#include <time.h>
#include <sys/time.h>
typedef struct {
  int mday;    // Day.     [1-31]
  int mon;     // Month.   [1-12]
  int year;    // Year.    [1970-2037]
  int wday;    // Day of week. [0-6]
  int yday;    // Days in year.[0-365]
} TDate;

typedef struct {
  int sec;     // Seconds. [0-61] (2 leap second) */
  int min;     // Minutes. [0-59] */
  int hour;    // Hours.   [0-23] */
} TTime;

typedef struct {
  TDate date;  // Date
  TTime time;  // Time
  long gmtoff; // Registrant timezone (GMT offset)
} TDateTime;

/*----------------------------------------------------------------------------*/
/* DateTime class                                                             */
/*----------------------------------------------------------------------------*/
/*! @brief Class of date and time.
 */
class DateTime {
protected:
  TDateTime tmpDateTime; // Date and time
  String *tmpString; // Date and time string for getString()
public:
  DateTime();
  DateTime(DateTime &datetime);
  DateTime(TDateTime &datetime);
  DateTime(time_t time_t_time, long gmtoff = 0L);
  DateTime(String &datetimestr);
  virtual ~DateTime();

  // Deletion of object instance
  virtual bool clear();

  // Setup of date and time
  virtual DateTime &operator = (const DateTime &datetime);
  virtual bool setNow();
  virtual bool set(const DateTime &datetime);
  virtual bool set(const String &datetimestr);
  virtual bool set(const TDateTime &datetime);
  virtual bool set(const time_t time_t_time, long gmtoff = 0L);
  virtual bool setDate(const TDate &date);
  virtual bool setTime(const TTime &time);
  virtual bool setGMTOffset(long gmtoff);

  // Getting date and time
  virtual String &toString(const String &format);
#define DATEFORMAT_RFC822  "%a, %02d %b %04Y %02H:%02M:%02S %z"
#define DATEFORMAT_INTERNALDATE "%02d-%b-%04Y %02H:%02M:%02S %z"
#define DATEFORMAT_ISO8601 "%04Y-%02m-%02dT%02H:%02M:%02S%Z"
  virtual const TDateTime &getDateTime() const;
  virtual const TDate &getDate() const;
  virtual const TTime &getTime() const;
  virtual long getGmtOff() const;

  // Calculating date and time
  virtual bool adjustYear(int year);
  virtual bool adjustMonth(int month);
  virtual bool adjustDate(int day);
  virtual bool adjustHour(int hour);
  virtual bool adjustSeconds(int sec);
  virtual bool changeGMTOffset(long gmtoff);

  // Comparison
  virtual int compare(const DateTime &value) const;

  virtual int operator == (const DateTime &value) const;
  virtual int operator != (const DateTime &value) const;
  virtual int operator <= (const DateTime &value) const;
  virtual int operator >= (const DateTime &value) const;
  virtual int operator < (const DateTime &value) const;
  virtual int operator > (const DateTime &value) const;
};


/*----------------------------------------------------------------------------*/
/* Socket class                                                               */
/*----------------------------------------------------------------------------*/
/*! @brief Class of basic networking of TCP socket.
 */
class Socket {
protected:
  int dstSocket; // Destination socket number
  int nTimeout; // Timeout value (second)
  int nErrno; // Error number
  bool bConnected; // false: Disconnected  true: Connected
  String *tmpString; // Cash for receiving data
  bool bSSLStream; // false: non SSL  true: SSL
  void *pSSL; // SSL*
  void *pSSLCTX; // SSL_CTX*
public:
  Socket(int recycle_dstSocket = -1);
  virtual ~Socket();

  // Deletion of object instance
  virtual bool clear();

  // Connection
  virtual bool connect(const char *host, const char *port, bool ssl = false);

  // Disconnection
  virtual bool disconnect();

  // Timeout setup
  virtual bool setTimeout(int sec);

  // Refer to error number
  virtual int error() const;

  // Connection status check
  virtual bool connected() const;

  // STARTTLS
  virtual bool startTls();

  // Data transmission
  virtual bool send(const String &str);
  virtual bool send(const char *str);

  // Data reception
  virtual String &receive(long size);
  virtual String &receiveLine();
  virtual String &receivedData() const;
};


/*----------------------------------------------------------------------------*/
/* CGI class                                                                  */
/*----------------------------------------------------------------------------*/
/*! @brief CGI Class.
 */
class CGI {
protected:
  Keys requestData; // CGI request values
  Keys requestFileName; // CGI request file names
  Keys cookie; // CGI request file names
public:
  CGI(int argc, const char *argv[]);
  virtual ~CGI();

  // Deletion of object instance
  virtual bool clear();

  virtual String& getValue(const String &key);
  virtual String& getValue(const char *key);
  virtual String& getFileName(const String &key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
  virtual String& getFileName(const char *key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
  virtual String& getCookie(const char *key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
};


/*----------------------------------------------------------------------------*/
/* Utility functions                                                          */
/*----------------------------------------------------------------------------*/
/*! @brief Utility functions.
 */ 
bool isFileExist(const String &filename);
bool isDirExist(const String &dirname);
bool isEmailAddress(const String &email);
bool isEmailLocalPart(const String &local_part);
bool isEmailDomain(const String &domain);


/*----------------------------------------------------------------------------*/
/* ANSI compatible functions                                                  */
/*----------------------------------------------------------------------------*/
/*! @brief ANSI clone functions.
 */ 
#define strcasestr(x,y) apolloron::apl_strcasestr((x),(y))
char *apl_strcasestr(const char *src, const char *needle);
#define strcasecmp(x,y) apolloron::apl_strcasecmp((x),(y))
int apl_strcasecmp(const char *str1, const char *str2);
#define strncasecmp(x,y,z) apolloron::apl_strncasecmp((x),(y),(z))
int apl_strncasecmp(const char *str1, const char *str2, long n);
int itoa(char *buf, int value, int s=1, int base=10);
int ltoa(char *buf, long value, int s=1, int base=10);
int dtoa(char *buf, double value);

#if defined(__GNUC__)
#ifdef strlen
#undef strlen
#endif
#define strlen(x) ({ \
    const char *_strlen_str; \
    int _strlen_len; \
    _strlen_str = (const char *)(x); \
    _strlen_len = 0; \
    while (_strlen_str[_strlen_len] != '\0') _strlen_len++; \
    (_strlen_len); \
})
#endif

/* ctype.h macros */
#ifndef _CTYPE_H  /* for Solaris */
#define _CTYPE_H
#endif
#ifndef _CTYPE_H_ /* for MacOSX */
#define _CTYPE_H_
#endif
#ifndef __H_CTYPE /* for AIX */
#define __H_CTYPE
#endif
#ifndef _CTYPE_INCLUDED /* for HP-UX */
#define _CTYPE_INCLUDED
#endif

#ifdef isalnum
#undef isalnum
#endif
#define isalnum(c)   (isalpha(c) || isdigit(c))

#ifdef isalpha
#undef isalpha
#endif
#define isalpha(c)   (isupper(c) || islower(c))

#ifdef isascii
#undef isascii
#endif
#define isascii(c)   (0 < ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)0x7F)

#ifdef iscntrl
#undef iscntrl
#endif
#define iscntrl(c)   (((unsigned char)(c)) <= (unsigned char)0x1F || ((unsigned char)(c)) == (unsigned char)0x7F)

#ifdef isdigit
#undef isdigit
#endif
#define isdigit(c)   ((unsigned char)'0' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'9')

#ifdef isgraph
#undef isgraph
#endif
#define isgraph(c)   (((unsigned char)(c)) != (unsigned char)' ' && isprint(c))

#ifdef islower
#undef islower
#endif
#define islower(c)   ((unsigned char)'a' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'z')

#ifdef isprint
#undef isprint
#endif
#define isprint(c)   ((unsigned char)' ' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'~')

#ifdef ispunct
#undef ispunct
#endif
#define ispunct(c)   (((unsigned char)' ' < ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'~') && !isalnum(c))

#ifdef isspace
#undef isspace
#endif
#define isspace(c)   (((unsigned char)(c)) == (unsigned char)' ' || ((unsigned char)(c)) == (unsigned char)'\f' || ((unsigned char)(c)) == (unsigned char)'\n' || ((unsigned char)(c)) == (unsigned char)'\r' || ((unsigned char)(c)) == (unsigned char)'\t' || ((unsigned char)(c)) == (unsigned char)'\v')

#ifdef isblank
#undef isblank
#endif
#define isblank(c)   ((unsigned char)(c)) == (unsigned char)' ')

#ifdef isupper
#undef isupper
#endif
#define isupper(c)   ((unsigned char)'A' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'Z')

#ifdef isxdigit
#undef isxdigit
#endif
#define isxdigit(c)  (isxupper(c) || isxlower(c))

#ifdef isxlower
#undef isxlower
#endif
#define isxlower(c)  (isdigit(c) || ((unsigned char)'a' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'f'))

#ifdef isxupper
#undef isxupper
#endif
#define isxupper(c)  (isdigit(c) || ((unsigned char)'A' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'F'))

#ifdef tolower
#undef tolower
#endif
#define tolower(c)   (isupper(c) ? (((unsigned char)(c)) - (unsigned char)'A' + (unsigned char)'a') : (c))

#ifdef toupper
#undef toupper
#endif
#define toupper(c)   (islower(c) ? (((unsigned char)(c)) - (unsigned char)'a' + (unsigned char)'A') : (c))


#include <sys/file.h>
#ifdef _AIX 
#undef LOCK_SH
#undef LOCK_EX
#undef LOCK_NB
#undef LOCK_UN
#endif
#if !defined(LOCK_SH) || defined(_AIX)
#define LOCK_SH F_LOCK   /* shared lock (for reading) */
#define LOCK_EX F_LOCK   /* exclusive lock (for writing) */
#define LOCK_NB 0        /* don't block when locking */
#define LOCK_UN F_ULOCK  /* unlock */
#define flock(fd, operation) lockf((int)(fd), (int)(operation), 0)
#endif

} // namespace apolloron

#endif
