﻿using System;
using System.Collections.Generic;
using System.Text;
using IrrlichtNetSwig;

// Irrlicht MLの場合は以下のコメントアウトをはずす
// #define MULTI_LANGUAGE

namespace UserInterface
{
    class Program
    {
        static void Main(string[] args)
        {
            // ask user for driver

            E_DRIVER_TYPE driverType;

            System.Console.WriteLine(
@"Please select the driver you want for this example:
 (a) Direct3D 9.0c
 (b) Direct3D 8.1
 (c) OpenGL 1.5
 (d) Software Renderer
 (e) Burning's Software Renderer
 (f) NullDevice
 (otherKey) exit");

            ConsoleKeyInfo keyInfo = Console.ReadKey();

            switch (keyInfo.KeyChar)
            {
                case 'a': driverType = E_DRIVER_TYPE.EDT_DIRECT3D9; break;
                case 'b': driverType = E_DRIVER_TYPE.EDT_DIRECT3D8; break;
                case 'c': driverType = E_DRIVER_TYPE.EDT_OPENGL; break;
                case 'd': driverType = E_DRIVER_TYPE.EDT_SOFTWARE; break;
                case 'e': driverType = E_DRIVER_TYPE.EDT_BURNINGSVIDEO; break;
                case 'f': driverType = E_DRIVER_TYPE.EDT_NULL; break;
                default: return;
            }

            // create device and exit if creation failed

            IrrlichtDevice device = IrrlichtNet.createDevice(driverType, new dimension2di(640, 480));

            if (device == null)
                return; // could not create selected driver.

            /* The creation was successful, now we set the event receiver and
                store pointers to the driver and to the gui environment. */

            device.setWindowCaption("Irrlicht Engine - User Interface Demo");

            IVideoDriver driver = device.getVideoDriver();
            IGUIEnvironment env = device.getGUIEnvironment();

            /*
            To make the font a little bit nicer, we load an external font
            and set it as the new default font in the skin.
            To keep the standard font for tool tip text, we set it to
            the built-in font.
            */

            IGUISkin skin = env.getSkin();
            IGUIFont font = null;
#if MULTI_LANGUAGE
            string tWinDir = Environment.GetEnvironmentVariable("windir");
            string tFontPath = System.IO.Path.Combine(tWinDir, "Fonts");
            tFontPath = System.IO.Path.Combine(tFontPath, "msgothic.ttc");
            font = env.getFont(tFontPath, 10, true, true);
#else
            font = env.getFont("../../media/fonthaettenschweiler.bmp");
#endif
            if (font != null)
                skin.setFont(font);

            skin.setFont(env.getBuiltInFont(), EGUI_DEFAULT_FONT.EGDF_TOOLTIP);

            /*
            We add three buttons. The first one closes the engine. The second
            creates a window and the third opens a file open dialog. The third
            parameter is the id of the button, with which we can easily identify
            the button in the event receiver.
            */

            env.addButton(new recti(10, 240, 110, 240 + 32), null, (int)GUI_ID.QUIT_BUTTON,
                    "Quit", "Exits Program");
            env.addButton(new recti(10, 280, 110, 280 + 32), null, (int)GUI_ID.NEW_WINDOW_BUTTON,
                    "New Window", "Launches a new Window");
            env.addButton(new recti(10, 320, 110, 320 + 32), null, (int)GUI_ID.FILE_OPEN_BUTTON,
                    "File Open", "Opens a file");

            /*
            Now, we add a static text and a scrollbar, which modifies the
            transparency of all gui elements. We set the maximum value of
            the scrollbar to 255, because that's the maximal value for
            a color value.
            Then we create an other static text and a list box.
            */

            env.addStaticText("Transparent Control:", new recti(150, 20, 350, 40), true);
            IGUIScrollBar scrollbar = env.addScrollBar(true,
                    new recti(150, 45, 350, 60), null, (int)GUI_ID.TRANSPARENCY_SCROLL_BAR);
            scrollbar.setMax(255);

            // set scrollbar position to alpha value of an arbitrary element
            scrollbar.setPos((int)env.getSkin().getColor(EGUI_DEFAULT_COLOR.EGDC_WINDOW).getAlpha());

            env.addStaticText("Logging ListBox:", new recti(50, 110, 250, 130), true);
            IGUIListBox listbox = env.addListBox(new recti(50, 140, 250, 210));
            env.addEditBox("Editable Text", new recti(350, 80, 550, 100));

            // Store the appropriate data in a context structure.
            SAppContext context = new SAppContext();
            context.device = device;
            context.counter = 0;
            context.listbox = listbox;

            // Then create the event receiver, giving it that context structure.
            SEventReceiver receiver = new SEventReceiver(context);

            // And tell the device to use our custom event receiver.
            device.setEventReceiver(receiver);


            /*
            And at last, we create a nice Irrlicht Engine logo in the top left corner. 
            */
            env.addImage(driver.getTexture("../../media/irrlichtlogo2.png"),
                    new position2di(10, 10));


            /*
            That's all, we only have to draw everything.
            */

            while (device.run() == true && driver != null)
                if (device.isWindowActive())
                {
                    driver.beginScene(true, true, new SColor(0, 200, 200, 200));

                    env.drawAll();

                    driver.endScene();
                }

            //device.drop();

            return;
        }
    }

        
    // Declare a structure to hold some context for the event receiver so that it
    // has it available inside its OnEvent() method.
    struct SAppContext
    {
	    public IrrlichtDevice device;
	    public int			counter;
	    public IGUIListBox	listbox;
    };

    // Define some values that we'll use to identify individual GUI controls.
    enum GUI_ID
    {
	    QUIT_BUTTON = 101,
	    NEW_WINDOW_BUTTON,
	    FILE_OPEN_BUTTON,
	    TRANSPARENCY_SCROLL_BAR
    };

    internal class SEventReceiver : IEventReceiver
    {
        public SEventReceiver(SAppContext context)
        {
            Context = context;
        }

        public override bool OnEvent(SEvent ev)
        {
            if (ev.EventType == EEVENT_TYPE.EET_GUI_EVENT)
            {
                int id = ev.getGUIEvent().Caller.getID();
                IGUIEnvironment env = Context.device.getGUIEnvironment();

                switch (ev.getGUIEvent().EventType)
                {

                    /*
			        If a scrollbar changed its scroll position, and it is
			        'our' scrollbar (the one with id GUI_ID_TRANSPARENCY_SCROLL_BAR), then we change
			        the transparency of all gui elements. This is a very
			        easy task: There is a skin object, in which all color
			        settings are stored. We simply go through all colors
			        stored in the skin and change their alpha value.
			        */
                    case EGUI_EVENT_TYPE.EGET_SCROLL_BAR_CHANGED:
                        if (id == (int) GUI_ID.TRANSPARENCY_SCROLL_BAR)
                        {
                            IGUIElement gui = ev.getGUIEvent().Caller;
                            IGUIScrollBar scrollBar = IGUIScrollBar.cast(gui);
                            int pos = scrollBar.getPos();

                            for (uint i = 0; i < (int)EGUI_DEFAULT_COLOR.EGDC_COUNT; i++)
                            {
                                SColor col = env.getSkin().getColor((EGUI_DEFAULT_COLOR) i);
                                col.setAlpha((uint) pos);
                                env.getSkin().setColor((EGUI_DEFAULT_COLOR) i, col);
                            }

                        }
                        break;

                    /*
			        If a button was clicked, it could be one of 'our'
			        three buttons. If it is the first, we shut down the engine.
			        If it is the second, we create a little window with some
			        text on it. We also add a string to the list box to log
			        what happened. And if it is the third button, we create
			        a file open dialog, and add also this as string to the list box.
			        That's all for the event receiver.
			        */
                    case EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED:
                        switch (id)
                        {
                            case (int) GUI_ID.QUIT_BUTTON:
                                Context.device.closeDevice();
                                return true;

                            case (int) GUI_ID.NEW_WINDOW_BUTTON:
                                {
                                    Context.listbox.addItem("Window created");
                                    Context.counter += 30;
                                    if (Context.counter > 200)
                                        Context.counter = 0;

                                    IGUIWindow window = env.addWindow(
                                        new recti(100 + Context.counter, 100 + Context.counter, 300 + Context.counter,
                                                  200 + Context.counter),
                                        false, // modal?
                                        "Test window");

                                    env.addStaticText("Please close me",
                                                      new recti(35, 35, 140, 50),
                                                      true, // border?
                                                      false, // wordwrap?
                                                      window);
                                }
                                return true;

                            case (int) GUI_ID.FILE_OPEN_BUTTON:
                                Context.listbox.addItem("File open");
                                env.addFileOpenDialog("Please choose a file.");
                                return true;

                            default:
                                return false;
                        }

                    default:
                        break;
                }
            }

            return false;
        }

        private SAppContext Context;
    } ;


}
