/*
 * Copyright (C) 2002 Robert Ernst <robert.ernst@linux-solutions.at>
 *
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file LICENSE.GPL included in the
 * packaging of this file.
 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See COPYING for GPL licensing information.
 *
 */

#include <qpe/qpemenubar.h>
#include <qpe/config.h>
#include <qapplication.h>
#include <qmainwindow.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qpainter.h>
#include <qdir.h>
#include "ShisenSho.h"
#include "Board.h"
#include <stdlib.h>
#include "StringTable.h"
#include "CConfigDlg.h"
#include "ScoreListDlg.h"

#define TIME_TO_SEC(h,m,s) (h*3600+m*60+s)

//Score Files
#define SCORE_DIR       ".m-shisensho"
#define C_SCORE_FNAME01 ".m-shisensho/score01.dat"
#define C_SCORE_FNAME02 ".m-shisensho/score02.dat"
#define C_SCORE_FNAME03 ".m-shisensho/score03.dat"
#define C_SCORE_FNAME04 ".m-shisensho/score04.dat"

ShisenSho::ShisenSho(QWidget *parent, const char *name, WFlags f) :
    QMainWindow(parent, name, f)
{
    /* initialize random number generator */
    srand(::time(0));

    /* set windows style and caption */
    setCaption(STRING_UTF8(ID_STRING_00));

    /* setup score directory */
    QDir dir;
    if( !dir.exists( SCORE_DIR ) ){
      dir.mkdir( SCORE_DIR );
    }
    initHiScoreMng();

    /* create the play field as the central widget */
    m_Status = new CStatus();
    m_ConfigInfo = new CConfigInfo();
    m_board = new Board(this);
    m_board->setStatus(m_Status);
   
    setCentralWidget(m_board);
    m_board->show();
    connect(m_board, SIGNAL(changed()), this, SLOT(enableItems()));
    connect(m_board, SIGNAL(endOfGame()), this, SLOT(gameOver()));
    connect(m_board, SIGNAL(pushStart()), this, SLOT(startGame()));
    connect(m_board, SIGNAL(pushConfig()), this, SLOT(changeConfig()));

    /* get a timer that updates the board */
    QTimer *timer = new QTimer(this);
    timer->start(1000);
    connect(timer, SIGNAL(timeout()), this, SLOT(updateScore()));

    /* create the game submenu */
    m_gameMenu = new QPopupMenu(0, "game_menu");
    m_idNewgame = m_gameMenu->insertItem(STRING_UTF8(ID_STRING_101), this, SLOT(start()));
    m_idGiveUp = m_gameMenu->insertItem("Give Up", this, SLOT(giveup()));
    m_gameMenu->insertSeparator();
    m_idHint = m_gameMenu->insertItem(STRING_UTF8(ID_STRING_102), this, SLOT(hint()));
    m_idUndo = m_gameMenu->insertItem(STRING_UTF8(ID_STRING_103), this, 
			   SLOT(undo()), 0, 0x0100);
    m_idRedo = m_gameMenu->insertItem(STRING_UTF8(ID_STRING_104), this, 
			   SLOT(redo()), 0, 0x0101);
    m_idScore = m_gameMenu->insertItem("Hi-Score", this, 
			   SLOT(score()), 0, 0x0102);
    m_gameMenu->insertSeparator();
    m_gameMenu->insertItem(STRING_UTF8(ID_STRING_105), this, SLOT(close()));

    m_gameMenu->setItemEnabled(m_idNewgame, false);
    m_gameMenu->setItemEnabled(m_idGiveUp, false);
    m_gameMenu->setItemEnabled(m_idHint, false);

    /* create the config submenu */
    m_configMenu = new QPopupMenu(0, "config_menu");
    m_idConfig = m_configMenu->insertItem(STRING_UTF8(ID_STRING_110), this, 
			   SLOT(changeConfig()));

    /* create the menu */
    QPEMenuBar *menu = new QPEMenuBar(this);
    menu->insertItem(STRING_UTF8(ID_STRING_1000), m_gameMenu);
    menu->insertItem(STRING_UTF8(ID_STRING_1001), m_configMenu);

    /* read the configuration file */
    readConfig();

    /* at last, set up the board */
    m_board->setSolvableFlag(true);
    m_board->setGravityFlag(m_gravity);
    m_board->setShuffle(m_iq);
    m_board->setDelay(m_speed);
    switch (m_size) {
    case 0: m_board->setSize(8, 6); break;
    case 1: m_board->setSize(8, 8); break;
    case 2: m_board->setSize(10, 6); break;
    case 3: m_board->setSize(10, 8); break;
    }

    disableItems();

}

ShisenSho::~ShisenSho()
{
    writeConfig();
    deleteHiScoreMng();
}

void ShisenSho::readConfig(void)
{
    Config cfg("ShisenSho");

    cfg.setGroup("Settings");

    int speed = cfg.readNumEntry("Speed", 500);
    if (speed <= 125) {
	m_speed = 125;
    } else if (speed <= 250) {
	m_speed = 250;
    } else if (speed <= 500) {
	m_speed = 500;
    } else if (speed <= 750) {
	m_speed = 750;
    } else {
	m_speed = 1000;
    }

    int size = cfg.readNumEntry("Size", 3);
    if (size <= 0) {
	m_size = 0;
    } else if (size >= 3) {
	m_size = 3;
    } else {
	m_size = size;
    }

    int iq = cfg.readNumEntry("IQ", 5);
    if (iq <= 1) {
	m_iq = 1;
    } else if (iq <= 5) {
	m_iq = 5;
    } else {
	m_iq = 9;
    }

    int gravity = cfg.readNumEntry("Gravity", 0);
    if (gravity <= 0) {
	m_gravity = 0;
    } else {
	m_gravity = 1;
    }
}

void ShisenSho::writeConfig(void)
{
    Config cfg("ShisenSho");
    cfg.setGroup("Settings");
    cfg.writeEntry("Speed", m_speed);
    cfg.writeEntry("Size", m_size);
    cfg.writeEntry("IQ", m_iq);
    cfg.writeEntry("Gravity", m_gravity);
}

void ShisenSho::updateSpeed(int speed)
{
    m_speedMenu->setItemChecked(m_speed, false);
    m_speed = speed;
    m_speedMenu->setItemChecked(m_speed, true);
    writeConfig();
    if (!m_board->canUndo()) {
	m_board->setDelay(speed);
    } else {
	m_board->setDelay(speed);
	start();
    }
}

void ShisenSho::updateSize(int size)
{
    m_sizeMenu->setItemChecked(m_size, false);
    m_size = size;
    m_sizeMenu->setItemChecked(m_size, true);
    writeConfig();
    switch (size) {
    case 0: m_board->setSize(8, 6); break;
    case 1: m_board->setSize(8, 8); break;
    case 2: m_board->setSize(10, 6); break;
    case 3: m_board->setSize(10, 8); break;
    }
    start();
}

void ShisenSho::updateIQ(int iq)
{
    m_iqMenu->setItemChecked(m_iq, false);
    m_iq = iq;
    m_iqMenu->setItemChecked(m_iq, true);
    writeConfig();
    if (!m_board->canUndo()) {
	m_board->setShuffle(iq);
    } else {
	m_board->setShuffle(iq);
	start();
    }
}

void ShisenSho::updateGravity(int gravity)
{
    m_gravityMenu->setItemChecked(m_gravity, false);
    m_gravity = gravity;
    m_gravityMenu->setItemChecked(m_gravity, true);
    writeConfig();
    if (!m_board->canUndo()) {
	m_board->setGravityFlag(gravity);
    } else {
	m_board->setGravityFlag(gravity);
	start();
    }
}

void ShisenSho::updateScore(void)
{
  m_Status->event(CStatus::TIMEOUT);
  //taiga
  int status;
  int x,y;

  status = m_Status->getStatus();

  switch(status){
  case STS_TITLE:
  case STS_UNDER_BOOT:
    if( m_Status->SyncStatus() ){
      m_board->repaint(0, 0, width(), height(), false);
    }
    break;
  case STS_REQSTART:
    m_ConfigInfo->GetSize(&x, &y);
    m_board->setSize(x, y);
    m_Status->event(CStatus::SETUP_COMPLETED);
    break;
  case STS_UNDERPLAY:
    m_board->repaint(0, 0, width(), 22, false);
    break;
  }

}

void ShisenSho::start(void)
{
    m_board->setSolvableFlag(true);
    m_board->setCheated(false);
    m_board->newGame();
    enableItems();
}

void ShisenSho::hint(void)
{
    qApp->processEvents();
    m_board->getHint();
    m_board->setCheated(true);
    enableItems();
}

void ShisenSho::undo(void)
{
    if (m_board->canUndo()) {
	m_board->undo();
	m_board->setCheated(true);
	enableItems();
    }
}

void ShisenSho::redo(void)
{
    if(m_board->canRedo()) {
	m_board->redo();
	enableItems();
    }
}

void ShisenSho::enableItems(void)
{
    m_gameMenu->setItemEnabled(m_idUndo, m_board->canUndo());
    m_gameMenu->setItemEnabled(m_idRedo, m_board->canRedo());

    m_gameMenu->setItemEnabled(m_idNewgame, true);
    m_gameMenu->setItemEnabled(m_idGiveUp, true);
    m_gameMenu->setItemEnabled(m_idHint, true);

    m_configMenu->setItemEnabled(m_idConfig, false);

}
void ShisenSho::disableItems(void)
{
  m_gameMenu->setItemEnabled(m_idNewgame, false);
  m_gameMenu->setItemEnabled(m_idGiveUp, false);
  m_gameMenu->setItemEnabled(m_idHint, false);

  m_gameMenu->setItemEnabled(m_idUndo, false);
  m_gameMenu->setItemEnabled(m_idRedo, false);

  m_configMenu->setItemEnabled(m_idConfig, true);

}
void ShisenSho::gameOver(void)
{
    if (m_board->tilesLeft()) {
	QMessageBox::information(this, STRING_UTF8(ID_STRING_200),
				 STRING_UTF8(ID_STRING_2001));
    } else {
      struct tm *ptm;
      time_t now;
      bool ret;
      now = time(NULL);
      ptm = localtime(&now);
      int seconds = m_board->getTimeForGame();

      QString s,msg;
      m_pScoreMng[m_ConfigInfo->GetSize()]->RegistHiScore(ptm,seconds);
      ret = m_pScoreMng[m_ConfigInfo->GetSize()]->IsRankIn(seconds);
      if( ret ){
	msg = STRING_UTF8(ID_STRING_2004); 
      } else {
	msg = STRING_UTF8(ID_STRING_2003);
      }
      m_pScoreMng[m_ConfigInfo->GetSize()]->SaveHiScore();
      s.sprintf("%02d:%02d:%02d", seconds / 3600, (seconds / 60) % 60, 
		seconds % 60);
      QMessageBox::information(this, STRING_UTF8(ID_STRING_2002),msg.arg(s));

    }
    printf("%d\n",m_size);
    start();
}

void ShisenSho::startGame(void)
{
  int x,y;

  m_ConfigInfo->GetSize(&x,&y);
  m_board->setSize(x, y);
  m_board->setGravityFlag(m_ConfigInfo->GetGravity());
  start();

}
void ShisenSho::changeConfig(void)
{

    CConfigDlg win(this, "Options", true);
    
    win.setConfigInfo(m_ConfigInfo);
    win.setCaption( STRING_UTF8(ID_STRING_110));

    int ret = win.exec();
    if( ret == QDialog::Accepted ){
      win.GetConfigItem();
    }
}
void ShisenSho::score(void)
{
    CScoreListDlg win(this, "Options", true);
    win.SetScoreMng(&m_pScoreMng[0]);
    int ret = win.exec();
}
void ShisenSho::giveup(void)
{
 
  disableItems();
  m_Status->init();
  m_board->repaint(0, 0, width(), height(), false);
}
//
// Init ScoreData
//
void ShisenSho::registHiScoreData( CScoreMng *pScoreMng )
{
  struct tm *ptm;
  time_t now;

  now = time(NULL);
  ptm = localtime(&now);
  pScoreMng->RegistHiScore( ptm, TIME_TO_SEC(0,10,0) );
  pScoreMng->RegistHiScore( ptm, TIME_TO_SEC(0,10,10) );
  pScoreMng->RegistHiScore( ptm, TIME_TO_SEC(0,10,20) );
  pScoreMng->RegistHiScore( ptm, TIME_TO_SEC(0,10,30) );
  pScoreMng->RegistHiScore( ptm, TIME_TO_SEC(0,10,40) );
  pScoreMng->SaveHiScore();
  
}
//
// Create Score Managers
//
void ShisenSho::initHiScoreMng()
{
    
  // create score Manager
  m_pScoreMng[0] = new CScoreMng( C_SCORE_FNAME01 );
  m_pScoreMng[1] = new CScoreMng( C_SCORE_FNAME02 );
  m_pScoreMng[2] = new CScoreMng( C_SCORE_FNAME03 );
  m_pScoreMng[3] = new CScoreMng( C_SCORE_FNAME04 );

  for(int i=0; i < CATEGORY_NUM; i++){
    m_pScoreMng[i]->SetOrder( false );
    if( !m_pScoreMng[i]->DownLoadHiScore() ){
      // read scorefile failed.
      registHiScoreData( m_pScoreMng[i] );
    }    
  }
  
}
//
// Delete Score Managers
//
void ShisenSho::deleteHiScoreMng()
{
  for(int i=0; i < CATEGORY_NUM; i++){
    delete m_pScoreMng[i];
  }  
}
