/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <p><code>VideoBounds</code>クラスは、画像のサイズと2次元平面上での位置をカプセル化したものです。
 * サイズは2つのint値で保持します。位置は2つのdouble値で保持します。</p>
 * 
 * <p><code>VideoBounds</code>オブジェクトは不変です。
 * <code>VideoBounds</code>オブジェクトは作成したあとに変更できないため、共用することができます。</p>
 */
public final class VideoBounds {

	/**
	 * X軸方向の位置です。
	 */
	public final double x;

	/**
	 * Y軸方向の位置です。
	 */
	public final double y;

	/**
	 * 画像の幅です。
	 */
	public final int width;

	/**
	 * 画像の高さです。
	 */
	public final int height;


	/**
	 * 位置とサイズを指定して<code>VideoBounds</code>オブジェクトを生成します。
	 * 
	 * @param x			X軸方向の位置
	 * @param y			Y軸方向の位置
	 * @param width		画像の幅
	 * @param height	画像の高さ
	 * @throws IllegalArgumentException	引数<code>width</code>または<code>height</code>が負の値の場合に発生
	 */
	public VideoBounds(double x, double y, int width, int height) {
		super();

		if (width < 0 || height < 0) {
			throw new IllegalArgumentException("width and height must be grater than or equal 0");
		}

		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
	}

	/**
	 * サイズを指定し、位置(0, 0)で<code>VideoBounds</code>オブジェクトを生成します。
	 * 
	 * @param width		画像の幅
	 * @param height	画像の高さ
	 * @throws IllegalArgumentException	引数<code>width</code>または<code>height</code>が負の値の場合に発生
	 */
	public VideoBounds(int width, int height) {
		this(0, 0, width, height);
	}

	/**
	 * サイズを指定し、位置(0, 0)で<code>VideoBounds</code>オブジェクトを生成します。
	 * 
	 * @param size 画像のサイズ
	 * @throws IllegalArgumentException	引数<code>size</code>の幅または高さが負の値の場合に発生
	 */
	public VideoBounds(Size2i size) {
		this(0, 0, size.width, size.height);
	}

	/**
	 * この<code>VideoBounds</code>オブジェクトのサイズが 0 の場合は true を返します。
	 * これは、<code>width</code> または <code>height</code> の少なくともどちらか一方が 0 の場合です。
	 * 
	 * @return この<code>VideoBounds</code>オブジェクトのサイズが 0 の場合は true、そうでない場合は false
	 */
	public boolean isEmpty() {
		return (width == 0 || height == 0);
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + height;
		result = prime * result + width;
		long temp;
		temp = Double.doubleToLongBits(x);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(y);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		VideoBounds other = (VideoBounds) obj;
		if (height != other.height)
			return false;
		if (width != other.width)
			return false;
		if (Double.doubleToLongBits(x) != Double.doubleToLongBits(other.x))
			return false;
		if (Double.doubleToLongBits(y) != Double.doubleToLongBits(other.y))
			return false;
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "VideoBounds [x=" + x + ", y=" + y + ", width=" + width + ", height=" + height + "]";
	}

}
