/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import java.util.Collection;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;

public interface IVideoRenderSupport {

	/**
	 * 引数 <code>bounds</code> の位置とサイズ、コンテキストの色深度でビデオ用バッファを作成します。
	 * 
	 * @param bounds 作成するビデオ用バッファの位置とサイズ
	 * @return 作成されたビデオ用バッファ
	 */
	IVideoBuffer createVideoBuffer(VideoBounds bounds);

	/**
	 * 引数 <code>bounds</code> の位置とサイズ、引数 <code>colorMode</code> の色深度でビデオ用バッファを作成します。
	 * 
	 * @param bounds 作成するビデオ用バッファの位置とサイズ
	 * @param colorMode 作成するビデオ用バッファの色深度
	 * @return 作成されたビデオ用バッファ
	 */
	IVideoBuffer createVideoBuffer(VideoBounds bounds, ColorMode colorMode);

	/**
	 * <p>
	 * <code>output</code> のテクスチャをフレームバッファの出力先とし、<code>operation</code> の処理を実行します。
	 * <code>input</code> は可変長引数で、0番目がテクスチャ番号0に、1番目がテクスチャ番号1に、N番目がテクスチャ番号Nにバインドされます。
	 * また、テクスチャマッピングは有効な状態となります。
	 * このメソッドは <code>operation</code> の処理の前後で <code>glPushAttrib</code> / <code>glPopAttrib</code> を呼び出し、
	 * OpenGL の状態を保存／復帰します。<code>pushAttribs</code> はその際 <code>glPushAttrib</code> に渡されるビットマスクです。 
	 * (実際には <code>pushAttribs</code> のビットに加え、いくつかのビットが追加されて glPushAttrib が呼び出されますが、
	 * その動作に依存して <code>pushAttribs</code> の設定を省略してはいけません。<code>operation</code>
	 * の処理で変更される可能性がある状態に対しては、必ず <code>pushAttribs</code> のビットを設定してください。)
	 * </p>
	 * <p>
	 * <code>output</code> に <code>null</code> を指定した場合、<code>input</code>
	 * の最初のバッファと同じ位置とサイズで出力バッファが作成され、戻り値に返されます。
	 * <code>input</code> に何も指定しないこともできますが、<code>output</code> に <code>null</code>
	 * を指定した場合は1つ以上 <code>input</code> が必要です。
	 * </p>
	 * 
	 * @param operation フレームバッファを使用して行う処理
	 * @param pushAttribs <code>operation</code> の処理前後で保存／復帰する OpenGL の状態を指定するビットマスク
	 * @param output フレームバッファの出力先となるバッファ
	 * @param input <code>operation</code> の処理時にバインドされるバッファ
	 * @return <code>output</code> が <code>null</code> の場合は新たに作成されたバッファ、そうでない場合は <code>output</code>
	 */
	IVideoBuffer useFramebuffer(
			Runnable operation, int pushAttribs,
			IVideoBuffer output, IVideoBuffer... input);

	/**
	 * <code>program</code> のシェーダプログラムを使用して <code>operation</code> の処理を実行します。
	 * <code>uniforms</code> はシェーダプログラムへ渡す Uniform 変数のコレクションです。
	 * <code>operation</code>, <code>pushAttribs</code>, <code>output</code> および <code>input</code> は
	 * {@link #useFramebuffer(Runnable, int, IVideoBuffer, IVideoBuffer...)} の引数とほぼ同じですが、
	 * テクスチャマッピングは有効にならないことが異なります (テクスチャのバインドは行われます) 。 
	 * 
	 * @param program <code>operation</code> の処理の際に使用するシェーダプログラム
	 * @param uniforms シェーダプログラムへ渡す Uniform 変数のコレクション
	 * @param operation シェーダプログラムを使用して実行する処理
	 * @param pushAttribs <code>operation</code> の処理前後で保存／復帰する OpenGL の状態を指定するビットマスク
	 * @param output 出力バッファ
	 * @param input <code>operation</code> の処理時にバインドされるバッファ
	 * @return <code>output</code> が <code>null</code> の場合は新たに作成されたバッファ、そうでない場合は <code>output</code>
	 */
	IVideoBuffer useShaderProgram(
			IShaderProgram program, Collection<GLUniformData> uniforms,
			Runnable operation, int pushAttribs,
			IVideoBuffer output, IVideoBuffer... input);

	/**
	 * {@link #useShaderProgram(IShaderProgram, Collection, Runnable, int, IVideoBuffer, IVideoBuffer...)}
	 * のマルチレンダーターゲット(MRT)版です。<code>program</code> にはMRTを使用したシェーダプログラムを指定します。
	 * <code>output</code> には、シェーダプログラムが使用するレンダーターゲットと同じ数の出力バッファを指定します。
	 * <code>program</code> は、MRTを使用しないシェーダプログラムでも構いません。
	 * その場合、<code>output</code> に指定できる出力バッファは1つだけです。
	 * 
	 * @param program <code>operation</code> の処理の際に使用するシェーダプログラム
	 * @param uniforms シェーダプログラムへ渡す Uniform 変数のコレクション
	 * @param operation シェーダプログラムを使用して実行する処理
	 * @param pushAttribs <code>operation</code> の処理前後で保存／復帰する OpenGL の状態を指定するビットマスク
	 * @param output 出力バッファの配列
	 * @param input <code>operation</code> の処理時にバインドされるバッファ
	 */
	void useShaderProgramMRT(
			IShaderProgram program, Collection<GLUniformData> uniforms,
			Runnable operation, int pushAttribs,
			IVideoBuffer output[], IVideoBuffer... input);

	/**
	 * <p>
	 * <code>program</code> のシェーダプログラムを使用して <code>output.getBounds()</code>
	 * の矩形を描画し、<code>output</code> に結果を出力します。このとき <code>output.getBounds()</code>
	 * の領域全体がちょうど視体積に収まるように {@link #ortho2D(IVideoBuffer)} で平行投影されます。
	 * {@link #useShaderProgram(IShaderProgram, Collection, Runnable, int, IVideoBuffer, IVideoBuffer...)}
	 * と同様にテクスチャのバインドが行われます。また、<code>output</code> と各 <code>input</code>
	 * の位置関係を元にテクスチャ座標が設定されます。
	 * </p>
	 * <p>
	 * <code>output</code> に <code>null</code> を指定した場合、<code>input</code>
	 * の最初のバッファと同じ位置とサイズで出力バッファが作成され、戻り値に返されます。
	 * <code>input</code> に何も指定しないこともできますが、<code>output</code> に <code>null</code>
	 * を指定した場合は1つ以上 <code>input</code> が必要です。
	 * </p>
	 * 
	 * @param program シェーダプログラム
	 * @param uniforms シェーダプログラムへ渡す Uniform 変数のコレクション
	 * @param output 出力バッファ
	 * @param input 入力バッファ
	 * @return <code>output</code> が <code>null</code> の場合は新たに作成されたバッファ、そうでない場合は <code>output</code>
	 */
	IVideoBuffer useShaderProgram(
			IShaderProgram program, Collection<GLUniformData> uniforms,
			IVideoBuffer output, IVideoBuffer... input);

	/**
	 * {@link #useShaderProgram(IShaderProgram, Collection, IVideoBuffer, IVideoBuffer...)}
	 * のマルチレンダーターゲット(MRT)版です。<code>program</code> にはMRTを使用したシェーダプログラムを指定します。
	 * <code>output</code> には、シェーダプログラムが使用するレンダーターゲットと同じ数の出力バッファを指定します。
	 * <code>program</code> は、MRTを使用しないシェーダプログラムでも構いません。
	 * その場合、<code>output</code> に指定できる出力バッファは1つだけです。
	 * 
	 * @param program シェーダプログラム
	 * @param uniforms シェーダプログラムへ渡す Uniform 変数のコレクション
	 * @param output 出力バッファの配列
	 * @param input 入力バッファ
	 */
	void useShaderProgramMRT(
			IShaderProgram program, Collection<GLUniformData> uniforms,
			IVideoBuffer[] output, IVideoBuffer... input);

	/**
	 * <p>
	 * <code>output.getBounds()</code> の矩形領域全体が視体積にちょうど収まるように平行投影の設定を行います
	 * (ビューポートと投影行列が設定されます。またモデルビュー行列は単位行列に初期化されます) 。
	 * </p>
	 * <p>
	 * これは <code>ortho2D(output.getBounds())</code> と同じです。 
	 * </p>
	 * 
	 * @param output 平行投影の対象領域を持つビデオバッファ
	 * @see #ortho2D(VideoBounds)
	 */
	void ortho2D(IVideoBuffer output);

	/**
	 * <code>bounds</code> の矩形領域全体が視体積にちょうど収まるように平行投影の設定を行います
	 * (ビューポートと投影行列が設定されます。またモデルビュー行列は単位行列に初期化されます) 。
	 * 
	 * @param bounds 平行投影の対象領域
	 * @see #ortho2D(IVideoBuffer)
	 */
	void ortho2D(VideoBounds bounds);

	void quad2D(IVideoBuffer output, IVideoBuffer... input);

	void quad2D(VideoBounds bounds, double[][]... texCoords);

	void quad2D(double left, double top, double right, double bottom, double[][]... texCoords);

	void polygon2D(double[][][] contours, IVideoBuffer output, IVideoBuffer... input);

	void polygon2D(double[][][] contours, VideoBounds bounds, double[][]... texCoords);

	/**
	 * 引数 <code>src</code> のバッファが保持する画像を、引数 <code>dst</code> のバッファにコピーします。
	 * <code>src</code> の <code>getBounds</code> メソッドが返す矩形に対応する
	 * <code>dst</code> 内の領域にコピーされます。この矩形が <code>dst</code> 内に収まらない場合、
	 * <code>dst</code> 内に収まる部分だけがコピーされます。
	 * 
	 * @param src コピー元のバッファ
	 * @param dst コピー先のバッファ
	 * 
	 * @since 0.5.0
	 */
	void copy(IVideoBuffer src, IVideoBuffer dst);

}
