/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.TimeStretchLayerOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.views.layercomp.LayerElement;
import ch.kuramo.javie.app.views.layercomp.TimelineManager;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.Util;

public class TimeStretchLayerAction extends Action {

	private final TreeViewer viewer;

	public TimeStretchLayerAction(TreeViewer viewer) {
		super("時間伸縮...");
		this.viewer = viewer;

		setEnabled(false);

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				boolean enabled = false;

				TreeSelection selection = (TreeSelection) event.getSelection();
				if (!selection.isEmpty()) {
					enabled = true;
					for (Object element : selection.toList()) {
						if (!(element instanceof LayerElement)) {
							enabled = false;
							break;
						}
					}
				}

				setEnabled(enabled);
			}
		});
	}

	public void run() {
		final List<Layer> layers = Util.newList();

		TreeSelection selection = (TreeSelection) viewer.getSelection();
		for (Object element : selection.toList()) {
			if (!(element instanceof LayerElement)) {
				return;
			}
			layers.add(((LayerElement) element).layer);
		}

		if (layers.isEmpty()) {
			return;
		}


		LayerComposition comp = (LayerComposition) viewer.getData(LayerCompositionView.LAYER_COMPOSITION);
		Layer firstLayer = layers.get(0);

		final Time frameDuration = comp.getFrameDuration();

		final double oldRate = firstLayer.getRate();
		final Time oldDuration = (oldRate > 0) ? firstLayer.getOutPoint().subtract(firstLayer.getInPoint())
											   : firstLayer.getInPoint().subtract(firstLayer.getOutPoint());
		final String oldTimeCode = TimeCode.toTimeCode(oldDuration, frameDuration);

		final int base = (int)Math.round((double)frameDuration.timeScale / frameDuration.timeValue);
		final boolean drop = (oldTimeCode.indexOf(';') != -1);

		Time origDuration = new Time((long)(oldDuration.timeValue*oldRate), oldDuration.timeScale);
		final String origTimeCode = TimeCode.toTimeCode(origDuration, frameDuration);


		Dialog dialog = new Dialog(viewer.getTree().getShell()) {
			private Text stretchText;
			private Text durationText;
			private Label baseLabel;
			private Button inPointBtn;
			private Button curFrameBtn;
			private Button outPointBtn;

			private ModifyListener stretchListener = new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					double newRate;
					try {
						newRate = 100 / Double.parseDouble(stretchText.getText().trim());
					} catch (NumberFormatException ex) {
						newRate = 0;
					}
					if (newRate == 0) {
						e.display.beep();
						baseLabel.setText("");
						getButton(IDialogConstants.OK_ID).setEnabled(false);
						return;
					} else {
						getButton(IDialogConstants.OK_ID).setEnabled(true);
					}

					Time newDuration = new Time((long)(oldDuration.timeValue*oldRate/newRate), oldDuration.timeScale);
					String newTimeCode = TimeCode.toTimeCode(newDuration, frameDuration);

					durationText.removeModifyListener(durationListener);
					durationText.setText(newTimeCode);
					durationText.addModifyListener(durationListener);

					baseLabel.setText(formatBase(newTimeCode, base, drop));
				}
			};

			private ModifyListener durationListener = new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					String newTimeCode = durationText.getText().trim();
					long frames = (newTimeCode.length() > 0)
							? TimeCode.parseTimeCode(newTimeCode, frameDuration, drop) : 0;
					if (frames == 0) {
						e.display.beep();
						baseLabel.setText("");
						getButton(IDialogConstants.OK_ID).setEnabled(false);
						return;
					} else {
						getButton(IDialogConstants.OK_ID).setEnabled(true);
					}

					Time newDuration = Time.fromFrameNumber(frames, frameDuration);
					double newRate = oldDuration.timeValue*oldRate / newDuration.timeValue;

					stretchText.removeModifyListener(stretchListener);
					stretchText.setText(formatStretch(100/newRate));
					stretchText.addModifyListener(stretchListener);

					newTimeCode = TimeCode.toTimeCode(newDuration, frameDuration);
					baseLabel.setText(formatBase(newTimeCode, base, drop));
				}
			};

			protected void configureShell(Shell newShell) {
				super.configureShell(newShell);
				newShell.setText("時間伸縮");
			}

			protected Control createDialogArea(Composite parent) {
				Composite composite = (Composite) super.createDialogArea(parent);
				RowLayout layout = new RowLayout(SWT.VERTICAL);
				layout.fill = true;
				layout.marginWidth = 5;
				layout.marginHeight = 5;
				composite.setLayout(layout);

				createGroup1(composite);
				createGroup2(composite);

				parent.pack();
				return composite;
			}

			private void createGroup1(Composite parent) {
				Group group1 = new Group(parent, SWT.NONE);
				group1.setText("伸縮");
				FillLayout layout1 = new FillLayout();
				layout1.marginWidth = 5;
				layout1.marginHeight = 5;
				group1.setLayout(layout1);

				GridBuilder gb1 = new GridBuilder(group1, 3, false);

								gb1.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "元のデュレーション:");
								gb1.hSpan(1).hAlign(SWT.FILL).label(SWT.NULL, origTimeCode + "      ");
								gb1.hSpan(1).size(10, 10).composite(SWT.NULL);

								gb1.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "伸縮比率:");
				stretchText =	gb1.hSpan(1).hAlign(SWT.FILL).text(SWT.BORDER, formatStretch(100/oldRate));
								gb1.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "%");

								gb1.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "新規デュレーション:");
				durationText =	gb1.hSpan(1).hAlign(SWT.FILL).text(SWT.BORDER, oldTimeCode);
				baseLabel =		gb1.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, formatBase(oldTimeCode, base, drop));

				stretchText.addModifyListener(stretchListener);
				durationText.addModifyListener(durationListener);
			}

			private void createGroup2(Composite parent) {
				Group group2 = new Group(parent, SWT.NONE);
				group2.setText("基準にする地点");
				RowLayout layout2 = new RowLayout(SWT.VERTICAL);
				layout2.marginWidth = 7;
				layout2.marginHeight = 7;
				group2.setLayout(layout2);

				inPointBtn = new Button(group2, SWT.RADIO);
				curFrameBtn = new Button(group2, SWT.RADIO);
				outPointBtn = new Button(group2, SWT.RADIO);
				inPointBtn.setText("レイヤーインポイント");
				curFrameBtn.setText("現在のフレーム");
				outPointBtn.setText("レイヤーアウトポイント");
				inPointBtn.setSelection(true);
			}

			private String formatStretch(double stretch) {
				String s = String.format("%.4f", stretch);
				int dotIndex = s.indexOf('.');
				if (dotIndex != -1) {
					for (int i = s.length()-1; i >= dotIndex; --i) {
						char c = s.charAt(i);
						if (c == '0') continue;
						return s.substring(0, (c == '.') ? i : i+1);
					}
				}
				return s;
			}

			private String formatBase(String timeCode, int base, boolean drop) {
				return String.format("= %s (ベース %d%s)  ", timeCode, base, drop ? "ドロップ" : "");
			}

			protected void okPressed() {
				double newRate = 100 / Double.parseDouble(stretchText.getText().trim());
				boolean inPoint = inPointBtn.getSelection();
				boolean curFrame = curFrameBtn.getSelection();

				super.okPressed();

				ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
				if (curFrame) {
					TimelineManager tlm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
					pm.postOperation(new TimeStretchLayerOperation(pm, layers, newRate, tlm.getCurrentTime()));
				} else {
					pm.postOperation(new TimeStretchLayerOperation(pm, layers, newRate, inPoint));
				}
			}
		};

		dialog.open();
	}

}
