/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class DuplicateEffectsOperation extends ProjectOperation {

	private final String _compId;

	private final List<Object[]> _srcData;

	private final String _dstLayerId;

	private final int _dstIndex;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public DuplicateEffectsOperation(ProjectManager pm,
			List<EffectableLayer> srcLayers, List<Effect> effects,
			EffectableLayer dstLayer, int insertBefore) {

		super(pm, "エフェクトの複製");
		InjectorHolder.getInjector().injectMembers(this);

		if (srcLayers.size() != effects.size()) {
			throw new IllegalArgumentException();
		}

		if (srcLayers.isEmpty()) {
			_noEffect = true;
			_compId = null;
			_srcData = null;
			_dstLayerId = null;
			_dstIndex = 0;
			return;
		}

		LayerComposition comp = pm.checkLayer(srcLayers.get(0));
		if (dstLayer != null && pm.checkLayer(dstLayer) != comp) {
			throw new IllegalArgumentException();
		}

		_compId = comp.getId();
		_srcData = Util.newList();

		if (dstLayer != null) {
			_dstLayerId = dstLayer.getId();

			int dstEffectCount = dstLayer.getEffects().size();
			if (insertBefore < 0 || insertBefore > dstEffectCount) {
				_dstIndex = dstEffectCount;
			} else {
				_dstIndex = insertBefore;
			}
		} else {
			_dstLayerId = null;
			_dstIndex = 0;
		}

		for (ListIterator<EffectableLayer> it = srcLayers.listIterator(); it.hasNext(); ) {
			int i = it.nextIndex();
			EffectableLayer srcLayer = it.next();
			Effect effect = effects.get(i);

			int layerIndex = comp.getLayers().indexOf(srcLayer);
			if (layerIndex == -1) {
				throw new IllegalArgumentException();
			}

			int effectIndex = srcLayer.getEffects().indexOf(effect);
			if (effectIndex == -1) {
				throw new IllegalArgumentException();
			}

			_srcData.add(new Object[] { layerIndex, effectIndex, _encoder.encodeElement(effect) });
		}

		Collections.sort(_srcData, new Comparator<Object[]>() {
			public int compare(Object[] o1, Object[] o2) {
				int compare = (Integer)o1[0] - (Integer)o2[0];
				return (compare != 0) ? compare : (Integer)o2[1] - (Integer)o1[1];	// エフェクトは逆順
			}
		});
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		List<Object[]> data = null;
		if (pm != null) {
			data = Util.newList();
		}

		Set<Effect> newEffects = Util.newSet();
		try {
			if (_dstLayerId != null) {
				EffectableLayer dstLayer = (EffectableLayer) comp.getLayer(_dstLayerId);

				for (Object[] o : _srcData) {
					Effect effect = _decoder.decodeElement((String)o[2], Effect.class);
					effect.afterDecode(project);
				
					dstLayer.getEffects().add(_dstIndex, effect);
					newEffects.add(effect);

					if (pm != null) {
						data.add(new Object[] { dstLayer, _dstIndex + data.size() });
					}
				}

			} else {
				for (Object[] o : _srcData) {
					Effect effect = _decoder.decodeElement((String)o[2], Effect.class);
					effect.afterDecode(project);
					
					EffectableLayer srcLayer = (EffectableLayer) comp.getLayers().get((Integer)o[0]);
					srcLayer.getEffects().add((Integer)o[1]+1, effect);
					newEffects.add(effect);

					if (pm != null) {
						data.add(new Object[] { srcLayer, effect });
					}
				}
				if (pm != null) {
					for (Object[] o : data) {
						o[1] = ((EffectableLayer) o[0]).getEffects().indexOf(o[1]);
					}
				}
			}
		} catch (ProjectDecodeException e) {
			// 元の状態に戻す
			for (Layer layer : comp.getLayers()) {
				if (layer instanceof EffectableLayer) {
					((EffectableLayer) layer).getEffects().removeAll(newEffects);
				}
			}
			throw new ExecutionException("error decoding effect data", e);
		}

		if (pm != null) {
			pm.fireEffectsAdd(comp, data.toArray(new Object[data.size()][]));
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		if (_dstLayerId != null) {
			EffectableLayer dstLayer = (EffectableLayer) comp.getLayer(_dstLayerId);
			dstLayer.getEffects().subList(_dstIndex, _dstIndex + _srcData.size()).clear();

			if (pm != null) {
				pm.fireEffectsRemove(comp, new Object[][] { new Object[] { dstLayer, -1 } });
			}

		} else {
			List<Object[]> data = null;
			if (pm != null) {
				data = Util.newList();
			}

			for (ListIterator<Object[]> it = _srcData.listIterator(_srcData.size()); it.hasPrevious(); ) {
				Object[] o = it.previous();

				EffectableLayer srcLayer = (EffectableLayer) comp.getLayers().get((Integer)o[0]);
				srcLayer.getEffects().remove((Integer)o[1]+1);

				if (pm != null) {
					data.add(new Object[] { srcLayer, -1 });
				}
			}

			if (pm != null) {
				pm.fireEffectsRemove(comp, data.toArray(new Object[data.size()][]));
			}
		}

		return Status.OK_STATUS;
	}

}
