/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;

public class ModifyEffectAnimatableValueOperation<V> extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final int _effectIndex;

	private final String _property;

	private final V _oldValue;

	private volatile V _newValue;


	public ModifyEffectAnimatableValueOperation(
			ProjectManager projectManager, EffectableLayer layer, int effectIndex,
			String property, V value, String relation, String name) {

		super(projectManager, name + "の変更", relation);

		LayerComposition comp = projectManager.checkLayer(layer);
		Effect effect = layer.getEffects().get(effectIndex);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(effect, property);

		_compId = comp.getId();
		_layerId = layer.getId();
		_effectIndex = effectIndex;
		_property = property;
		_oldValue = avalue.getStaticValue();
		_newValue = value;

		_noEffect = (_newValue != null && _newValue.equals(_oldValue)) || (_newValue == null && _oldValue == null);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyProperty(_newValue, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyProperty(_oldValue, project, pm);
		return Status.OK_STATUS;
	}

	private void modifyProperty(V value, Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer layer = (EffectableLayer) comp.getLayer(_layerId);
		Effect effect = layer.getEffects().get(_effectIndex);

		AnimatableValue<V> avalue = PropertyUtil.getProperty(effect, _property);
		if (avalue.hasKeyframe()) {
			throw new IllegalStateException("property has keyframe");
		}
		avalue.clearKeyframes(value);

		if (pm != null) {
			pm.fireEffectPropertyChange(layer, _effectIndex, _property);
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			// _newValue の値は、シャドウプロジェクトに反映される前に
			// 後続のマージ可能オペレーションで上書きされているかもしれない。
			// しかし、シャドウオペレーションのキューが全て実行された時点での値は一致するので問題ない。
			modifyProperty(_newValue, project, null);
			return true;
		}

		if (!(operation instanceof ModifyEffectAnimatableValueOperation<?>)) {
			return false;
		}

		@SuppressWarnings("unchecked")
		ModifyEffectAnimatableValueOperation<V> newOp = (ModifyEffectAnimatableValueOperation<V>) operation;
		if (!newOp._compId.equals(_compId)
				|| !newOp._layerId.equals(_layerId)
				|| newOp._effectIndex != _effectIndex
				|| !newOp._property.equals(_property)) {
			return false;
		}

		V newValue = newOp._newValue;
		if ((_newValue != null && !_newValue.equals(newValue)) || (_newValue == null && newValue != null)) {
			_newValue = newValue;
			modifyProperty(_newValue, project, pm);

			// 結合の結果、このオペレーションの直前の値に戻ったかどうか。
			_noEffect = (_newValue != null && _newValue.equals(_oldValue)) || (_newValue == null && _oldValue == null);
		}

		return true;
	}

}
