/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;

public class ModifyEffectPropertyOperation<T> extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final int _effectIndex;

	private final String _property;

	private final T _oldValue;

	private final T _newValue;


	public ModifyEffectPropertyOperation(
			ProjectManager pm, EffectableLayer layer, Effect effect,
			String property, T newValue, String label) {

		super(pm, label);

		LayerComposition comp = pm.checkLayer(layer);
		_compId = comp.getId();
		_layerId = layer.getId();
		_effectIndex = layer.getEffects().indexOf(effect);
		if (_effectIndex == -1) {
			throw new IllegalArgumentException();
		}
		_property = property;
		_oldValue = PropertyUtil.getProperty(effect, property);
		_newValue = newValue;

		_noEffect = (_newValue == _oldValue) || (_newValue != null && _newValue.equals(_oldValue));
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyProperty(_newValue, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyProperty(_oldValue, project, pm);
		return Status.OK_STATUS;
	}

	private void modifyProperty(T value, Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer layer = (EffectableLayer) comp.getLayer(_layerId);
		Effect effect = layer.getEffects().get(_effectIndex);

		PropertyUtil.setProperty(effect, _property, value);

		if (pm != null) {
			pm.fireEffectPropertyChange(layer, _effectIndex, _property);
		}
	}

}
