/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.SortedMap;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class ModifyLayerKeyframeOperation<V> extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final String _property;

	private final Collection<Keyframe<V>> _oldKeyframes;

	private volatile Keyframe<V> _newKeyframe;


	public ModifyLayerKeyframeOperation(
			ProjectManager projectManager, Layer layer,
			String property, Time time, V value, String relation, String name) {

		super(projectManager, name + "の変更", relation);

		LayerComposition comp = projectManager.checkLayer(layer);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(layer, property);

		Time frameDuration = comp.getFrameDuration();
		Time halfOfFrameDuration = new Time(frameDuration.timeValue/2, frameDuration.timeScale);
		Time t1 = time.subtract(halfOfFrameDuration);
		Time t2 = time.add(frameDuration).subtract(halfOfFrameDuration);
		SortedMap<Time, Keyframe<V>> oldKeyframeMap = avalue.getKeyframeMap().subMap(t1, t2);

		setLabel(avalue.hasKeyframe(), !oldKeyframeMap.isEmpty(), (value != null));

		_compId = comp.getId();
		_layerId = layer.getId();
		_property = property;
		_oldKeyframes = Util.newSet(oldKeyframeMap.values());

		if (value != null) {
			Keyframe<V> kf = oldKeyframeMap.get(time);
			if (kf == null) {
				SortedMap<Time, Keyframe<V>> head = oldKeyframeMap.headMap(time);
				SortedMap<Time, Keyframe<V>> tail = oldKeyframeMap.tailMap(time);
				Keyframe<V> kf1 = head.isEmpty() ? null : head.get(head.lastKey());
				Keyframe<V> kf2 = tail.isEmpty() ? null : tail.get(tail.firstKey());
				if (kf1 != null && kf2 != null) {
					kf = kf2.time.subtract(time).before(time.subtract(kf1.time)) ? kf2 : kf1;
				} else if (kf1 != null) {
					kf = kf1;
				} else if (kf2 != null) {
					kf = kf2;
				}
			}
			_newKeyframe = new Keyframe<V>(time, value, defaultInterpolation(avalue, kf, time));
		} else {
			_newKeyframe = null;
		}
	}

	// TODO ModifyEffectKeyframeOperation にも同じメソッドがあるので整理する。
	private void setLabel(boolean hasKeyframe, boolean oldKeyframe, boolean newKeyframe) {
		if (!oldKeyframe) {
			if (!newKeyframe) {
				throw new IllegalArgumentException();
			} else if (hasKeyframe) {
				setLabel("キーフレームの追加");
			} else {
				setLabel("ストップウォッチの使用");
			}
		} else if (!newKeyframe) {
			setLabel("キーフレームの削除");
		}
	}

	// TODO ModifyEffectKeyframeOperation にも同じメソッドがあるので整理する。
	private Interpolation defaultInterpolation(AnimatableValue<V> avalue, Keyframe<V> oldKeyframe, Time time) {
		if (oldKeyframe != null) {
			return oldKeyframe.interpolation;
		} else {
			SortedMap<Time, Keyframe<V>> head = avalue.getKeyframeMap().headMap(time);
			if (!head.isEmpty()) {
				return head.get(head.lastKey()).interpolation;
			} else {
				return avalue.getDefaultInterpolation();
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replaceKeyframes(_oldKeyframes, Collections.singleton(_newKeyframe), project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replaceKeyframes(Collections.singleton(_newKeyframe), _oldKeyframes, project, pm);
		return Status.OK_STATUS;
	}

	private void replaceKeyframes(
			Collection<Keyframe<V>> target, Collection<Keyframe<V>> replacement,
			Project project, ProjectManager pm) {

		LayerComposition comp = project.getComposition(_compId);
		Layer layer = comp.getLayer(_layerId);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(layer, _property);

		Map<Time, Object[]> data = null;
		if (pm != null) {
			data = Util.newMap();
		}

		for (Keyframe<V> kf : target) {
			avalue.removeKeyframe(kf.time);
			if (pm != null) {
				data.put(kf.time, new Object[] { layer, -1, _property, kf.time });
			}
		}

		for (Keyframe<V> kf : replacement) {
			avalue.putKeyframe(kf);
			if (pm != null) {
				data.put(kf.time, new Object[] { layer, -1, _property, kf.time });
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data.values().toArray(new Object[data.size()][]));
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			// _newKeyframe の値は、シャドウプロジェクトに反映される前に
			// 後続のマージ可能オペレーションで上書きされているかもしれない。
			// しかし、シャドウオペレーションのキューが全て実行された時点での値は一致するので問題ない。
			replaceKeyframes(Collections.<Keyframe<V>>emptySet(), Collections.singleton(_newKeyframe), project, pm);
			return true;
		}

		if (!(operation instanceof ModifyLayerKeyframeOperation<?>)) {
			return false;
		}

		@SuppressWarnings("unchecked")
		ModifyLayerKeyframeOperation<V> newOp = (ModifyLayerKeyframeOperation<V>) operation;
		if (!newOp._compId.equals(_compId)
				|| !newOp._layerId.equals(_layerId)
				|| !newOp._property.equals(_property)) {
			return false;
		}

		Keyframe<V> newKeyframe = newOp._newKeyframe;

		if (_newKeyframe == null || newKeyframe == null) {
			return false;
		}

		if (!_newKeyframe.time.equals(newKeyframe.time)
				|| !_newKeyframe.interpolation.equals(newKeyframe.interpolation)) {
			return false;
		}

		if (!_newKeyframe.value.equals(newKeyframe.value)) {
			_newKeyframe = newKeyframe;
			replaceKeyframes(Collections.<Keyframe<V>>emptySet(), Collections.singleton(_newKeyframe), project, pm);

			// 結合の結果、このオペレーションの直前の値に戻ったかどうか。
			boolean noEffect = false;
			if (_oldKeyframes.size() == 1) {
				Keyframe<V> oldKeyframe = _oldKeyframes.iterator().next();
				noEffect = (oldKeyframe.time.equals(_newKeyframe.time) && oldKeyframe.value.equals(_newKeyframe.value));
			}
			_noEffect = noEffect;
		}

		return true;
	}

}
