/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.operations.IOperationHistory;
import org.eclipse.core.commands.operations.IOperationHistoryListener;
import org.eclipse.core.commands.operations.IUndoContext;
import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.commands.operations.OperationHistoryEvent;
import org.eclipse.core.commands.operations.UndoContext;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.player.PlayerLock;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ProjectManager implements IOperationHistoryListener {

	private static final Logger _logger = LoggerFactory.getLogger(ProjectManager.class);

	private static final String PROJECT_MANAGER = "PROJECT_MANAGER";


	public static ProjectManager forWorkbenchWindow(IWorkbenchWindow window) {
		return (ProjectManager) window.getShell().getData(PROJECT_MANAGER);
	}

	public static ProjectManager newProject(IWorkbenchWindow window) {
		ProjectManager pm = forWorkbenchWindow(window);
		if (pm != null) {
			throw new IllegalStateException("another project is opened on the window");
		}

		return new ProjectManager(null, null, window);
	}

	public static ProjectManager openProject(File file, IWorkbenchWindow window) throws IOException, ProjectDecodeException {
		ProjectManager pm = forWorkbenchWindow(window);
		if (pm != null) {
			throw new IllegalStateException("another project is opened on the window");
		}

		for (IWorkbenchWindow w : PlatformUI.getWorkbench().getWorkbenchWindows()) {
			ProjectManager pm2 = forWorkbenchWindow(w);
			if (pm2 != null && file.equals(pm2.getFile())) {
				// TODO 例外を投げずにウインドウ w をアクティブにする。そして null を返す。
				throw new IllegalStateException("the project is already opened");
			}
		}

		return new ProjectManager(loadProject(file, window), file, window);
	}

	private static Project loadProject(final File file, IWorkbenchWindow window) throws IOException, ProjectDecodeException {
		BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8"));
		try {
			Map<String, String> headerMap = Util.newLinkedHashMap();
			for (;;) {
				String header = br.readLine();
				if (header == null || (header = header.trim()).length() == 0) {
					break;
				}
				String[] split = header.split("\\s*:\\s*", 2);
				if (split.length == 2) {
					headerMap.put(split[0].toLowerCase(), split[1]);
				}
			}

			String jvpVersion = headerMap.get("jvp-version");
			if (!"0.9".equals(jvpVersion)) {
				throw new ProjectDecodeException("only jvpVersion 0.9 is supported: jvpVersion=" + jvpVersion);
			}

			final StringBuilder body = new StringBuilder();
			String line;
			while ((line = br.readLine()) != null) {
				body.append(line).append("\n");
			}

			final Project[] project = new Project[1];
			IRunnableWithProgress runnable = new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					try {
						monitor.beginTask("プロジェクトを読み込み中...", IProgressMonitor.UNKNOWN);
						ProjectDecoder decoder = InjectorHolder.getInjector().getInstance(ProjectDecoder.class);
						project[0] = decoder.decode(body.toString(), file.getParentFile());
						monitor.done();
					} catch (ProjectDecodeException e) {
						throw new InvocationTargetException(e);
					} catch (IOException e) {
						throw new InvocationTargetException(e);
					}
				}
			};

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(window.getShell()) {
				protected int getShellStyle() {
					return super.getShellStyle() | SWT.SHEET;
				}
			};
			try {
				dialog.run(true, false, runnable);
			} catch (InvocationTargetException e) {
				throw new ProjectDecodeException(e.getCause());
			} catch (InterruptedException e) {
				throw new ProjectDecodeException(e);
			}

			return project[0];

		} finally {
			br.close();
		}
	}


	public static final String DEFAULT_NAME = "名称未設定プロジェクト.jvp";

	private IWorkbenchWindow _window;

	private Project _project;

	private Project _shadow;

	private final ShadowOperationRunner shadowOperationRunner = new ShadowOperationRunner(this);

	private File _file;

	private IUndoContext _undoContext = new UndoContext();

	private IUndoableOperation _firstOperation;

	private boolean _dirty;

	private boolean _historyBusy;

	private final ListenerList _projectListeners = new ListenerList();

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;

	@Inject
	private ProjectElementFactory _elementFactory;


	private ProjectManager(Project project, File file, IWorkbenchWindow window) {
		super();
		InjectorHolder.getInjector().injectMembers(this);

		_window = window;

		if (project != null) {
			_project = project;
			_file = file;
		} else {
			_project = _elementFactory.newProject();
		}

		try {
			_shadow = _decoder.decodeElement(_encoder.encodeElement(_project), Project.class);
			_shadow.afterDecode();
		} catch (ProjectDecodeException e) {
			// たった今読み込んだばかりのプロジェクトや作ったばかりのプロジェクトのエンコード／デコードには失敗しないはず。
			throw new JavieRuntimeException(e);
		}

		IOperationHistory history = getOperationHistory();
		history.setLimit(_undoContext, 10000);		// TODO 設定でUndo/Redoの上限回数を指定できるようにする。
		history.addOperationHistoryListener(this);

		window.getShell().setData(PROJECT_MANAGER, this);

		ProjectEventHub.getHub().listen(this);
		fireProjectEvent(ProjectEvent.createProjectInitializeEvent(this));

		shadowOperationRunner.start();
	}

	protected void finalize() throws Throwable {
		// 何らかの理由でdisposeメソッドが呼ばれなかった場合のため。
		// ShadowOperationRunner#endは複数回呼び出しても問題ない。
		shadowOperationRunner.end();

		if (_project != null) {
			_logger.error("finalizing ProjectManager, but not disposed.");
		}

		super.finalize();
	}

	public void dispose() {
		shadowOperationRunner.end();

		IOperationHistory operationHistory = getOperationHistory();
		operationHistory.removeOperationHistoryListener(this);

		if (_undoContext != null) {
			operationHistory.dispose(_undoContext, true, true, true);
			_undoContext = null;
		}

		if (_shadow != null) {
			for (Item i : _shadow.getItems()) {
				i.dispose();
			}
			_shadow = null;
		}
		if (_project != null) {
			for (Item i : _project.getItems()) {
				i.dispose();
			}
			_project = null;
		}

		fireProjectEvent(ProjectEvent.createProjectDisposeEvent(this));

		_window.getShell().setData(PROJECT_MANAGER, null);
	}

	public IWorkbenchWindow getWorkbenchWindow() {
		return _window;
	}

	public Project getProject() {
		return _project;
	}

	public Project getShadow() {
		return _shadow;
	}

	public File getFile() {
		return _file;
	}

	public IUndoContext getUndoContext() {
		return _undoContext;
	}

	public void addProjectListener(ProjectListener listener) {
		_projectListeners.add(listener);
	}

	public void removeProjectListener(ProjectListener listener) {
		_projectListeners.remove(listener);
	}

	private void fireProjectEvent(ProjectEvent event) {
		for (Object l : _projectListeners.getListeners()) {
			((ProjectListener) l).handleProjectEvent(event);
		}
	}

	void fireItemNameChange(Item item) {
		fireProjectEvent(ProjectEvent.createItemNameChangeEvent(this, item));
	}

	void fireItemUpdate(Item item) {
		fireProjectEvent(ProjectEvent.createItemUpdateEvent(this, item));
	}

	void fireItemsAdd(Collection<? extends Item> items, Collection<? extends Item> relatedItems) {
		fireProjectEvent(ProjectEvent.createItemsAddEvent(this, items, relatedItems));
	}

	void fireItemsRemove(Collection<? extends Item> items, Collection<? extends Item> relatedItems) {
		fireProjectEvent(ProjectEvent.createItemsRemoveEvent(this, items, relatedItems));
	}

	void fireItemsReparent(Collection<? extends Item> items, Collection<? extends Item> relatedItems) {
		fireProjectEvent(ProjectEvent.createItemsReparentEvent(this, items, relatedItems));
	}

	void fireCompositionSettingsChange(Composition comp) {
		fireProjectEvent(ProjectEvent.createCompositionSettingsChangeEvent(this, comp));
	}

	void fireCompositionPropertyChange(Composition comp, String property) {
		fireProjectEvent(ProjectEvent.createCompositionPropertyChangeEvent(this, comp, property));
	}

	void fireLayerPropertyChange(Layer layer, String property) {
		fireProjectEvent(ProjectEvent.createLayerPropertyChangeEvent(this, layer, property));
	}

	void fireLayerItemChange(ItemLayer<?> layer) {
		fireProjectEvent(ProjectEvent.createLayerItemChangeEvent(this, layer));
	}

	void fireLayersAdd(LayerComposition comp, Collection<? extends Layer> layers) {
		fireProjectEvent(ProjectEvent.createLayersAddEvent(this, comp, layers));
	}

	void fireLayersRemove(LayerComposition comp, Collection<? extends Layer> layers) {
		fireProjectEvent(ProjectEvent.createLayersRemoveEvent(this, comp, layers));
	}

	void fireLayersReorder(LayerComposition comp, Collection<? extends Layer> layers) {
		fireProjectEvent(ProjectEvent.createLayersReorderEvent(this, comp, layers));
	}

	void fireTextAnimatorsAdd(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createTextAnimatorsAddEvent(this, comp, data));
	}

	void fireTextAnimatorsRemove(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createTextAnimatorsRemoveEvent(this, comp, data));
	}

	void fireTASelectorsAdd(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createTASelectorsAddEvent(this, comp, data));
	}

	void fireTASelectorsRemove(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createTASelectorsRemoveEvent(this, comp, data));
	}

	void fireTAPropertiesAdd(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createTAPropertiesAddEvent(this, comp, data));
	}

	void fireTAPropertiesRemove(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createTAPropertiesRemoveEvent(this, comp, data));
	}

	void fireEffectPropertyChange(EffectableLayer layer, int effectIndex, String property) {
		fireProjectEvent(ProjectEvent.createEffectPropertyChangeEvent(this, layer, effectIndex, property));
	}

	void fireEffectsAdd(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createEffectsAddEvent(this, comp, data));
	}

	void fireEffectsRemove(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createEffectsRemoveEvent(this, comp, data));
	}

	void fireLayerExpressionChange(Layer layer, String property) {
		fireProjectEvent(ProjectEvent.createLayerExpressionChangeEvent(this, layer, property));
	}

	void fireEffectExpressionChange(EffectableLayer layer, int effectIndex, String property) {
		fireProjectEvent(ProjectEvent.createEffectExpressionChangeEvent(this, layer, effectIndex, property));
	}

	void fireExpressionsAdd(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createExpressionsAddEvent(this, comp, data));
	}

	void fireExpressionsRemove(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createExpressionsRemoveEvent(this, comp, data));
	}

	void fireLayerTimesChange(LayerComposition comp, Collection<? extends Layer> layers) {
		fireProjectEvent(ProjectEvent.createLayerTimesChangeEvent(this, comp, layers));
	}

	void fireKeyframesChange(LayerComposition comp, Object[][] data) {
		fireProjectEvent(ProjectEvent.createKeyframesChangeEvent(this, comp, data));
	}

	void fireLayerSlipEdit(LayerComposition comp, Object[] data) {
		fireProjectEvent(ProjectEvent.createLayerSlipEditEvent(this, comp, data));
	}

	void fireShadowOperationExecution() {
		fireProjectEvent(ProjectEvent.createShadowOperationExecutionEvent(this));
	}

	public void undo() {
		if (!_historyBusy) {
			_historyBusy = true;
			try {
				getOperationHistory().undo(_undoContext, null, null);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			} finally {
				_historyBusy = false;
			}
		} else {
			_logger.warn("the operation history is busy.");
		}
	}

	public void redo() {
		if (!_historyBusy) {
			_historyBusy = true;
			try {
				getOperationHistory().redo(_undoContext, null, null);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			} finally {
				_historyBusy = false;
			}
		} else {
			_logger.warn("the operation history is busy.");
		}
	}

	public boolean canUndo() {
		return getOperationHistory().canUndo(_undoContext);
	}

	public boolean canRedo() {
		return getOperationHistory().canRedo(_undoContext);
	}

	public String getUndoLabel() {
		IUndoableOperation op = getOperationHistory().getUndoOperation(_undoContext);
		return (op != null) ? op.getLabel() : null;
	}

	public String getRedoLabel() {
		IUndoableOperation op = getOperationHistory().getRedoOperation(_undoContext);
		return (op != null) ? op.getLabel() : null;
	}

	public void historyNotification(OperationHistoryEvent event) {
		IUndoableOperation op = event.getOperation();
		if (!op.hasContext(_undoContext)) {
			return;
		}

		switch (event.getEventType()) {
			case OperationHistoryEvent.OPERATION_ADDED:
				if (!_dirty) {
					_firstOperation = event.getOperation();
				}
				// fall through
			case OperationHistoryEvent.REDONE:
			case OperationHistoryEvent.UNDONE:
				updateDirty();
				break;
		}
	}

	private IOperationHistory getOperationHistory() {
		return PlatformUI.getWorkbench().getOperationSupport().getOperationHistory();
	}

	public void postOperation(ProjectOperation operation) {
		if (!operation.projectManagerMatches(this)) {
			throw new IllegalArgumentException("different ProjectManager");
		}

		if (operation.isNoEffect()) {
			return;
		}

		do {
			if (operation.getRelation() == null) {
				break;
			}

			if (!_dirty) {
				// ドラッグジェスチャーでプロパティを更新しながらCmd+Sして保存した場合など。
				// この場合は新規のオペレーションとして扱わないと矛盾した状態になってしまう。
				break;
			}

			IOperationHistory history = getOperationHistory();
			if (history.canRedo(_undoContext)) {
				// Redo可能なオペレーションがあるなら、Undoの先頭にあるオペレーションは完結しているということなので結合不可。
				// そして、その場合はrelationの値が一致する状況は通常なら起きないが、
				// ドラッグジェスチャーでプロパティを更新しながらCmd+Zすると起きる。他にも起きるケースがあるかもしれない。
				break;
			}

			IUndoableOperation undoOperation = history.getUndoOperation(_undoContext);
			if (!(undoOperation instanceof ProjectOperation)) {
				break;
			}

			ProjectOperation prevOp = (ProjectOperation) undoOperation;
			if (!operation.getRelation().equals(prevOp.getRelation())) {
				break;
			}

			if (!_historyBusy) {
				_historyBusy = true;
				try {
					if (!prevOp.tryMerging(operation, null, null)) {
						break;
					}

					history.operationChanged(prevOp);

					// オペレーションを結合した結果、そのオペレーションの直前の値に戻った場合はそのオペレーション自体は何も為さないものになる。
					// その場合、ヒストリーに残しておくと紛らわしいので削除する。
					// TODO ヒストリーから削除するメソッドが無いようなので replaceOperation で代用したが、それで問題無いかどうか十分に確認していない。
					if (prevOp.isNoEffect()) {
						history.replaceOperation(prevOp, new IUndoableOperation[0]);

						if (_firstOperation == prevOp) {
							clearDirty();
						}
					}
				} finally {
					_historyBusy = false;
				}
			} else {
				_logger.warn("the operation history is busy.");
			}
			return;
		} while (false);


		if (!_historyBusy) {
			_historyBusy = true;
			try {
				operation.addContext(_undoContext);
				getOperationHistory().execute(operation, null, null);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			} finally {
				_historyBusy = false;
			}
		} else {
			_logger.warn("the operation history is busy.");
		}
	}

	void postShadowOperation(boolean sync, final Runnable shadowOperation) {
		if (sync) {
			final Object monitor = new Object();
			Runnable wrapper = new Runnable() {
				public void run() {
					shadowOperation.run();
					synchronized (monitor) {
						monitor.notify();
					}
				}
			};
			synchronized (monitor) {
				shadowOperationRunner.put(wrapper);
				try {
					monitor.wait();
				} catch (InterruptedException e) {
					throw new JavieRuntimeException("unexpected interruption", e);
				}
			}
		} else {
			shadowOperationRunner.put(shadowOperation);
		}
	}

	public boolean isDirty() {
		return _dirty;
	}

	private void updateDirty() {
		IUndoableOperation nextRedo = getOperationHistory().getRedoOperation(_undoContext);
		boolean newDirty = (nextRedo != _firstOperation);
		if (_dirty != newDirty) {
			_dirty = newDirty;
			fireProjectEvent(ProjectEvent.createDirtyChangeEvent(this));
		}
	}

	private void clearDirty() {
		_firstOperation = getOperationHistory().getRedoOperation(_undoContext);
		updateDirty();
	}

	public void save() throws IOException {
		if (_file == null) {
			throw new IllegalStateException("no file is specified");
		}
		save(_file);
		clearDirty();
	}

	public void saveAs(File file) throws IOException {
		save(file);

		if (!file.equals(_file)) {
			_file = file;
			fireProjectEvent(ProjectEvent.createFileChangeEvent(this));
		}

		clearDirty();
	}

	public void saveCopy(File file) throws IOException {
		// TODO saveCopyの場合はプロジェクト内の全idを採番しなおして保存すべき？
		save(file);
	}

	private void save(File file) throws IOException {
		String body = _encoder.encode(_project, file.getParentFile(), true);

		PrintWriter pw = new PrintWriter(new BufferedWriter(
				new OutputStreamWriter(new FileOutputStream(file), "UTF-8")));
		try {
			// JSONICの prettyPrint を有効にしたとき、
			// 改行コードがプラットフォーム固有のものではなくLFのみになるようなので、
			// 以下では println ではなく print('\n') としている。
			pw.print("JVP-Version: 0.9\n\n");
			pw.print(body);
			pw.print('\n');
		} finally {
			pw.close();
		}
	}

	public String getUnusedItemName(String prefix) {
		Set<String> names = Util.newSet();
		for (Item i : _project.getItems()) {
			names.add(i.getName());
		}

		int n = 1;
		String name;
		while (true) {
			name = prefix + " " + (n++);
			if (!names.contains(name)) {
				return name;
			}
		}
	}

	public String getUnusedLayerName(LayerComposition comp, String prefix) {
		checkComposition(comp);

		Set<String> names = Util.newSet();
		for (Layer l : comp.getLayers()) {
			names.add(l.getName());
		}

		int n = 1;
		String name;
		while (true) {
			name = prefix + " " + (n++);
			if (!names.contains(name)) {
				return name;
			}
		}
	}

	public List<Item> listChildren(Folder parent) {
		List<Item> list = Util.newList();

		for (Item i : _project.getItems()) {
			if (i.getParent() == parent) {
				list.add(i);
			}
		}

		return list;
	}

	public boolean hasChildren(Folder parent) {
		for (Item i : _project.getItems()) {
			if (i.getParent() == parent) {
				return true;
			}
		}

		return false;
	}

	void checkItem(Item item) {
		if (item != null && _project.getItem(item.getId()) != item) {
			throw new IllegalArgumentException("no such item found: " + item.getId());
		}
	}

	void checkComposition(Composition comp) {
		if (comp != null && _project.getComposition(comp.getId()) != comp) {
			throw new IllegalArgumentException("no such composition found: " + comp.getId());
		}
	}

	LayerComposition checkLayer(Layer layer) {
		if (layer == null) {
			return null;
		}

		for (Composition comp : _project.getCompositions()) {
			if (comp instanceof LayerComposition) {
				LayerComposition layerComp = (LayerComposition) comp;
				if (layerComp.getLayer(layer.getId()) == layer) {
					return layerComp;
				}
			}
		}
		throw new IllegalArgumentException("no such layer found: " + layer.getId());
	}

}

class ShadowOperationRunner extends Thread {

	private static final Runnable END = new Runnable() { public void run() { } };

	private static final Logger logger = LoggerFactory.getLogger(ShadowOperationRunner.class);

	private final BlockingQueue<Runnable> shadowOperations = new LinkedBlockingQueue<Runnable>();

	private final ProjectManager projectManager;


	ShadowOperationRunner(ProjectManager pm) {
		super("ShadowOperationRunner");
		projectManager = pm;
	}

	void put(Runnable operation) {
		try {
			shadowOperations.put(operation);
		} catch (InterruptedException e) {
			throw new JavieRuntimeException(e);
		}
	}

	void end() {
		put(END);
	}

	@Override
	public void run() {
		List<Runnable> drain = Util.newList();
		for (;;) {
			drain.clear();

			try {
				drain.add(shadowOperations.take());
			} catch (InterruptedException e) {
				logger.warn("unexpected interrupt", e);
				continue;
			}

			shadowOperations.drainTo(drain);

			PlayerLock.writeLock().lock();
			try {
				for (Runnable op : drain) {
					if (op == END) {
						return;
					}
					op.run();
				}
			} catch (Throwable e) {
				logger.error("error running shadow operation", e);
				// シャドウオペレーション実行中の例外発生は回復不可能なため、このスレッドはただちに終了する。
				// TODO エラーダイアログなどを表示すべき。
				return;

			} finally {
				PlayerLock.writeLock().unlock();
			}

			projectManager.fireShadowOperationExecution();
		}
	}

}
