/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;

public class ShiftKeyframesOperation extends ProjectOperation {

	private static class Entry<V> {
		final String layerId;
		final int effectIndex;
		final String property;
		final Set<Keyframe<V>> keyframesToShift;
		final Set<Keyframe<V>> restOfKeyframes;

		Entry(Layer layer, Object[] data) {
			layerId = layer.getId();
			effectIndex = (Integer) data[1];
			property = (String) data[2];

			@SuppressWarnings("unchecked")
			Set<Keyframe<V>> tmp = (Set<Keyframe<V>>) data[3];
			keyframesToShift = tmp;

			AnimatableValue<V> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				PropertyDescriptor pd = effect.getEffectDescriptor().getPropertyDescriptor(property);
				@SuppressWarnings("unchecked")
				AnimatableValue<V> tmp2 = (AnimatableValue<V>) pd.get(effect);
				avalue = tmp2;
			}
			restOfKeyframes = Util.<Keyframe<V>>newSet(avalue.getKeyframeMap().values());
			restOfKeyframes.removeAll(keyframesToShift);
		}
	}

	private final String _compId;

	private final List<Entry<?>> _entries;

	private volatile Time _deltaTime;


	public ShiftKeyframesOperation(
			ProjectManager projectManager, LayerComposition comp,
			Object[][] keyframeData, Time deltaTime, String relation) {

		super(projectManager, "キーフレームの移動", relation);

		projectManager.checkComposition(comp);
		_compId = comp.getId();
		_deltaTime = deltaTime;

		List<Entry<?>> entries = null;

		if (keyframeData != null) {
			if (_deltaTime.timeValue != 0) {
				entries = Util.newList();
				for (Object[] data : keyframeData) {
					Layer layer = (Layer) data[0];
					if (projectManager.checkLayer(layer) != comp) {
						throw new IllegalArgumentException();
					}
					@SuppressWarnings("unchecked")
					Entry<?> entry = new Entry(layer, data);
					entries.add(entry);
				}
			} else {
				_noEffect = true;
			}
		}

		_entries = entries;
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		shiftKeyframesOrRestore(true, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		shiftKeyframesOrRestore(false, project, pm);
		return Status.OK_STATUS;
	}

	private void shiftKeyframesOrRestore(boolean shift, Project project, ProjectManager pm) throws ExecutionException {
		if (_entries == null) {
			throw new IllegalStateException();
		}

		LayerComposition comp = project.getComposition(_compId);
		boolean fire = (pm != null);
		List<Object[]> data = fire ? Util.<Object[]>newList() : null;

		for (Entry<?> e : _entries) {
			Layer layer = comp.getLayer(e.layerId);

			AnimatableValue<Object> avalue;
			if (e.effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, e.property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(e.effectIndex);
				PropertyDescriptor pd = effect.getEffectDescriptor().getPropertyDescriptor(e.property);
				@SuppressWarnings("unchecked")
				AnimatableValue<Object> tmp = (AnimatableValue<Object>) pd.get(effect);
				avalue = tmp;
			}

			@SuppressWarnings("unchecked")
			Entry<Object> eo = (Entry<Object>) e;
			List<Time> times;
			if (shift) {
				times = shiftKeyframesOrRestore(true, avalue, eo, fire);
			} else {
				times = shiftKeyframesOrRestore(false, avalue, eo, fire);
			}

			if (fire) {
				data.add(new Object[] { layer, e.effectIndex, e.property, times });
			}
		}

		if (fire) {
			pm.fireKeyframesChange(comp, data.toArray(new Object[data.size()][]));
		}
	}

	private <V> List<Time> shiftKeyframesOrRestore(boolean shift, AnimatableValue<V> avalue, Entry<V> entry, boolean returnTimes) {
		avalue.clearKeyframes(avalue.getStaticValue());
		for (Keyframe<V> kf : entry.restOfKeyframes) {
			avalue.putKeyframe(kf);
		}

		List<Time> times;
		if (returnTimes) {
			times = new ArrayList<Time>(entry.keyframesToShift.size());
		} else {
			@SuppressWarnings("serial")
			List<Time> tmp = new ArrayList<Time>() { public boolean add(Time o) { return false; } };
			times = tmp;
		}

		if (shift) {
			for (Keyframe<V> kf : entry.keyframesToShift) {
				Time time = _deltaTime.add(kf.time);
				avalue.putKeyframe(new Keyframe<V>(time, kf.value, kf.interpolation));
				times.add(time);
			}
		} else {
			for (Keyframe<V> kf : entry.keyframesToShift) {
				avalue.putKeyframe(kf);
				times.add(kf.time);
			}
		}

		return returnTimes ? times : null;
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			// _deltaTime は、シャドウプロジェクトに反映される前に
			// 後続のマージ可能オペレーションで上書きされているかもしれない。
			// しかし、シャドウオペレーションのキューが全て実行された時点での値は一致するので問題ない。
			try {
				executeOrRedo(monitor, info, project, null);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}
			return true;
		}

		ShiftKeyframesOperation newOp = (ShiftKeyframesOperation) operation;

		if (newOp._entries != null) {
			throw new IllegalStateException();
		}

		if (!newOp._deltaTime.equals(_deltaTime)) {
			_deltaTime = newOp._deltaTime;

			try {
				executeOrRedo(monitor, info, project, pm);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}

			// 結合の結果、このオペレーションの直前の値に戻ったかどうか。
			_noEffect = (_deltaTime.timeValue == 0);
		}

		return true;
	}

}
