/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;

public class ShiftLayerTimesOperation extends ProjectOperation {

	private static class Entry {
		final String layerId;
		final Time startTime;
		final Time inPoint;
		final Time outPoint;
		final List<Set<Keyframe<?>>> keyframes = Util.newList();

		Entry(Layer layer) {
			layerId = layer.getId();
			startTime = layer.getStartTime();
			inPoint = layer.getInPoint();
			outPoint = layer.getOutPoint();

			for (String property : PropertyUtil.getLayerPropertyNames(layer)) {
				Object o = PropertyUtil.getProperty(layer, property);
				if (o instanceof AnimatableValue<?>) {
					AnimatableValue<?> avalue = (AnimatableValue<?>) o;
					keyframes.add(Util.<Keyframe<?>>newSet(avalue.getKeyframeMap().values()));
				}
			}

			if (layer instanceof EffectableLayer) {
				for (Effect effect : ((EffectableLayer) layer).getEffects()) {
					for (PropertyDescriptor pd : effect.getEffectDescriptor().getPropertyDescriptors()) {
						Object o = pd.get(effect);
						if (o instanceof AnimatableValue<?>) {
							AnimatableValue<?> avalue = (AnimatableValue<?>) o;
							keyframes.add(Util.<Keyframe<?>>newSet(avalue.getKeyframeMap().values()));
						}
					}
				}
			}
		}
	}


	private final String _compId;

	private final List<Entry> _entries = Util.newList();

	private volatile Time _deltaTime;


	public ShiftLayerTimesOperation(
			ProjectManager projectManager, LayerComposition comp,
			List<Layer> layers, Time deltaTime, String relation) {

		super(projectManager, "レイヤー時間の変更", relation);

		projectManager.checkComposition(comp);
		_compId = comp.getId();
		_deltaTime = deltaTime;

		for (Layer layer : layers) {
			if (projectManager.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}
			_entries.add(new Entry(layer));
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		shiftOrRestore(true, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		shiftOrRestore(false, project, pm);
		return Status.OK_STATUS;
	}

	private void shiftOrRestore(boolean shift, Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);
		List<Layer> layers = Util.newList();

		for (Entry e : _entries) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setStartTime(shift ? _deltaTime.add(e.startTime) : e.startTime);
			layer.setInPoint(shift ? _deltaTime.add(e.inPoint) : e.inPoint);
			layer.setOutPoint(shift ? _deltaTime.add(e.outPoint) : e.outPoint);

			Iterator<?> it = e.keyframes.iterator();

			for (String property : PropertyUtil.getLayerPropertyNames(layer)) {
				Object o = PropertyUtil.getProperty(layer, property);
				if (o instanceof AnimatableValue<?>) {
					@SuppressWarnings("unchecked")
					AnimatableValue<Object> avalue = (AnimatableValue<Object>) o;
					@SuppressWarnings("unchecked")
					Set<Keyframe<Object>> origKeyframes = (Set<Keyframe<Object>>) it.next();
					if (shift) {
						shiftKeyframes(avalue, origKeyframes);
					} else {
						restore(avalue, origKeyframes);
					}
				}
			}

			if (layer instanceof EffectableLayer) {
				for (Effect effect : ((EffectableLayer) layer).getEffects()) {
					for (PropertyDescriptor pd : effect.getEffectDescriptor().getPropertyDescriptors()) {
						Object o = pd.get(effect);
						if (o instanceof AnimatableValue<?>) {
							@SuppressWarnings("unchecked")
							AnimatableValue<Object> avalue = (AnimatableValue<Object>) o;
							@SuppressWarnings("unchecked")
							Set<Keyframe<Object>> origKeyframes = (Set<Keyframe<Object>>) it.next();
							if (shift) {
								shiftKeyframes(avalue, origKeyframes);
							} else {
								restore(avalue, origKeyframes);
							}
						}
					}
				}
			}

			layers.add(layer);
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}
	}

	private <V> void shiftKeyframes(AnimatableValue<V> avalue, Set<Keyframe<V>> origKeyframes) {
		avalue.clearKeyframes(avalue.getStaticValue());
		for (Keyframe<V> kf : origKeyframes) {
			avalue.putKeyframe(new Keyframe<V>(_deltaTime.add(kf.time), kf.value, kf.interpolation));
		}
	}

	private <V> void restore(AnimatableValue<V> avalue, Set<Keyframe<V>> origKeyframes) {
		avalue.clearKeyframes(avalue.getStaticValue());
		for (Keyframe<V> kf : origKeyframes) {
			avalue.putKeyframe(kf);
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			try {
				executeOrRedo(monitor, info, project, null);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}
			return true;
		}

		ShiftLayerTimesOperation newOp = (ShiftLayerTimesOperation) operation;

		if (!newOp._deltaTime.equals(_deltaTime)) {
			_deltaTime = newOp._deltaTime;

			try {
				executeOrRedo(monitor, info, project, pm);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}

			_noEffect = (_deltaTime.timeValue == 0);
		}

		return true;
	}

}
