/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class TimeStretchLayerOperation extends ProjectOperation {

	private static class LayerEntry {
		private final String layerId;
		private final Time startTime;
		private final Time inPoint;
		private final Time outPoint;
		private final double rate;

		private LayerEntry(Layer layer) {
			layerId = layer.getId();
			startTime = layer.getStartTime();
			inPoint = layer.getInPoint();
			outPoint = layer.getOutPoint();
			rate = layer.getRate();
		}
	}

	private static class PropertyEntry {
		private final String layerId;
		private final int effectIndex;
		private final String property;
		private final String data;

		public PropertyEntry(String layerId, int effectIndex, String property, String data) {
			this.layerId = layerId;
			this.effectIndex = effectIndex;
			this.property = property;
			this.data = data;
		}
	}


	private final double _newRate;

	private final Object _timeToHold;

	private final String _compId;

	private final Set<LayerEntry> _layers = Util.newSet();

	private final Set<PropertyEntry> _properties = Util.newSet();

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	private TimeStretchLayerOperation(ProjectManager pm,
			List<Layer> layers, double newRate, Object timeToHold) {

		super(pm, "時間伸縮");
		InjectorHolder.getInjector().injectMembers(this);

		if (!(timeToHold instanceof Time || timeToHold instanceof Boolean)) {
			throw new IllegalArgumentException();
		}

		LayerComposition comp = pm.checkLayer(layers.iterator().next());

		_newRate = newRate;
		_timeToHold = timeToHold;
		_compId = comp.getId();

		for (Layer layer : layers) {
			if (pm.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}

			if (layer.getRate() == newRate) {
				continue;
			}

			_layers.add(new LayerEntry(layer));

			for (String property : PropertyUtil.getLayerPropertyNames(layer)) {
				Object o = PropertyUtil.getProperty(layer, property);
				if (o instanceof AnimatableValue<?>) {
					AnimatableValue<?> avalue = (AnimatableValue<?>) o;
					if (avalue.hasKeyframe()) {
						_properties.add(new PropertyEntry(layer.getId(), -1, property, _encoder.encodeElement(avalue)));
					}
				}
			}

			if (layer instanceof EffectableLayer) {
				List<Effect> effects = ((EffectableLayer) layer).getEffects();
				for (int i = 0; i < effects.size(); ++i) {
					Effect e = effects.get(i);
					for (PropertyDescriptor pd : e.getEffectDescriptor().getPropertyDescriptors()) {
						Object o = pd.get(e);
						if (o instanceof AnimatableValue<?>) {
							AnimatableValue<?> avalue = (AnimatableValue<?>) o;
							if (avalue.hasKeyframe()) {
								_properties.add(new PropertyEntry(layer.getId(), i, pd.getName(), _encoder.encodeElement(avalue)));
							}
						}
					}
				}
			}
		}

		_noEffect = _layers.isEmpty();
	}

	public TimeStretchLayerOperation(ProjectManager pm,
			List<Layer> layers, double newRate, Time timeToHold) {

		this(pm, layers, newRate, (Object)timeToHold);
	}

	public TimeStretchLayerOperation(ProjectManager pm,
			List<Layer> layers, double newRate, boolean holdInPoint) {

		this(pm, layers, newRate, (Object)holdInPoint);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		stretchKeyframes(project, pm);
		stretchLayers(project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		restoreKeyframes(project, pm);
		restoreLayers(project, pm);
		return Status.OK_STATUS;
	}

	private Time timeToHold(Layer layer) {
		if (_timeToHold instanceof Time) {
			return (Time) _timeToHold;
		} else if (Boolean.TRUE.equals(_timeToHold)) {
			return layer.getInPoint();
		} else {
			return layer.getOutPoint();
		}
	}

	private Time stretchTime(Time oldTime, double oldRate, Time timeToHold) {
		Time sub = oldTime.subtract(timeToHold);
		return new Time((long)(sub.timeValue*oldRate/_newRate), sub.timeScale).add(timeToHold);
	}

	private void stretchKeyframes(Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		List<Object[]> data = (pm != null) ? Util.<Object[]>newList() : null;

		for (PropertyEntry e : _properties) {
			Layer layer = comp.getLayer(e.layerId);
			AnimatableValue<Object> avalue;
			if (e.effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, e.property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(e.effectIndex);
				avalue = PropertyUtil.getProperty(effect, e.property);
			}

			double oldRate = layer.getRate();
			Time timeToHold = timeToHold(layer);

			List<Keyframe<Object>> keyframes = Util.newList(avalue.getKeyframeMap().values());
			avalue.clearKeyframes(avalue.getStaticValue());
			for (Keyframe<Object> kf : keyframes) {
				Time time = stretchTime(kf.time, oldRate, timeToHold);
				avalue.putKeyframe(time, kf.value, kf.interpolation);
			}

			if (pm != null) {
				data.add(new Object[] { layer, e.effectIndex, e.property, null });
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data.toArray(new Object[data.size()][]));
		}
	}

	// TODO ReverseLayerTimeOperation に同じメソッドがあるので整理する。
	private void restoreKeyframes(Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);
		List<Object[]> data = (pm != null) ? Util.<Object[]>newList() : null;

		for (PropertyEntry e : _properties) {
			Layer layer = comp.getLayer(e.layerId);
			AnimatableValue<Object> avalue;
			if (e.effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, e.property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(e.effectIndex);
				avalue = PropertyUtil.getProperty(effect, e.property);
			}

			restore(e.data, avalue);

			if (pm != null) {
				data.add(new Object[] { layer, e.effectIndex, e.property, null });
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data.toArray(new Object[data.size()][]));
		}
	}

	// TODO ThreeDSwitchOperation 等にほぼ同じメソッドがあるので整理する。
	private <A extends AnimatableValue<V>, V> void restore(String data, A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(data, (Class<A>) dst.getClass());

			dst.reset(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframeMap().values()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

	private void stretchLayers(Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		Set<Layer> layers = Util.newSet();

		for (LayerEntry e : _layers) {
			Layer layer = comp.getLayer(e.layerId);

			double oldRate = layer.getRate();
			Time timeToHold = timeToHold(layer);

			Time newStartTime, newInPoint, newOutPoint;
			if (oldRate * _newRate > 0) {
				newInPoint = stretchTime(layer.getInPoint(), oldRate, timeToHold);
				newOutPoint = stretchTime(layer.getOutPoint(), oldRate, timeToHold);
				newStartTime = stretchTime(layer.getStartTime(), oldRate, timeToHold);

			} else {
				newInPoint = stretchTime(layer.getOutPoint(), oldRate, timeToHold);
				newOutPoint = stretchTime(layer.getInPoint(), oldRate, timeToHold);
				newStartTime = null;

				if (layer instanceof MediaLayer) {
					MediaInput mi = ((MediaLayer) layer).getMediaInput();
					Time duration = mi.getDuration();
					if (duration != null) {
						Time endTime = layer.getStartTime().add(
								new Time((long)(duration.timeValue/Math.abs(oldRate)), duration.timeScale));
						newStartTime = stretchTime(endTime, oldRate, timeToHold);
					}
				}
				if (newStartTime == null) {
					newStartTime = stretchTime(layer.getStartTime(), oldRate, timeToHold);
				}
			}

			layer.setStartTime(newStartTime);
			layer.setInPoint(newInPoint);
			layer.setOutPoint(newOutPoint);
			layer.setRate(_newRate);
			layers.add(layer);
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}
	}

	private void restoreLayers(Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		Set<Layer> layers = Util.newSet();

		for (LayerEntry e : _layers) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setStartTime(e.startTime);
			layer.setInPoint(e.inPoint);
			layer.setOutPoint(e.outPoint);
			layer.setRate(e.rate);
			layers.add(layer);
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}
	}

}
