/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.wizards;

import java.io.File;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.app.BatchStore;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.output.WindowsDirectShowOutput;
import ch.kuramo.javie.core.output.Output.VideoChannels;
import ch.kuramo.javie.core.output.WindowsDirectShowOutput.VideoCompressorDescriptor;
import ch.kuramo.javie.core.output.WindowsDirectShowOutput.VideoCompressorSettings;

public class AVIOutputSettingsWizardPage extends WizardPage {

	private static final VideoCompressorDescriptor VIDEOCOMPDESC_UNCOMPRESSED
							= VideoCompressorDescriptor.uncompressedDescriptor("非圧縮");

	private static boolean batchDefault = false;

	private static boolean videoEnabledDefault = true;

	private static VideoChannels videoChannelsDefault = VideoChannels.RGB;

	private static Color colorMatteDefault = Color.BLACK;

	private static VideoCompressorSettings vcSettingsDefault
							= new VideoCompressorSettings(VIDEOCOMPDESC_UNCOMPRESSED);

	private static boolean audioEnabledDefault = true;

	private static AudioMode audioModeDefault = AudioMode.STEREO_48KHZ_INT16;


	private final Composition composition;

	private File defaultFile;

	private File file;

	private boolean batch = batchDefault;

	private boolean videoEnabled = videoEnabledDefault;

	private VideoChannels videoChannels = videoChannelsDefault;

	private Color colorMatte = colorMatteDefault;

	private VideoCompressorSettings vcSettings = vcSettingsDefault;

	private boolean audioEnabled = audioEnabledDefault;

	private AudioMode audioMode = audioModeDefault;

	private Image colorMatteImage;


	public AVIOutputSettingsWizardPage(Composition composition, File defaultFile) {
		super("AVIOutputSettingsWizardPage");
		setTitle("出力設定");

		this.composition = composition;
		this.defaultFile = defaultFile;
	}

	public void dispose() {
		if (colorMatteImage != null) {
			colorMatteImage.dispose();
			colorMatteImage = null;
		}
		super.dispose();
	}

	public void createControl(Composite parent) {
		GridBuilder gb = new GridBuilder(parent, 1, false);
		((GridLayout) gb.getComposite().getLayout()).verticalSpacing = 0;

		createFileGroup(gb);
		gb.size(15, 15).composite(SWT.NULL);
		createVideoGroup(gb);
		gb.size(15, 15).composite(SWT.NULL);
		createAudioGroup(gb);

		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		setControl(grid);
		doValidate();
	}

	private void createFileGroup(GridBuilder gb) {
		Group group = gb.hAlign(SWT.FILL).group(SWT.NONE, null);
		group.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb2 = new GridBuilder(group, 3, false);
		GridLayout layout = (GridLayout) gb2.getComposite().getLayout();
		layout.marginWidth = 15;

								gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "出力ファイル:");
		final Text fileText =	gb2.hSpan(2).hAlign(SWT.FILL).hGrab().text(SWT.READ_ONLY, "");

								gb2.hSpan(1).size(10, 10).composite(SWT.NULL);
		Button batchCheck =		gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.CHECK, "バッチ登録");
		Button fileButton =		gb2.hSpan(1).hAlign(SWT.RIGHT).tabBefore(batchCheck).button(SWT.PUSH, "ファイルを指定...");

		fileText.setBackground(fileText.getDisplay().getSystemColor(SWT.COLOR_WHITE));

		batchCheck.setSelection(batch);
		batchCheck.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				batch = ((Button)e.widget).getSelection();
				doValidate();
			}
		});

		fileButton.setFocus();
		fileButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				File newFile = showSaveDialog(defaultFile.getParentFile(), defaultFile.getName());
				if (newFile != null) {
					defaultFile = file = newFile;
					fileText.setText(newFile.getAbsolutePath());
					fileText.setFocus();
					fileText.selectAll();
					doValidate();
				}
			}
		});

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createVideoGroup(GridBuilder gb) {
		String[] channelItems = {
				"RGB", "アルファ", "RGB + アルファ（ストレート）", "RGB + アルファ（乗算済み）"
		};

		final Button enabledCheck = gb.hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "ビデオ出力");
		enabledCheck.setSelection(videoEnabled);

		Group group = gb.hAlign(SWT.FILL).group(SWT.NONE, null);
		group.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb2 = new GridBuilder(group, 6, false);
		GridLayout layout = (GridLayout) gb2.getComposite().getLayout();
		layout.marginWidth = 15;
		layout.marginBottom = 3;

		final Label chLabel = gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "出力チャンネル:");
		final Combo chCombo = gb2.hSpan(2).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, 0,
										channelItems, channelItems[videoChannels.ordinal()]);
		gb2.hSpan(1).size(10, 10).composite(SWT.NULL);
		final Label matteLabel = gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "カラーマット:");
		final Button matteButton = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.PUSH | SWT.FLAT, "");

		final Label vcompLabel = gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "ビデオ圧縮:");
		final Button vcompButton = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.PUSH, "設定...");
		final Label vcompName = gb2.hSpan(4).hAlign(SWT.FILL).hGrab().label(SWT.NULL, vcSettings.friendlyName);


		enableControls(videoEnabled, chLabel, chCombo, vcompLabel, vcompButton, vcompName);
		enableControls(videoEnabled && videoChannels == VideoChannels.RGBA_PREMULTIPLIED, matteLabel, matteButton);
		updateColorMatteImage(matteButton, toRGB(colorMatte));

		enabledCheck.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				videoEnabled = enabledCheck.getSelection();
				enableControls(videoEnabled, chLabel, chCombo, vcompLabel, vcompButton, vcompName);
				enableControls(videoEnabled && videoChannels == VideoChannels.RGBA_PREMULTIPLIED, matteLabel, matteButton);
				doValidate();
			}
		});

		chCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				videoChannels = VideoChannels.values()[chCombo.getSelectionIndex()];
				enableControls(videoChannels == VideoChannels.RGBA_PREMULTIPLIED, matteLabel, matteButton);
				doValidate();
			}
		});

		matteButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				ColorDialog dialog = new ColorDialog(getShell());
				dialog.setRGB(toRGB(colorMatte));
				RGB rgb = dialog.open();
				if (rgb != null) {
					colorMatte = toColor(rgb);
					updateColorMatteImage((Button)e.widget, rgb);
					doValidate();
				}
				doValidate();
			}
		});

		vcompButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				VideoCompressorSettingsDialog dialog
						= new VideoCompressorSettingsDialog(getShell(), composition, vcSettings);
				if (dialog.open() == IDialogConstants.OK_ID) {
					vcSettings = dialog.getSettings();
					vcompName.setText(vcSettings.friendlyName);
					doValidate();
				}
			}
		});


		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createAudioGroup(GridBuilder gb) {
		String[] audioItems = {
				"48kHz, 16bit", "48kHz, 32bit", "48kHz, 32bit浮動小数",
				"96kHz, 16bit", "96kHz, 32bit", "96kHz, 32bit浮動小数",
		};

		final Button enabledCheck = gb.hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "オーディオ出力");
		enabledCheck.setSelection(audioEnabled);

		Group group = gb.hAlign(SWT.FILL).group(SWT.NONE, null);
		group.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb3 = new GridBuilder(group, 2, false);
		GridLayout layout = (GridLayout) gb3.getComposite().getLayout();
		layout.marginWidth = 15;
		layout.marginBottom = 3;

		final Label amodeLabel = gb3.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "内部処理形式:");
		final Combo amodeCombo = gb3.hSpan(1).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, 0,
													audioItems, audioItems[audioMode.ordinal()]);


		enableControls(audioEnabled, amodeLabel, amodeCombo);

		enabledCheck.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				audioEnabled = enabledCheck.getSelection();
				enableControls(audioEnabled, amodeLabel, amodeCombo);
				doValidate();
			}
		});

		amodeCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				audioMode = AudioMode.values()[amodeCombo.getSelectionIndex()];
				doValidate();
			}
		});


		Composite grid3 = gb3.getComposite();
		grid3.setTabList(gb3.getTabList());
	}

	private File showSaveDialog(File folder, String fileName) {
		String[] filterNames = new String[] { "AVI ファイル", "すべてのファイル" };
		String[] filterExtensions = new String[] { "*.avi", "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterExtensions[filterExtensions.length-1] = "*.*";
		}

		FileDialog dialog = new FileDialog(getShell(), SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(fileName);
		dialog.setOverwrite(true);

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void enableControls(boolean enabled, Control... controls) {
		org.eclipse.swt.graphics.Color disabledColor
				= getShell().getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY);

		for (Control c : controls) {
			if (c instanceof Label) {
				c.setForeground(enabled ? c.getParent().getForeground() : disabledColor);
			} else {
				c.setEnabled(enabled);
			}
		}
	}

	private RGB toRGB(Color color) {
		return new RGB((int)(color.r*255), (int)(color.g*255), (int)(color.b*255));
	}

	private Color toColor(RGB rgb) {
		return new Color(rgb.red/255.0, rgb.green/255.0, rgb.blue/255.0, 1.0);
	}

	private void updateColorMatteImage(Button button, RGB rgb) {
		PaletteData palette = new PaletteData(new RGB[] { new RGB(0, 0, 0), rgb });
		byte[] data = new byte[64];
		Arrays.fill(data, (byte) 255);
		ImageData imageData = new ImageData(32, 16, 1, palette, 1, data);

		Image image = new Image(button.getDisplay(), imageData);
		button.setImage(image);
		if (colorMatteImage != null) {
			colorMatteImage.dispose();
		}
		colorMatteImage = image;
	}

	private void doValidate() {
		setPageComplete(false);

		if (file == null) {
			setErrorMessage("出力ファイルが指定されていません。");
			return;
		}

		if (!videoEnabled && !audioEnabled) {
			setErrorMessage("ビデオ出力とオーディオ出力の少なくともどちらか一方は有効にする必要があります。");
			return;
		}

//		if (videoEnabled) {
//			switch (videoChannels) {
//				case RGBA_STRAIGHT:
//				case RGBA_PREMULTIPLIED:
//					if (vcSettings.depth < 32) {
//						setErrorMessage(String.format(
//								"ビデオ圧縮 “%s” は “RGB + アルファ” の出力に対応していません。", vcSettings.friendlyName));
//						return;
//					}
//					break;
//			}
//		}

		StringBuilder warning = new StringBuilder();

		if (batch && BatchStore.isReserved(file)) {
			warning.append("同名の出力ファイルでのバッチ登録がすでにあるため、保留状態で登録されます。\n");
		}

		setMessage(warning.length() > 0 ? warning.toString() : null, IMessageProvider.WARNING);
		setErrorMessage(null);
		setPageComplete(true);
	}

	public void saveDefaults() {
		batchDefault = batch;
		videoEnabledDefault = videoEnabled;
		videoChannelsDefault = videoChannels;
		colorMatteDefault = colorMatte;
		vcSettingsDefault = vcSettings;
		audioEnabledDefault = audioEnabled;
		audioModeDefault = audioMode;
	}

	public File getFile() {
		return file;
	}

	public boolean isBatch() {
		return batch;
	}

	public boolean isVideoEnabled() {
		return videoEnabled;
	}

	public VideoChannels getVideoChannels() {
		return videoChannels;
	}

	public Color getColorMatte() {
		return colorMatte;
	}

	public VideoCompressorSettings getVideoCompressorSettings() {
		return vcSettings;
	}

	public boolean isAudioEnabled() {
		return audioEnabled;
	}

	public AudioMode getAudioMode() {
		return audioMode;
	}


	private static class VideoCompressorSettingsDialog extends Dialog {

		private final Composition composition;

		private final List<VideoCompressorDescriptor> compressors;

		private final Map<Integer, VideoCompressorSettings> settingsCache = Util.newMap();

		private int selectedIndex;

		private Combo compressorCombo;

		private Text qualityText;

		private Text keyFrameText;

		private Text PFramesText;

		private Text bitRateText;

		private Text windowSizeText;

		private Button configButton;

		private Button resetButton;

		private Button aboutButton;

		private Label errorLabel;

		private boolean updatingControls;


		private VideoCompressorSettingsDialog(Shell parentShell,
					Composition composition, VideoCompressorSettings defaultSettings) {
			super(parentShell);

			this.composition = composition;

			compressors = WindowsDirectShowOutput.listVideoCompressors(composition);
			Collections.sort(compressors, new Comparator<VideoCompressorDescriptor>() {
				public int compare(VideoCompressorDescriptor o1, VideoCompressorDescriptor o2) {
					return o1.friendlyName.compareTo(o2.friendlyName);
				}
			});
			compressors.add(VIDEOCOMPDESC_UNCOMPRESSED);
			selectedIndex = compressors.size()-1;

			for (int i = 0, n = compressors.size()-1; i < n; ++i) {
				VideoCompressorDescriptor desc = compressors.get(i);
				if (desc.moniker.equals(defaultSettings.moniker)) {
					settingsCache.put(i, defaultSettings.clone());
					selectedIndex = i;
					break;
				}
			}
		}

		private VideoCompressorSettings getSettings() {
			return settingsCache.get(selectedIndex);
		}

		protected void configureShell(Shell newShell) {
			super.configureShell(newShell);
			newShell.setText("ビデオ圧縮");
		}

		protected Control createDialogArea(Composite parent) {
			Composite composite = (Composite) super.createDialogArea(parent);

			List<String> compressorNames = Util.newList();
			for (VideoCompressorDescriptor desc : compressors) {
				compressorNames.add(desc.friendlyName);
			}

			GridBuilder gb = new GridBuilder(composite, 10, true);

								gb.hSpan(2).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "圧縮プログラム:");
			compressorCombo = 	gb.hSpan(8).hAlign(SWT.FILL).hGrab().combo(SWT.READ_ONLY,
										compressorNames.toArray(new String[compressorNames.size()]));

								gb.hSpan(10).size(10, 10).composite(SWT.NULL);

								gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "圧縮の品質 (0～100):");
			qualityText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(compressorCombo).text(SWT.BORDER, "");
								gb.hSpan(2).size(10, 10).composite(SWT.NULL);
			configButton =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "設定");

								gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "キーフレーム (0:無し):");
			keyFrameText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(qualityText).text(SWT.BORDER, "");
								gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "フレームごと");
			resetButton =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "標準に戻す");

								gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "ｷｰﾌﾚｰﾑ毎のPﾌﾚｰﾑ:");
			PFramesText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(keyFrameText).text(SWT.BORDER, "");
								gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "フレーム");
			aboutButton =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "バージョン情報");

								gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "データレート (0:無し):");
			bitRateText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(PFramesText).text(SWT.BORDER, "");
								gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "KB/秒");
								gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "").setVisible(false);

								gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "ウインドウサイズ:");
			windowSizeText =	gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(bitRateText).text(SWT.BORDER, "");
								gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "フレーム");
								gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "").setVisible(false);

								gb.hSpan(10).size(15, 15).composite(SWT.NULL);
			errorLabel =		gb.hSpan(10).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");

			ModifyListener modifyListener = new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					if (!updatingControls) {
						updateSettings();
					}
				}
			};
			qualityText.addModifyListener(modifyListener);
			keyFrameText.addModifyListener(modifyListener);
			PFramesText.addModifyListener(modifyListener);
			bitRateText.addModifyListener(modifyListener);
			windowSizeText.addModifyListener(modifyListener);

			configButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					config();
				}
			});

			resetButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					reset();
				}
			});

			aboutButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					about();
				}
			});

			compressorCombo.setVisibleItemCount(20);
			compressorCombo.select(selectedIndex);
			compressorCombo.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					selectedIndex = compressorCombo.getSelectionIndex();
					updateControls();
				}
			});

			Composite grid = gb.getComposite();
			grid.setTabList(gb.getTabList());

			return composite;
		}

		public void create() {
			super.create();
			updateControls();
		}

		private void updateText(Text control, boolean enabled, String text) {
			control.setEnabled(enabled);
			control.setText(enabled ? text : "");
		}

		private void updateControls() {
			updatingControls = true;
			try {
				VideoCompressorDescriptor desc = compressors.get(selectedIndex);
				VideoCompressorSettings settings = settingsCache.get(selectedIndex);
				if (settings == null) {
					settings = new VideoCompressorSettings(desc);
					settingsCache.put(selectedIndex, settings);
				}

				updateText(qualityText, desc.canQuality, String.valueOf((int)(settings.quality*100)));
				updateText(keyFrameText, desc.canKeyFrame, String.valueOf(settings.keyFrameRate));
				updateText(PFramesText, desc.canBFrame, String.valueOf(settings.PFramesPerKey));
				updateText(bitRateText, desc.canCrunch, String.valueOf(settings.bitRate >> 13));
				updateText(windowSizeText, desc.canWindow, String.valueOf(settings.windowSize));

				boolean enableReset = desc.canQuality | desc.canKeyFrame | desc.canBFrame
									| desc.canCrunch | desc.canWindow | desc.canConfigDialog;

				configButton.setEnabled(desc.canConfigDialog);
				resetButton.setEnabled(enableReset);
				aboutButton.setEnabled(desc.canAboutDialog);

				errorMessage(null);
				dialogComplete(true);

			} finally {
				updatingControls = false;
			}
		}

		private void updateSettings() {
			dialogComplete(false);

			VideoCompressorDescriptor desc = compressors.get(selectedIndex);
			VideoCompressorSettings settings = settingsCache.get(selectedIndex);

			double quality = settings.quality;
			if (desc.canQuality) {
				try {
					quality = Integer.parseInt(qualityText.getText().trim());
				} catch (NumberFormatException e) {
					quality = -1;
				}
				if (quality < 0 || quality > 100) {
					errorMessage("圧縮の品質には0から100までの整数を入力してください。");
					return;
				} else {
					quality /= 100;
				}
			}

			int keyFrameRate = settings.keyFrameRate;
			if (desc.canKeyFrame) {
				try {
					keyFrameRate = Integer.parseInt(keyFrameText.getText().trim());
				} catch (NumberFormatException e) {
					keyFrameRate = -1;
				}
				if (keyFrameRate < 0) {
					errorMessage("キーフレームには0以上の整数を入力してください。");
					return;
				}
			}

			int PFramesPerKey = settings.PFramesPerKey;
			if (desc.canBFrame) {
				try {
					PFramesPerKey = Integer.parseInt(PFramesText.getText().trim());
				} catch (NumberFormatException e) {
					PFramesPerKey = -1;
				}
				if (PFramesPerKey < 0) {
					errorMessage("ｷｰﾌﾚｰﾑ毎のPﾌﾚｰﾑには0以上の整数を入力してください。");
					return;
				}
			}

			long bitRate = settings.bitRate;
			if (desc.canCrunch) {
				try {
					bitRate = Integer.parseInt(bitRateText.getText().trim());
				} catch (NumberFormatException e) {
					bitRate = -1;
				}
				if (bitRate < 0) {
					errorMessage("データレートには0以上の整数を入力してください。");
					return;
				} else {
					bitRate = Math.min(bitRate << 13, Integer.MAX_VALUE);
				}
			}

			long windowSize = settings.windowSize;
			if (desc.canWindow) {
				try {
					windowSize = Long.parseLong(windowSizeText.getText().trim());
				} catch (NumberFormatException e) {
					windowSize = 0;
				}
				if (windowSize <= 0) {
					errorMessage("ウインドウサイズには1以上の整数を入力してください。");
					return;
				}
			}

			settings.quality = quality;
			settings.keyFrameRate = keyFrameRate;
			settings.PFramesPerKey = PFramesPerKey;
			settings.bitRate = bitRate;
			settings.windowSize = windowSize;

			errorMessage(null);
			dialogComplete(true);
		}

		private void errorMessage(String message) {
			errorLabel.setText(message != null ? message : "");
		}

		private void dialogComplete(boolean complete) {
			Button okButton = getButton(IDialogConstants.OK_ID);
			okButton.setEnabled(complete);
		}

		private void reset() {
			VideoCompressorDescriptor desc = compressors.get(selectedIndex);
			settingsCache.put(selectedIndex, new VideoCompressorSettings(desc));

			updateControls();
		}

		private void config() {
			VideoCompressorDescriptor desc = compressors.get(selectedIndex);
			VideoCompressorSettings settings = settingsCache.get(selectedIndex);

			WindowsDirectShowOutput.openVideoCompressorConfigDialog(
					composition, desc, settings, getHWND());
		}

		private void about() {
			VideoCompressorDescriptor desc = compressors.get(selectedIndex);

			WindowsDirectShowOutput.openVideoCompressorAboutDialog(desc, getHWND());
		}

		private long getHWND() {
			try {
				Shell shell = getShell();
				Class<?> clazz = shell.getClass();
				Field handleField = clazz.getField("handle");
				return (Integer) /*(Long)*/ handleField.get(shell);
			} catch (NoSuchFieldException e) {
				throw new JavieRuntimeException(e);
			} catch (IllegalAccessException e) {
				throw new JavieRuntimeException(e);
			}
		}

	}

}
