/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.wizards;

import java.util.Arrays;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.app.widgets.FrameRateComboUtil;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaOptions;
import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.MediaOptions.AlphaType;
import ch.kuramo.javie.core.MediaOptions.Option;

public class MediaOptionsWizardPage extends WizardPage {

	private final MediaSource source;

	private final MediaOptions options;

	private final MediaOptions origOptions;

	private Image colorMatteImage;

	private Button customFpsButton;

	private Combo customFpsCombo;

	private Text durationText;

	private Text widthText;

	private Text heightText;

	private Button freezeRatioCheck;

	private int widthRatio;

	private int heightRatio;


	public MediaOptionsWizardPage(MediaItem mediaItem) {
		super("MediaInputOptionsWizardPage");
		this.source = mediaItem.getMediaSource();
		this.options = mediaItem.getMediaOptions();
		this.origOptions = options.clone();

		setTitle("メディアオプション：" + mediaItem.getName());
	}

	public void dispose() {
		if (colorMatteImage != null) {
			colorMatteImage.dispose();
			colorMatteImage = null;
		}
		super.dispose();
	}

	public void createControl(Composite parent) {
		GridBuilder gb = new GridBuilder(parent, 1, false);
		((GridLayout) gb.getComposite().getLayout()).verticalSpacing = 0;

		createAlphaGroup(gb);
		gb.size(10, 10).composite(SWT.NULL);
		createFrameRateAndDurationGroup(gb);
		gb.size(10, 10).composite(SWT.NULL);
		createSizeGroup(gb);
		gb.size(10, 10).composite(SWT.NULL);
		createMiscGroup(gb);

		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		setControl(grid);
		doValidate();
	}

	private void createAlphaGroup(GridBuilder gb) {
		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "アルファチャンネル");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 4, false);

		Button ignoreButton = 		gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.RADIO, "無視");
									gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.PUSH | SWT.FLAT, "-").setVisible(false);	// サイズ調整のためのダミー

		Button straightButton =		gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.RADIO, "ストレートアルファ");
									gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.PUSH | SWT.FLAT, "-").setVisible(false);	// サイズ調整のためのダミー

		Button premultButton =		gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.RADIO, "乗算済みアルファ");
									gb2.hSpan(1).size(10, 10).composite(SWT.NULL);
		final Label matteLabel =	gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "カラーマット:");
		final Button matteButton =	gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.PUSH | SWT.FLAT, "");

		AlphaType alphaType = options.getAlphaType();
		ignoreButton.setEnabled(options.isAvailable(Option.IGNORE_ALPHA));
		ignoreButton.setSelection(alphaType == AlphaType.IGNORE);
		straightButton.setEnabled(options.isAvailable(Option.STRAIGHT_ALPHA));
		straightButton.setSelection(alphaType == AlphaType.STRAIGHT);
		premultButton.setEnabled(options.isAvailable(Option.PREMULTIPLIED_ALPHA));
		premultButton.setSelection(alphaType == AlphaType.PREMULTIPLIED);

		if (options.isAvailable(Option.IGNORE_ALPHA)) {
			ignoreButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					options.setAlphaType(AlphaType.IGNORE);
					enableControls(false, matteLabel, matteButton);
					doValidate();
				}
			});
		}

		if (options.isAvailable(Option.STRAIGHT_ALPHA)) {
			straightButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					options.setAlphaType(AlphaType.STRAIGHT);
					enableControls(false, matteLabel, matteButton);
					doValidate();
				}
			});
		}

		if (options.isAvailable(Option.PREMULTIPLIED_ALPHA)) {
			premultButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					options.setAlphaType(AlphaType.PREMULTIPLIED);
					enableControls(true, matteLabel, matteButton);
					doValidate();
				}
			});

			matteButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					ColorDialog dialog = new ColorDialog(getShell());
					dialog.setRGB(toRGB(options.getColorMatte()));
					RGB rgb = dialog.open();
					if (rgb != null) {
						options.setColorMatte(toColor(rgb));
						updateColorMatteImage((Button)e.widget, rgb);
						doValidate();
					}
				}
			});

			enableControls(alphaType == AlphaType.PREMULTIPLIED, matteLabel, matteButton);
			updateColorMatteImage(matteButton, toRGB(options.getColorMatte()));

		} else {
			enableControls(false, matteLabel, matteButton);
			updateColorMatteImage(matteButton, toRGB(Color.BLACK));
		}

		if (!options.isAvailable(Option.IGNORE_ALPHA)
				&& !options.isAvailable(Option.STRAIGHT_ALPHA)
				&& !options.isAvailable(Option.PREMULTIPLIED_ALPHA)) {

			enableControls(false, group);
		}

		// ラジオグループを含む場合、タブリストを設定するとWindowsで不適切な動作をする。
		//Composite grid2 = gb2.getComposite();
		//grid2.setTabList(gb2.getTabList());
	}

	private void createFrameRateAndDurationGroup(GridBuilder gb) {
		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "フレームレートとデュレーション");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 5, false);

		Button soruceFpsButton =	gb2.hSpan(3).hAlign(SWT.LEFT).button(SWT.RADIO, "ファイルのフレームレートを使用:");
		Label soruceFpsLabel =		gb2.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");

		customFpsButton =			gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.RADIO, "次のフレームレートに調整:");
		customFpsCombo =			gb2.hSpan(2).hAlign(SWT.LEFT).combo(SWT.NULL, 0, FrameRateComboUtil.getComboItems(), "");
		Label framePerSecLabel =	gb2.hSpan(1).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "フレーム／秒");

									gb2.hSpan(5).size(10, 10).composite(SWT.NULL);

		Label durationLabel =		gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "デュレーション:");
		final Label durationLabel2;
		if (options.isAvailable(Option.DURATION)) {
			durationText =			gb2.hSpan(2).hAlign(SWT.FILL).text(SWT.BORDER, "");
			durationLabel2 =		gb2.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");
		} else {
			durationText =			null;
			durationLabel2 =		gb2.hSpan(4).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");
		}

		Time soruceFrameDuration = source.getVideoFrameDuration(null);
		if (soruceFrameDuration != null && soruceFrameDuration.timeValue > 0) {
			soruceFpsButton.setSelection(true);
			soruceFpsLabel.setText(FrameRateComboUtil.toComboItem(soruceFrameDuration) + " フレーム／秒");
		} else {
			soruceFpsButton.setEnabled(false);
		}

		if (options.isAvailable(Option.VIDEO_FRAME_DURATION)) {
			Time frameDuration = options.getVideoFrameDuration();
			if (frameDuration == null) {
				soruceFpsButton.setSelection(true);
				customFpsButton.setSelection(false);
			} else {
				soruceFpsButton.setSelection(false);
				customFpsButton.setSelection(true);
				customFpsCombo.setText(FrameRateComboUtil.toComboItem(frameDuration));
			}

			updateDuration(durationLabel2);

			customFpsCombo.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					doValidate();
					updateDuration(durationLabel2);
				}
			});

			customFpsCombo.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent e) {
					doValidate();
					updateDuration(durationLabel2);
				}
				public void widgetDefaultSelected(SelectionEvent e) {
					doValidate();
					updateDuration(durationLabel2);
				}
			});

		} else if (options.isAvailable(Option.DURATION)) {
			enableControls(false, soruceFpsButton, soruceFpsLabel,
					customFpsButton, customFpsCombo, framePerSecLabel);

			updateDuration(durationText, durationLabel2);

			durationText.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					doValidate();
					updateDuration(null, durationLabel2);
				}
			});

		} else {
			enableControls(false, soruceFpsButton, soruceFpsLabel,
					customFpsButton, customFpsCombo, framePerSecLabel, durationLabel, durationLabel2);

			updateDuration(durationLabel2);
		}

		if (!options.isAvailable(Option.VIDEO_FRAME_DURATION)
				&& !options.isAvailable(Option.DURATION)) {

			enableControls(false, group);
		}

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createSizeGroup(GridBuilder gb) {
		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "サイズ");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 3, false);

		VideoBounds bounds = source.getVideoFrameBounds(options);

		if (options.isAvailable(Option.SIZE)) {
								gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "幅:");
			widthText =			gb2.hSpan(1).hAlign(SWT.FILL).width(70)
										.text(SWT.BORDER, String.valueOf(bounds.width));
			freezeRatioCheck =	gb2.span(1, 2).hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "縦横比を固定");
	
								gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "高さ:");
			heightText =		gb2.hSpan(1).hAlign(SWT.FILL).width(70).tabAfter(widthText)
										.text(SWT.BORDER, String.valueOf(bounds.height));

			final ModifyListener modifyListener = new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					updateSize((Control)e.widget, this);
					doValidate();
				}
			};

			widthText.addModifyListener(modifyListener);
			heightText.addModifyListener(modifyListener);

			freezeRatioCheck.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					updateSize(freezeRatioCheck, modifyListener);
					doValidate();	// FIXME この doValidate は不要なのでは？
				}
			});

			updateSize(null, modifyListener);

		} else {
			Label widthLabel1 =		gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "幅:");
			Label widthLabel2 =		gb2.hSpan(2).hAlign(SWT.LEFT).label(SWT.NULL, String.valueOf(bounds.width));
			Label heightLabel1 =	gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "高さ:");
			Label heightLabel2 =	gb2.hSpan(2).hAlign(SWT.LEFT).label(SWT.NULL, String.valueOf(bounds.height));
			enableControls(false, group, widthLabel1, widthLabel2, heightLabel1, heightLabel2);
		}

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createMiscGroup(GridBuilder gb) {
		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "その他のオプション");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 1, false);

		Button flipVertCheck = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.CHECK, "上下反転");

		if (options.isAvailable(Option.FLIP_VERTICAL)) {
			flipVertCheck.setSelection(options.isFlipVertical());

			flipVertCheck.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					options.setFlipVertical(((Button)e.widget).getSelection());
					doValidate();
				}
			});

		} else {
			flipVertCheck.setEnabled(false);
		}

		if (!options.isAvailable(Option.FLIP_VERTICAL)
				/* && !...  */) {

			enableControls(false, group);
		}

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void enableControls(boolean enabled, Control... controls) {
		org.eclipse.swt.graphics.Color disabledColor
				= getShell().getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY);

		for (Control c : controls) {
			if (c instanceof Label || c instanceof Group) {
				c.setForeground(enabled ? c.getParent().getForeground() : disabledColor);
			} else {
				c.setEnabled(enabled);
			}
		}
	}

	private RGB toRGB(Color color) {
		return new RGB((int)(color.r*255), (int)(color.g*255), (int)(color.b*255));
	}

	private Color toColor(RGB rgb) {
		return new Color(rgb.red/255.0, rgb.green/255.0, rgb.blue/255.0, 1.0);
	}

	private void updateColorMatteImage(Button button, RGB rgb) {
		PaletteData palette = new PaletteData(new RGB[] { new RGB(0, 0, 0), rgb });
		byte[] data = new byte[64];
		Arrays.fill(data, (byte) 255);
		ImageData imageData = new ImageData(32, 16, 1, palette, 1, data);

		Image image = new Image(button.getDisplay(), imageData);
		button.setImage(image);
		if (colorMatteImage != null) {
			colorMatteImage.dispose();
		}
		colorMatteImage = image;
	}

	private void updateDuration(Label label) {
		Time duration = source.getDuration(options);
		Time frameDuration = source.getVideoFrameDuration(options);

		String timeCode = TimeCode.toTimeCode(duration, frameDuration);
		int base = (int)Math.round((double)frameDuration.timeScale / frameDuration.timeValue);
		boolean drop = (timeCode.indexOf(';') != -1);

		label.setText(String.format("%s (ベース %d%s) , %dフレーム",
				timeCode, base, drop ? "ドロップ" : "", duration.toFrameNumber(frameDuration)));
	}

	private void updateDuration(Text text, Label label) {
		Time duration = source.getDuration(options);
		Time frameDuration = source.getVideoFrameDuration(options);
		if (frameDuration.timeValue == 0) {
			frameDuration = FrameDuration.FPS_59_94;
		}

		String timeCode = TimeCode.toTimeCode(duration, frameDuration);
		int base = (int)Math.round((double)frameDuration.timeScale / frameDuration.timeValue);
		boolean drop = (timeCode.indexOf(';') != -1);

		label.setText(String.format(" = %s (ベース %d%s)", timeCode, base, drop ? "ドロップ" : ""));

		if (text != null) {
			text.setText(timeCode);
		}
	}

	private void updateSize(Control control, ModifyListener modifyListener) {
		String wStr = null, hStr = null;
		int width = 0, height = 0;

		wStr = widthText.getText().trim();
		try {
			width = Integer.parseInt(wStr);
		} catch (NumberFormatException e) {
			width = 0;
		}

		hStr = heightText.getText().trim();
		try {
			height = Integer.parseInt(hStr);
		} catch (NumberFormatException e) {
			height = 0;
		}

		if (freezeRatioCheck.getSelection()) {
			if (control == widthText && width > 0) {
				height = Math.max(1, width * heightRatio / widthRatio);
				heightText.removeModifyListener(modifyListener);
				heightText.setText(String.valueOf(height));
				heightText.addModifyListener(modifyListener);
			} else if (control == heightText && height > 0) {
				width = Math.max(1, height * widthRatio / heightRatio);
				widthText.removeModifyListener(modifyListener);
				widthText.setText(String.valueOf(width));
				widthText.addModifyListener(modifyListener);
			}
		} else if ((control == null || control == widthText || control == heightText) && width > 0 && height > 0) {
			int gcd = gcd(width, height);
			widthRatio = width / gcd;
			heightRatio = height / gcd;
			freezeRatioCheck.setText(String.format("縦横比を%d:%dに固定", widthRatio, heightRatio));
			freezeRatioCheck.getParent().layout(new Control[] { freezeRatioCheck });
		}
	}

	private int gcd(int x, int y) {
		while (y != 0) {
			int t = x % y;
			x = y;
			y = t;
		}
		return x;
	}

	private void doValidate() {
		setPageComplete(false);

		if (options.isAvailable(Option.VIDEO_FRAME_DURATION) && customFpsButton.getSelection()) {
			String fpsStr = customFpsCombo.getText().trim();
			if (fpsStr.length() == 0) {
				setErrorMessage("フレームレートが入力されていません。");
				return;
			}

			double fps;
			try {
				fps = Double.parseDouble(fpsStr);
			} catch (NumberFormatException e) {
				fps = 0;
			}

			// TODO フレームレートの上限はどうする？
			if (fps < 1) {
				setErrorMessage("フレームレートには1以上の数値を入力してください。");
				return;
			}

			options.setVideoFrameDuration(FrameRateComboUtil.toFrameDuration(fpsStr));

		} else if (options.isAvailable(Option.DURATION)) {
			Time frameDuration = source.getVideoFrameDuration(options);
			if (frameDuration.timeValue == 0) {
				frameDuration = FrameDuration.FPS_59_94;
			}

			String durationStr = durationText.getText().trim();
			if (durationStr.length() == 0) {
				setErrorMessage("デュレーションが入力されていません。");
				return;
			}

			long frames = TimeCode.parseTimeCode(durationStr, frameDuration);

			// TODO デュレーションの上限はどうする？
			if (frames <= 0) {
				setErrorMessage("デュレーションは1フレーム以上必要です。");
				return;
			}

			options.setDuration(Time.fromFrameNumber(frames, frameDuration));
		}

		if (options.isAvailable(Option.SIZE)) {
			String wStr = null, hStr = null;
			int width = 0, height = 0;

			wStr = widthText.getText().trim();
			try {
				width = Integer.parseInt(wStr);
			} catch (NumberFormatException e) {
				width = 0;
			}

			hStr = heightText.getText().trim();
			try {
				height = Integer.parseInt(hStr);
			} catch (NumberFormatException e) {
				height = 0;
			}

			if (wStr.length() == 0) {
				setErrorMessage("幅が入力されていません。");
				return;
			}
			if (width < 1 || width > CompositionWizardPage.MAX_SIZE) {
				setErrorMessage(String.format("幅には1から%dまでの整数を入力してください。", CompositionWizardPage.MAX_SIZE));
				return;
			}
			if (hStr.length() == 0) {
				setErrorMessage("高さが入力されていません。");
				return;
			}
			if (height < 1 || height > CompositionWizardPage.MAX_SIZE) {
				setErrorMessage(String.format("高さには1から%dまでの整数を入力してください。", CompositionWizardPage.MAX_SIZE));
				return;
			}

			options.setSize(new Size2i(width, height));
		}

		setErrorMessage(null);
		setPageComplete(!options.equals(origOptions));
	}

	public MediaOptions getMediaOptions() {
		return options;
	}

}
