/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.voodoo2javie;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectElementFactory;

class Voodoo2Javie {

	private static final Logger logger = LoggerFactory.getLogger(Voodoo2Javie.class);


	private List<String> data;

	private int width;

	private int height;


	boolean readFile(final File file, IRunnableContext rc) {
		IRunnableWithProgress runnable = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
				try {
					readFile(file, monitor);
				} catch (IOException e) {
					throw new InvocationTargetException(e);
				}
			}
		};

		try {
			logger.info("ファイルの読み込みを開始しました...");
			rc.run(true, true, runnable);
			if (data != null) {
				logger.info("ファイルの読み込みを完了しました");
				return true;
			} else {
				logger.info("データがありません");
				return false;
			}
		} catch (InterruptedException ex) {
			logger.info("ファイルの読み込みを中断しました");
		} catch (InvocationTargetException ex) {
			Throwable t = ex.getTargetException();
			logger.error("エラーが発生しました", (t != null) ? t : ex);
		}
		return false;
	}

	private void readFile(File file, IProgressMonitor monitor) throws IOException {
		List<String> data = Util.newList();

		monitor.beginTask("ファイルを読み込み中...", IProgressMonitor.UNKNOWN);
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), /*"UTF-8"*/ "US-ASCII"));
			for (String line; (line = reader.readLine()) != null; ) {
				if (line.startsWith("#timeindex ")) {
					int timeindex = Integer.parseInt(line.substring(11));
					if (timeindex != data.size()) {
						data.clear();
						logger.error("データが壊れています。");
						break;
					}
					data.add(reader.readLine());
				} else if (data.size() > 0) {
					break;
				}
			}

			if (data.isEmpty()) {
				this.data = null;
				width = 0;
				height = 0;
			} else {
				this.data = data;
				String[] array = data.get(0).split("\\s+");
				width = Integer.parseInt(array[16]);
				height = Integer.parseInt(array[17]);
			}

		} finally {
			monitor.done();
			if (reader != null) reader.close();
		}
	}

	void createCamera(double scale, Time frameDuration,
			Project project, ProjectElementFactory elementFactory,
			IProgressMonitor monitor) throws InterruptedException {

		if (data == null) {
			throw new IllegalStateException();
		}

		monitor.beginTask("カメラデータを処理中...", data.size());
		try {
			// コンポジションの作成
			Size2i size = new Size2i((int)Math.round(width*scale), (int)Math.round(height*scale));
			Time duration = Time.fromFrameNumber(data.size(), frameDuration);
			LayerComposition layerComp = elementFactory.newLayerComposition(
					ColorMode.RGBA8, size, frameDuration, duration);

			CompositionItem compItem = elementFactory.newCompositionItem(layerComp);
			compItem.setName("Voodoo Camera");

			project.getCompositions().add(layerComp);
			project.getItems().add(compItem);


			// カメラレイヤーの作成
			CameraLayer cameraLayer = elementFactory.newCameraLayer();
			Time time0 = Time.fromFrameNumber(0, frameDuration);
			cameraLayer.setStartTime(time0);
			cameraLayer.setInPoint(time0);
			cameraLayer.setOutPoint(duration);


			// カメラの初期パラメータの設定（キーフレームを設定するので、しなくてもよいが）
			String[] array = data.get(0).split("\\s+");
			double sx = Double.parseDouble(array[14]);
			double f = Double.parseDouble(array[20]);
			double zoom = f / sx * scale;
			cameraLayer.getZoom().reset(zoom);

			double x = Double.parseDouble(array[0]) / sx;
			double y = Double.parseDouble(array[1]) / sx;
			double z = Double.parseDouble(array[2]) / sx;
			cameraLayer.getPosition().reset(new Vec3d(x, y, z));
			cameraLayer.getPointOfInterest().reset(new Vec3d(x, y, z+zoom));

			cameraLayer.getOrientation().reset(calcOrientation(array));


			// カメラレイヤーをコンポジションに追加
			cameraLayer.setName("Voodoo Camera");
			layerComp.getLayers().add(cameraLayer);


			for (ListIterator<String> it = data.listIterator(); it.hasNext(); ) {
				if (monitor.isCanceled()) {
					throw new InterruptedException("canceled");
				}

				int index = it.nextIndex();
				Time time = Time.fromFrameNumber(index, frameDuration);

				array = it.next().split("\\s+");

				sx = Double.parseDouble(array[14]);
				f = Double.parseDouble(array[20]);
				zoom = f / sx * scale;
				cameraLayer.getZoom().putKeyframe(time, zoom, Interpolation.LINEAR);

				x = Double.parseDouble(array[0]) / sx;
				y = Double.parseDouble(array[1]) / sx;
				z = Double.parseDouble(array[2]) / sx;
				cameraLayer.getPosition().putKeyframe(time, new Vec3d(x, y, z), Interpolation.LINEAR);
				cameraLayer.getPointOfInterest().putKeyframe(time, new Vec3d(x, y, z+zoom), Interpolation.LINEAR);

				cameraLayer.getOrientation().putKeyframe(time, calcOrientation(array), Interpolation.LINEAR);

				monitor.worked(1);
			}
		} finally {
			monitor.done();
		}
	}

	private Vec3d calcOrientation(String[] array) {
		double m00 = Double.parseDouble(array[22]);
		double m01 = Double.parseDouble(array[23]);
		//double m02 = Double.parseDouble(array[24]);
		double m10 = Double.parseDouble(array[25]);
		double m11 = Double.parseDouble(array[26]);
		//double m12 = Double.parseDouble(array[27]);
		double m20 = Double.parseDouble(array[3]);
		double m21 = Double.parseDouble(array[4]);
		double m22 = Double.parseDouble(array[5]);

		double x, y, z;
		if (m20 == 1) {
			x = 0;
			y = -Math.PI / 2;
			z = -Math.atan2(m01, m11);
		} else if (m20 == -1) {
			x = 0;
			y = Math.PI / 2;
			z = -Math.atan2(m01, m11);
		} else {
			x = Math.atan2(m21, m22);
			y = Math.asin(-m20);
			z = Math.atan2(m10, m00);
		}

		return new Vec3d(Math.toDegrees(-x), Math.toDegrees(-y), Math.toDegrees(-z));
	}

	public int getWidth() {
		return width;
	}

	public int getHeight() {
		return height;
	}

	public int getFrameCount() {
		return (data != null) ? data.size() : 0;
	}

}
