/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

class CameraImpl implements Camera {

	private final RenderContext _context;

	private final VideoRenderSupport _vrSupport;

	private final CameraLayer _cameraLayer;

	private final Size2i _compSize;

	private final Size2i _viewportSize;

	private final double[] _prjMatrix2D;

	private final double[] _mvMatrix2D;

	private final Map<Time, double[][]> _matrix3DCache;


	@Inject
	CameraImpl(RenderContext context, VideoRenderSupport vrSupport) {
		LayerComposition comp = (LayerComposition) context.getComposition();

		// 上のレイヤーから(リスト内では末尾から)順に有効なカメラレイヤーを探し、見つかればそれを使う。
		CameraLayer cameraLayer = null;
		List<Layer> layers = comp.getLayers();
		Time time = context.getTime();
		for (ListIterator<Layer> it = layers.listIterator(layers.size()); it.hasPrevious(); ) {
			Layer layer = it.previous();
			if (layer instanceof CameraLayer && LayerNature.isVideoEnabled(layer)
					&& !layer.getInPoint().after(time) && layer.getOutPoint().after(time)) {
				cameraLayer = (CameraLayer) layer;
				break;
			}
		}

		_context = context;
		_vrSupport = vrSupport;
		_cameraLayer = cameraLayer;
		_compSize = comp.getSize();
		_viewportSize = context.getVideoResolution().scale(_compSize);
		_prjMatrix2D = new double[16];
		_mvMatrix2D = new double[16];
		_matrix3DCache = Util.newMap();

		createMatrix2D();
	}

	private void createMatrix2D() {
		GL2 gl = _context.getGL().getGL2();
		GLU glu = _context.getGLU();

		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluOrtho2D(0, _viewportSize.width, 0, _viewportSize.height);

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glLoadIdentity();

		_vrSupport.getMatrix(_prjMatrix2D, _mvMatrix2D);
	}

	public Size2i getViewportSize() {
		return _viewportSize;
	}

	public double[] getProjection2D() {
		return _prjMatrix2D;
	}

	public double[] getModelView2D() {
		return _mvMatrix2D;
	}

	public double[] getProjection3D() {
		return getMatrix3D()[0];
	}

	public double[] getModelView3D() {
		return getMatrix3D()[1];
	}

	private double[][] getMatrix3D() {
		Time time = _context.getTime();
		double[][] matrix = _matrix3DCache.get(time);
		if (matrix != null) {
			return matrix;
		}

		Vec3d orientation, rotation, position, interest;
		double fovy, near, far;
		double aspect = (double) _viewportSize.width / _viewportSize.height;

		if (_cameraLayer != null) {
			orientation = _cameraLayer.getOrientation().value(_context);
			rotation = new Vec3d(
					_cameraLayer.getRotationX().value(_context),
					_cameraLayer.getRotationY().value(_context),
					_cameraLayer.getRotationZ().value(_context));
			position = _cameraLayer.getPosition().value(_context);
			interest = _cameraLayer.getPointOfInterest().value(_context);

			double zoom = _cameraLayer.getZoom().value(_context);
			fovy = Math.toDegrees(2 * Math.atan(_compSize.height / (2*zoom)));
			near = _cameraLayer.getNear().value(_context);
			far = _cameraLayer.getFar().value(_context);

		} else {
			double zoom = getDefaultZoom();
			fovy = Math.toDegrees(2 * Math.atan(_compSize.height / (2*zoom)));

			near = CameraLayerImpl.DEFAULT_NEAR;
			far = CameraLayerImpl.DEFAULT_FAR;

			orientation = new Vec3d(0, 0, 0);
			rotation = new Vec3d(0, 0, 0);
			position = new Vec3d(_compSize.width/2.0, _compSize.height/2.0, -zoom);
			interest = new Vec3d(position.x, position.y, 0);
		}

		Resolution resolution = _context.getVideoResolution();
		position = resolution.scale(position);
		interest = resolution.scale(interest);
		near = resolution.scale(near);
		far = resolution.scale(far);


		GL2 gl = _context.getGL().getGL2();
		GLU glu = _context.getGLU();

		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluPerspective(fovy, aspect, near, far);

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glLoadIdentity();

		_vrSupport.multCameraMatrix(rotation, orientation, interest, position, true);

		if (_cameraLayer != null) {
			LayerMatrixUtil.multParentMatrix(_cameraLayer, _context, _vrSupport);
		}

		matrix = new double[2][16];
		_vrSupport.getMatrix(matrix[0], matrix[1]);

		_matrix3DCache.put(time, matrix);
		return matrix;
	}

	private double getDefaultZoom() {
		// TODO デフォルトカメラの画角はこれで妥当か？ 50mmレンズの画角が39.6。
		double fovx = 39.6;

		// 画角fovxの画面内にコンポジションの横幅がちょうど収まるようにする。
		return _compSize.width / (2 * Math.tan(Math.toRadians(fovx/2)));
	}

	public void getProjectionMatrix(double[] matrix) {
		System.arraycopy(getProjection3D(), 0, matrix, 0, 16);
	}

	public void getModelViewMatrix(double[] matrix) {
		System.arraycopy(getModelView3D(), 0, matrix, 0, 16);
	}

}
