/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IAlphaChannelSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class AlphaChannelSupportImpl implements IAlphaChannelSupport {

	private final RenderContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram unmultiplyProgram;

	@Inject
	public AlphaChannelSupportImpl(
			RenderContext context, IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;

		unmultiplyProgram = shaders.getProgram(AlphaChannelSupportImpl.class, "UNMULTIPLY");
	}

	@ShaderSource
	public static final String[] UNMULTIPLY = {
		"uniform sampler2D texture;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	gl_FragColor = (color.a == 0.0) ? vec4(0.0) : vec4(color.rgb/color.a, color.a);",
		"}",
	};

	public IVideoBuffer unmultiply(IVideoBuffer buffer) {
		Set<GLUniformData> uniforms = Util.newSet();
		uniforms.add(new GLUniformData("texture", 0));
		return support.useShaderProgram(unmultiplyProgram, uniforms, null, buffer);
	}

	private void blend(
			final IVideoBuffer buffer, final Color color, final int equation,
			final int srcRGBFactor, final int dstRGBFactor,
			final int srcAlphaFactor, final int dstAlphaFactor) {

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();

				gl.glColor4f((float)color.r, (float)color.g, (float)color.b, (float)color.a);

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendEquation(equation);
				gl.glBlendFuncSeparate(srcRGBFactor, dstRGBFactor, srcAlphaFactor, dstAlphaFactor);

				support.ortho2D(buffer);
				support.quad2D(buffer);
			}
		};

		int pushAttribs = GL2.GL_CURRENT_BIT | GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

		support.useFramebuffer(operation, pushAttribs, buffer);
	}

	public void premultiply(IVideoBuffer buffer) {
		blend(buffer, Color.WHITE, GL2.GL_FUNC_ADD,
				GL2.GL_ZERO, GL2.GL_DST_ALPHA, GL2.GL_ZERO, GL2.GL_ONE);
	}

	public void discardAlpha(IVideoBuffer buffer) {
		blend(buffer, Color.WHITE, GL2.GL_FUNC_ADD,
				GL2.GL_ZERO, GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ZERO);
	}

	public void alphaToGrayscale(IVideoBuffer buffer, Color color) {
		blend(buffer, color, GL2.GL_FUNC_ADD,
				GL2.GL_DST_ALPHA, GL2.GL_ZERO, GL2.GL_ONE, GL2.GL_ZERO);
	}

	public void addColorMatte(IVideoBuffer buffer, Color color) {
		blend(buffer, color, GL2.GL_FUNC_ADD,
				GL2.GL_ONE_MINUS_DST_ALPHA, GL2.GL_ONE, GL2.GL_ZERO, GL2.GL_ONE);
	}

	public void removeColorMatte(IVideoBuffer buffer, Color color) {
		blend(buffer, color, GL2.GL_FUNC_REVERSE_SUBTRACT,
				GL2.GL_ONE_MINUS_DST_ALPHA, GL2.GL_ONE, GL2.GL_ZERO, GL2.GL_ONE);
	}

}
