/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.LinkedList;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ICamera;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.internal.services.VideoEffectPipelineImpl.VideoEffectContextImpl;
import ch.kuramo.javie.core.services.QCIntegrationSupport;

public class VideoEffectContextProxy
		extends AbstractEffectContextProxy<IVideoEffectContext>
		implements IVideoEffectContext, QCIntegrationSupport {

	private static final ThreadLocal<LinkedList<VideoEffectContextImpl>>
			_threadLocal = new ThreadLocal<LinkedList<VideoEffectContextImpl>>();

	static void push(VideoEffectContextImpl impl) {
		getStack(true).addLast(impl);
	}

	static void pop() {
		LinkedList<VideoEffectContextImpl> stack = getStack(false);

		stack.removeLast();

		if (stack.isEmpty()) {
			_threadLocal.remove();
		}
	}

	private static LinkedList<VideoEffectContextImpl> getStack(boolean create) {
		LinkedList<VideoEffectContextImpl> stack = _threadLocal.get();
		if (stack == null) {
			if (create) {
				stack = Util.newLinkedList();
				_threadLocal.set(stack);
			} else {
				throw new IllegalStateException("no stack is found");
			}
		} else if (stack.isEmpty()) {
			// popメソッドにおいて、スタックが空になったときにはスレッドローカルからスタックを取り除いているので、
			// ここで空のスタックが返されることは無い。
			throw new IllegalStateException("stack is empty");
		}
		return stack;
	}

	protected VideoEffectContextImpl getImpl() {
		return getStack(false).getLast();
	}

	public VideoBounds getPreviousBounds() {
		return getImpl().getPreviousBounds();
	}

	public IVideoBuffer doPreviousEffect() {
		return getImpl().doPreviousEffect();
	}

	public ICamera getCamera() {
		return getImpl().getCamera();
	}


	// QCIntegrationSupport は IVideoEffectContext とライフサイクルが同じなので、
	// IVideoEffectContext のライフサイクル管理に便乗して QCIntegrationSupport を管理している。

	private static final QCIntegrationSupport NULL_QCIS = new QCIntegrationSupport() {
		private final Object[] EMPTY = new Object[0];
		public void clear() { }
		public void setSyncWithCamera() { }
		public boolean isSyncWithCamera() { return false; }
		public void setValueForInputKey(String key, Boolean value) { }
		public void setValueForInputKey(String key, Integer value) { }
		public void setValueForInputKey(String key, Double value) { }
		public void setValueForInputKey(String key, String value) { }
		public void setValueForInputKey(String key, Color value) { }
		public Object[] getInputKeyValues() { return EMPTY; }
	};

	private QCIntegrationSupport getQCISImpl() {
		// IVideoEffectContext が存在しない場合 (.qtz を MediaPlayer で開いた場合) もあるので、そのチェックが必要。
		if (_threadLocal.get() != null) {
			return getImpl().getQCISImpl();
		}
		return NULL_QCIS;
	}

	public void clear() {
		getQCISImpl().clear();
	}

	public void setSyncWithCamera() {
		getQCISImpl().setSyncWithCamera();
	}

	public boolean isSyncWithCamera() {
		return getQCISImpl().isSyncWithCamera();
	}

	public void setValueForInputKey(String key, Boolean value) {
		getQCISImpl().setValueForInputKey(key, value);
	}

	public void setValueForInputKey(String key, Integer value) {
		getQCISImpl().setValueForInputKey(key, value);
	}

	public void setValueForInputKey(String key, Double value) {
		getQCISImpl().setValueForInputKey(key, value);
	}

	public void setValueForInputKey(String key, String value) {
		getQCISImpl().setValueForInputKey(key, value);
	}

	public void setValueForInputKey(String key, Color value) {
		getQCISImpl().setValueForInputKey(key, value);
	}

	public Object[] getInputKeyValues() {
		return getQCISImpl().getInputKeyValues();
	}

}
