﻿/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "stdafx.h"
#include "DirectShowInputJNI.h"
#include "DirectShowInput.h"


class PrimitiveArrayCritical
{
public:
	PrimitiveArrayCritical(JNIEnv* env, jarray array) : mEnv(env), mArray(array), mBuffer(NULL) { }
	~PrimitiveArrayCritical() { Release(); }

	operator void* ()
	{
		if (mBuffer == NULL) {
			mBuffer = mEnv->GetPrimitiveArrayCritical(mArray, NULL);
		}
		return mBuffer;
	}

	void Release()
	{
		if (mBuffer != NULL) {
			mEnv->ReleasePrimitiveArrayCritical(mArray, mBuffer, 0);
		}
	}

private:
	JNIEnv*	mEnv;
	jarray	mArray;
	void*	mBuffer;
};

class JStringToWSTR
{
public:
	JStringToWSTR(JNIEnv* env, jstring jstr) : mWSTR(NULL)
	{
		jsize jstrLen = env->GetStringLength(jstr);
		mWSTR = (LPWSTR) malloc(sizeof(jchar) * (jstrLen + 1));

		const jchar* jchars = env->GetStringChars(jstr, NULL);
		memcpy(mWSTR, jchars, sizeof(jchar) * jstrLen);
		mWSTR[jstrLen] = 0;

		env->ReleaseStringChars(jstr, jchars);
	}

	~JStringToWSTR()
	{
		if (mWSTR != NULL) {
			free(mWSTR);
			mWSTR = NULL;
		}
	}

	operator LPWSTR () { return mWSTR; }

private:
	LPWSTR	mWSTR;
};


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
	return JNI_VERSION_1_4;
}

JNIEXPORT jlongArray JNICALL Java_ch_kuramo_javie_core_internal_WindowsDirectShowSource_openVideo
  (JNIEnv* env, jobject jthis, jstring jfilename)
{
	HRESULT hr;

	hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
	if (FAILED(hr)) {
		return NULL;
	}

	VideoInput* videoInput = new VideoInput();
	if (videoInput == NULL) {
		//hr = E_OUTOFMEMORY;
		return NULL;
	}

	hr = videoInput->InitWithFile(JStringToWSTR(env, jfilename));
	if (FAILED(hr)) {
		delete videoInput;
		return NULL;
	}

	jlong buf[6] = { reinterpret_cast<jlong>(videoInput), videoInput->GetWidth(), videoInput->GetHeight(),
					videoInput->GetDuration(), videoInput->GetFrameDuration(), videoInput->GetFourCC() };
	jlongArray result = env->NewLongArray(6);
	env->SetLongArrayRegion(result, 0, 6, buf);
	return result;
}

JNIEXPORT jlongArray JNICALL Java_ch_kuramo_javie_core_internal_WindowsDirectShowSource_openAudio
  (JNIEnv* env, jobject jthis, jstring jfilename)
{
	HRESULT hr;

	hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
	if (FAILED(hr)) {
		return NULL;
	}

	AudioInput* audioInput = new AudioInput();
	if (audioInput == NULL) {
		//hr = E_OUTOFMEMORY;
		return NULL;
	}

	hr = audioInput->InitWithFile(JStringToWSTR(env, jfilename));
	if (FAILED(hr)) {
		delete audioInput;
		return NULL;
	}

	jlong buf[6] = { reinterpret_cast<jlong>(audioInput), audioInput->GetDuration(),
					audioInput->GetSampleRate(), audioInput->GetSampleSizeInBits(),
					audioInput->GetChannels(), audioInput->IsFloat() };
	jlongArray result = env->NewLongArray(6);
	env->SetLongArrayRegion(result, 0, 6, buf);
	return result;
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_WindowsDirectShowSource_closeVideo
  (JNIEnv* env, jobject jthis, jlong videoInputAddress)
{
	VideoInput* videoInput = reinterpret_cast<VideoInput*>(videoInputAddress);
	delete videoInput;

	CoUninitialize();
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_WindowsDirectShowSource_closeAudio
  (JNIEnv* env, jobject jthis, jlong audioInputAddress)
{
	AudioInput* audioInput = reinterpret_cast<AudioInput*>(audioInputAddress);
	delete audioInput;

	CoUninitialize();
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsDirectShowSource_frameImageAtTime
  (JNIEnv* env, jobject jthis, jlong videoInputAddress,
   jlong dsTime, jbyteArray byteArray)
{
	VideoInput* videoInput = reinterpret_cast<VideoInput*>(videoInputAddress);

	HRESULT hr = videoInput->FrameImageAtTime(
			dsTime, PrimitiveArrayCritical(env, byteArray));

	return hr;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_internal_WindowsDirectShowSource_audioChunkFromTime
  (JNIEnv* env, jobject jthis, jlong audioInputAddress,
   jlong dsTime, jobject byteArrayObj, jint offset, jint size)
{
	AudioInput* audioInput = reinterpret_cast<AudioInput*>(audioInputAddress);

	HRESULT hr = audioInput->FillBuffer(
			dsTime, (char*)(void*)PrimitiveArrayCritical(env, (jarray)byteArrayObj) + offset, size);

	return hr;
}
