/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "StdAfx.h"
#include "ForcibleARGB32.h"


// {21B07E37-0AF2-4bf6-8BA6-D693C5257236}
DEFINE_GUID(CLSID_ForcibleARGB32, 
0x21b07e37, 0xaf2, 0x4bf6, 0x8b, 0xa6, 0xd6, 0x93, 0xc5, 0x25, 0x72, 0x36);


/**********************************************
 *
 *  ForcibleARGB32
 *
 **********************************************/

ForcibleARGB32::ForcibleARGB32(LPUNKNOWN unk, HRESULT* hr)
	: CTransInPlaceFilter(NAME("ForcibleARGB32"), unk, CLSID_ForcibleARGB32, hr, false)
{
}

ForcibleARGB32::~ForcibleARGB32()
{
}

CBasePin* ForcibleARGB32::GetPin(int n)
{
	HRESULT hr = S_OK;

	if (m_pInput == NULL) {
		m_pInput = new ForcibleARGB32InputPin(NAME("ForcibleARGB32 input pin"), this, &hr, L"Input");
		ASSERT(SUCCEEDED(hr));
	}

	if (m_pInput != NULL && m_pOutput == NULL) {
		m_pOutput = new ForcibleARGB32OutputPin(NAME("ForcibleARGB32 output pin"), this, &hr, L"Output");
		ASSERT(SUCCEEDED(hr));
		if (m_pOutput == NULL) {
			delete m_pInput;
			m_pInput = NULL;
		}
	}

	ASSERT(n>=0 && n<=1);
	if (n == 0) {
		return m_pInput;
	} else if (n == 1) {
		return m_pOutput;
	} else {
		return NULL;
	}
}

HRESULT ForcibleARGB32::CheckInputType(const CMediaType* mtIn)
{
	if (!IsEqualGUID(*mtIn->FormatType(), FORMAT_VideoInfo)) {
		return VFW_E_TYPE_NOT_ACCEPTED;
	}

	if (!IsEqualGUID(*mtIn->Type(), MEDIATYPE_Video)) {
		return VFW_E_TYPE_NOT_ACCEPTED;
	}

	if (!IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_RGB32) && !IsEqualGUID(*mtIn->Subtype(), MEDIASUBTYPE_ARGB32)) {
		return VFW_E_TYPE_NOT_ACCEPTED;
	}

	return S_OK;
}

HRESULT ForcibleARGB32::CheckTransform(const CMediaType* mtIn, const CMediaType* mtOut)
{
	if (!IsEqualGUID(*mtOut->FormatType(), FORMAT_VideoInfo)) {
		return VFW_E_TYPE_NOT_ACCEPTED;
	}

	if (!IsEqualGUID(*mtOut->Type(), MEDIATYPE_Video)) {
		return VFW_E_TYPE_NOT_ACCEPTED;
	}

	if (!IsEqualGUID(*mtOut->Subtype(), MEDIASUBTYPE_ARGB32)) {
		return VFW_E_TYPE_NOT_ACCEPTED;
	}

	return S_OK;
}

HRESULT ForcibleARGB32::GetMediaType(int position, CMediaType* mediaType)
{
	if (position < 0) {
		return E_INVALIDARG;
	} else if (position > 0) {
		return VFW_S_NO_MORE_ITEMS;
	}

	*mediaType = m_pInput->CurrentMediaType();
	mediaType->SetSubtype(&MEDIASUBTYPE_ARGB32);

	return S_OK;
}

HRESULT ForcibleARGB32::CompleteConnect(PIN_DIRECTION dir, IPin* receivePin)
{
	if (!m_pGraph) {
		return VFW_E_NOT_IN_GRAPH;
	}

	if (dir == PINDIR_OUTPUT && m_pInput->IsConnected()) {
		return ReconnectPin(m_pInput, &m_pInput->CurrentMediaType());
	}

	return NOERROR;
}

HRESULT ForcibleARGB32::Transform(IMediaSample *sample)
{
	AM_MEDIA_TYPE* mediaType;
	if (UsingDifferentAllocators() && sample->GetMediaType(&mediaType) == S_OK) {
		mediaType->subtype = MEDIASUBTYPE_ARGB32;
		DeleteMediaType(mediaType);
	}

	return S_OK;
}

/**********************************************
 *
 *  ForcibleARGB32InputPin
 *
 **********************************************/

ForcibleARGB32InputPin::ForcibleARGB32InputPin(
	LPCTSTR objectName, ForcibleARGB32* filter, HRESULT* hr, LPCWSTR pinName)
	: CTransInPlaceInputPin(objectName, filter, hr, pinName)
{
}

ForcibleARGB32InputPin::~ForcibleARGB32InputPin()
{
}

STDMETHODIMP ForcibleARGB32InputPin::EnumMediaTypes(IEnumMediaTypes** enumMediaTypes)
{
	return CTransformInputPin::EnumMediaTypes(enumMediaTypes);
}

HRESULT ForcibleARGB32InputPin::CheckMediaType(const CMediaType* mediaType)
{
	return CTransformInputPin::CheckMediaType(mediaType);
}

/**********************************************
 *
 *  ForcibleARGB32OutputPin
 *
 **********************************************/

ForcibleARGB32OutputPin::ForcibleARGB32OutputPin(
	LPCTSTR objectName, ForcibleARGB32* filter, HRESULT* hr, LPCWSTR pinName)
	: CTransInPlaceOutputPin(objectName, filter, hr, pinName)
{
}

ForcibleARGB32OutputPin::~ForcibleARGB32OutputPin()
{
}

STDMETHODIMP ForcibleARGB32OutputPin::EnumMediaTypes(IEnumMediaTypes** enumMediaTypes)
{
	return CTransformOutputPin::EnumMediaTypes(enumMediaTypes);
}

HRESULT ForcibleARGB32OutputPin::CheckMediaType(const CMediaType* mediaType)
{
	return CTransformOutputPin::CheckMediaType(mediaType);
}
