/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.CompoundBlur", category=Categories.BLUR_AND_SHARPEN)
public class CompoundBlur {

	public enum MapPlacement { STRETCH, CENTER, TILE, MIRRORED_TILE }

	@Property
	private IAnimatableLayerReference blurMap;

	@Property(value="0", min="0"/*, max="500"*/)
	private IAnimatableDouble maximumBlur;

	@Property("STRETCH")
	private IAnimatableEnum<MapPlacement> mapPlacement;

	@Property
	private IAnimatableBoolean invertBlur;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IShaderRegistry shaders;

	private final IShaderProgram blurConvolutionProgram;

	@Inject
	public CompoundBlur(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.shaders = shaders;

		blurConvolutionProgram = shaders.getProgram(CompoundBlur.class, "BLUR_CONVOLUTION");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();
		double maximumBlur = resolution.scale(context.value(this.maximumBlur));
		if (maximumBlur == 0) {
			return source;
		}

		IVideoBuffer blurMap = null;
		IVideoBuffer mipmap = null;
		try {
			blurMap = context.getLayerVideoFrame(this.blurMap);
			if (blurMap == null) {
				IVideoBuffer result = source;
				source = null;
				return result;
			}

			MapPlacement mapPlacement = context.value(this.mapPlacement);
			boolean invertBlur = context.value(this.invertBlur);

			Quality quality = context.getQuality();
			VideoBounds blurMapBounds = blurMap.getBounds();
			blurMap.setTextureFilter((quality == Quality.DRAFT || resolution.scale < 1)
											? TextureFilter.NEAREST
									: (quality == Quality.BEST && mapPlacement == MapPlacement.STRETCH &&
													(blurMapBounds.width > bounds.width
													|| blurMapBounds.height > bounds.height))
											? TextureFilter.MIPMAP : TextureFilter.LINEAR);

			mipmap = createBlurMipmap(source);

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("mipmap", 1));
			uniforms.add(new GLUniformData("blurMap", 2));
			uniforms.add(new GLUniformData("maximumBlur", (float)maximumBlur));
			if (mapPlacement != MapPlacement.STRETCH) {
				uniforms.add(new GLUniformData("size0", 2, toFloatBuffer(bounds.width, bounds.height)));
				uniforms.add(new GLUniformData("size1", 2, toFloatBuffer(blurMapBounds.width, blurMapBounds.height)));
				switch (mapPlacement) {
					case TILE:
						blurMap.setTextureWrapMode(TextureWrapMode.REPEAT);
						break;
					case MIRRORED_TILE:
						blurMap.setTextureWrapMode(TextureWrapMode.MIRRORED_REPEAT);
						break;
				}
			}

			IShaderProgram program = getCompoundBlurProgram(mapPlacement, invertBlur);
			return support.useShaderProgram(program, uniforms, null, source, mipmap, blurMap);

		} finally {
			if (blurMap != null) blurMap.dispose();
			if (mipmap != null) mipmap.dispose();
			if (source != null) source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	private IVideoBuffer createBlurMipmap(IVideoBuffer source) {
		IVideoBuffer mipmap = null;
		IVideoBuffer tmpbuf = null;
		GL2 gl = context.getGL().getGL2();
		gl.glPushAttrib(GL2.GL_TEXTURE_BIT | GL2.GL_COLOR_BUFFER_BIT);
		try {
			mipmap = support.createVideoBuffer(source.getBounds());
			support.copy(source, mipmap);
			mipmap.setTextureFilter(TextureFilter.MIPMAP);

			gl.glActiveTexture(GL2.GL_TEXTURE0);

			int[] params = new int[2];
			for (int level = 0; ; ++level) {
				gl.glBindTexture(GL2.GL_TEXTURE_2D, mipmap.getTexture());
				gl.glGetTexLevelParameteriv(GL2.GL_TEXTURE_2D, level, GL2.GL_TEXTURE_WIDTH, params, 0);
				gl.glGetTexLevelParameteriv(GL2.GL_TEXTURE_2D, level, GL2.GL_TEXTURE_HEIGHT, params, 1);

				VideoBounds b = new VideoBounds(params[0], params[1]);
				tmpbuf = support.createVideoBuffer(b);

				// ミップマップレベルの縮小比率と合わせると、
				// ブラーの半径は pow(2,level)*(2 or 4 or 8) になる。
				double radius = (level == 0) ? 2 : (level == 1) ? 4 : 8;
				float[] kernel = blurSupport.createGaussianBlurKernel(radius);

				blurConvolution(gl, kernel, true, b, mipmap.getTexture(), level, tmpbuf.getTexture(), 0);
				blurConvolution(gl, kernel, false, b, tmpbuf.getTexture(), 0, mipmap.getTexture(), level);

				tmpbuf.dispose();
				tmpbuf = null;

				if (b.width == 1 && b.height == 1) {
					break;
				}
			}

			IVideoBuffer result = mipmap;
			mipmap = null;
			return result;

		} finally {
			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, 0, 0);
			gl.glPopAttrib();

			if (mipmap != null) mipmap.dispose();
			if (tmpbuf != null) tmpbuf.dispose();
		}
	}

	private void blurConvolution(
			final GL2 gl, float[] kernel, boolean horz, final VideoBounds b,
			int srcTex, int srcLevel, int dstTex, int dstLevel) {

		int ksize = kernel.length;
		float[] offset = new float[ksize*2];
		if (horz) {
			for (int i = 0; i < ksize; ++i) {
				offset[i*2] = (float)(i - ksize/2) / b.width;
				offset[i*2+1] = 0;
			}
		} else {
			for (int i = 0; i < ksize; ++i) {
				offset[i*2] = 0;
				offset[i*2+1] = (float)(i - ksize/2) / b.height;
			}
		}

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, dstTex, dstLevel);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);
		gl.glBindTexture(GL2.GL_TEXTURE_2D, srcTex);

		final Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("ksize", ksize));
		uniforms.add(new GLUniformData("kernel[0]", 1, FloatBuffer.wrap(kernel)));
		uniforms.add(new GLUniformData("offset[0]", 2, FloatBuffer.wrap(offset)));
		uniforms.add(new GLUniformData("lod", (float)srcLevel));

		blurConvolutionProgram.useProgram(new Runnable() {
			public void run() {
				for (GLUniformData data : uniforms) {
					data.setLocation(blurConvolutionProgram.getUniformLocation(data.getName()));
					gl.glUniform(data);
				}
				support.ortho2D(b);
				support.quad2D(b, new double[][] { {0, 0}, {1, 0}, {1, 1}, {0, 1} });
			}
		});
	}

	@ShaderSource
	public static final String[] BLUR_CONVOLUTION = {
		"uniform sampler2D texture;",
		"uniform int ksize;",
		"uniform float kernel[17];",
		"uniform vec2 offset[17];",
		"uniform float lod;",
		"",
		"void main(void)",
		"{",
		"	vec2 texCoord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	for (int i = 0; i < ksize; ++i) {",
		"		sum += kernel[i] * texture2DLod(texture, texCoord + offset[i], lod);",
		"	}",
		"	gl_FragColor = sum;",
		"}"
	};

	private IShaderProgram getCompoundBlurProgram(MapPlacement mapPlacement, boolean invert) {
		boolean stretch = (mapPlacement == MapPlacement.STRETCH);
		String programName = CompoundBlur.class.getName()
							+ (stretch ? ".STRETCH" : "") + (invert ? ".INVERT" : ""); 
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createCompoundBlurSource(stretch, invert);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, null, source);
		}
		return program;
	}

	private String[] createCompoundBlurSource(boolean stretch, boolean invert) {
		return new String[] {
	  stretch ? "#define STRETCH" : "",
	   invert ? "#define INVERT" : "",
				"",
				"uniform sampler2D texture;",
				"uniform sampler2D mipmap;",
				"uniform sampler2D blurMap;",
				"uniform float maximumBlur;",
				"",
				"#ifndef STRETCH",
				"	uniform vec2 size0;",
				"	uniform vec2 size1;",
				"#endif",
				"",
				"const vec3 lumaVec = vec3(0.299, 0.587, 0.114);",
				"",
				"vec4 mixLod(sampler2D sampler, vec2 coord, float lod) {",
				"	float lod1 = floor(lod);",
				"	return mix(texture2DLod(sampler, coord, lod1),",
				"			texture2DLod(sampler, coord, lod1+1.0), lod-lod1);",
				"}",
				"",
				"void main(void)",
				"{",
				"#ifdef STRETCH",
				"	vec2 tc = gl_TexCoord[0].st;",
				"#else",
				"	vec2 tc = ((gl_FragCoord.xy-0.5*size0) / size1) + vec2(0.5);",
				"#endif",
				"",
				"#ifdef INVERT",
				"	float blur = (1.0-dot(texture2D(blurMap, tc).rgb, lumaVec)) * maximumBlur;",
				"#else",
				"	float blur = dot(texture2D(blurMap, tc).rgb, lumaVec) * maximumBlur;",
				"#endif",
				"",
				"#ifndef STRETCH",
				"	tc = gl_TexCoord[0].st;",
				"#endif",
				"",
				"	if (blur > 32.0) {",
				"		float lod = log2(0.25*blur) - 1.0;",
				"		gl_FragColor = mixLod(mipmap, tc, lod);",
				"",
				"	} else if (blur > 2.0) {",
				"		float lod = 0.5*(log2(blur) - 1.0);",
				"		gl_FragColor = mixLod(mipmap, tc, lod);",
				"",
				"	} else if (blur > 0.0) {",
				"		gl_FragColor = mix(texture2D(texture, tc),",
				"						texture2DLod(mipmap, tc, 0.0), 0.5*blur);",
				"",
				"	} else {",
				"		gl_FragColor = texture2D(texture, tc);",
				"	}",
				"}"
		};
	}

}
