/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.colorCorrection;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Monochrome", category=Categories.COLOR_CORRECTION)
public class Monochrome {

	@ShaderSource
	public static final String[] MONOCHROME = {
		"uniform sampler2D texture;",
		"uniform vec3 highlight;",
		"uniform vec3 shadow;",
		"uniform float blend;",
		"",
		"const vec3 yvec = vec3(0.299, 0.587, 0.114);",
		"",
		"void main(void)",
		"{",
		"	vec4 src = texture2D(texture, gl_TexCoord[0].st);",
		"	float y = (src.a != 0.0) ? dot(src.rgb/src.a, yvec) : 0.0;",
		"	vec3 rgb = shadow*(1.0-y) + highlight*y;",
		"	float a = src.a*(1.0-blend);",
		"	gl_FragColor = src*blend + vec4(rgb*a, a);",
		"}"
	};


	// TODO 「セピア」「緑-黒」などをプリセットから選べるようにするといいかも。
	//private static final Color SEPIA_HIGHLIGHT = new Color(0.942, 0.785, 0.569);


	@Property("1,1,1")
	private IAnimatableColor highlight;

	@Property("0,0,0")
	private IAnimatableColor shadow;

	@Property(min="0", max="100")
	private IAnimatableDouble blendWithOriginal;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	@Inject
	public Monochrome(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		program = shaders.getProgram(Monochrome.class, "MONOCHROME");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		double blend = context.value(blendWithOriginal);
		if (blend == 100) {
			return input;
		}

		Color highlight = context.value(this.highlight);
		Color shadow = context.value(this.shadow);

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("highlight", 3, FloatBuffer.wrap(colorToFloat3(highlight))));
			uniforms.add(new GLUniformData("shadow", 3, FloatBuffer.wrap(colorToFloat3(shadow))));
			uniforms.add(new GLUniformData("blend", (float)(blend/100)));

			return support.useShaderProgram(program, uniforms, null, input);

		} finally {
			input.dispose();
		}
	}

	private float[] colorToFloat3(Color color) {
		return new float[] { (float)color.r, (float)color.g, (float)color.b };
	}

}
