/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.DifferenceMatte", category=Categories.KEYING)
public class DifferenceMatte {

	public enum IfLayerSizesDiffer {
		CENTER,
		STRETCH_TO_FIT
	}

	public enum Output {
		RESULT,
		SOURCE_ONLY,
		MATTE_ONLY
	}

	@Property
	private IAnimatableLayerReference differenceLayer;

	@Property("CENTER")
	private IAnimatableEnum<IfLayerSizesDiffer> ifLayerSizesDiffer;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble matchingTolerance;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble matchingSoftness;

	@Property(value="0", min="0", max="500")
	private IAnimatableDouble blurBeforeDifference;

	@Property("RESULT")
	private IAnimatableEnum<Output> output;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IShaderRegistry shaders;

	@Inject
	public DifferenceMatte(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.shaders = shaders;
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		final VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Output output = context.value(this.output);
		if (output == Output.SOURCE_ONLY) {
			return source;
		}

		IVideoBuffer diffLayer = null;
		IVideoBuffer[] blurred = null;
		try {
			diffLayer = context.getLayerVideoFrame(differenceLayer);
			if (diffLayer == null) {
				IVideoBuffer result = source;
				source = null;
				return result;
			}

			final boolean stretchToFit = (context.value(ifLayerSizesDiffer) == IfLayerSizesDiffer.STRETCH_TO_FIT);
			final VideoBounds diffBounds = diffLayer.getBounds();

			if (stretchToFit && !diffBounds.equals(bounds)) {
				IVideoBuffer old = diffLayer;
				diffLayer = stretchToFit(old, bounds);
				old.dispose();
			}

			Resolution resolution = context.getVideoResolution();
			double blur = resolution.scale(context.value(blurBeforeDifference));
			if (blur > 0) {
				blurred = new IVideoBuffer[2];
				blurred[0] = blurSupport.gaussianBlur(source, blur, BlurDimensions.BOTH, true, true);
				blurred[1] = blurSupport.gaussianBlur(diffLayer, blur, BlurDimensions.BOTH, true, true);
			}

			double tolerance = context.value(matchingTolerance) * Math.sqrt(3) / 100;
			double softness = context.value(matchingSoftness) / 100;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture1", 0));
			uniforms.add(new GLUniformData("texture2", 1));
			uniforms.add(new GLUniformData("t", (float)tolerance));
			uniforms.add(new GLUniformData("s", (float)((Math.sqrt(3)-tolerance)*softness)));

			if (output != Output.MATTE_ONLY) {
				uniforms.add(new GLUniformData("source", 2));
			}

			IVideoBuffer input1, input2;
			if (blurred != null) {
				input1 = blurred[0];
				input2 = blurred[1];
			} else {
				input1 = source;
				input2 = diffLayer;
			}

			Runnable operation = new Runnable() {
				public void run() {
					double[][][] texCoords;
					if (stretchToFit) {
						texCoords = new double[][][] { {{0,0},{1,0},{1,1},{0,1}}, {{0,0},{1,0},{1,1},{0,1}} };
					} else {
						double x = (diffBounds.width-bounds.width)*0.5 / diffBounds.width;
						double y = (diffBounds.height-bounds.height)*0.5 / diffBounds.height;
						texCoords = new double[][][] { {{0,0},{1,0},{1,1},{0,1}}, {{x,y},{1-x,y},{1-x,1-y},{x,1-y}} };
					}
					support.ortho2D(bounds);
					support.quad2D(bounds, texCoords);
				}
			};

			switch (output) {
				case RESULT:
					return support.useShaderProgram(getProgram(false), uniforms, operation, 0, null, input1, input2, source);
				case MATTE_ONLY:
					return support.useShaderProgram(getProgram(true), uniforms, operation, 0, null, input1, input2);
				default:
					// ここには到達しない。
					throw new Error();
			}

		} finally {
			if (source != null) source.dispose();
			if (diffLayer != null) diffLayer.dispose();
			if (blurred != null) {
				if (blurred[0] != null) blurred[0].dispose();
				if (blurred[1] != null) blurred[1].dispose();
			}
		}
	}

	private IVideoBuffer stretchToFit(IVideoBuffer diffLayer, final VideoBounds bounds) {
		// diffLayerのフィルタを変更するが、diffLayerはこのメソッドの呼び出しのあとすぐにdisposeされるので元に戻さない。
		//TextureFilter filter = diffLayer.getTextureFilter();
		IVideoBuffer buffer = null;
		try {
			buffer = support.createVideoBuffer(bounds);

			if (context.getQuality() == Quality.DRAFT || context.getVideoResolution().scale < 1) {
				diffLayer.setTextureFilter(TextureFilter.NEAREST);
			} else {
				VideoBounds diffBounds = diffLayer.getBounds();
				if (bounds.width < diffBounds.width || bounds.height < diffBounds.height) {
					diffLayer.setTextureFilter(TextureFilter.MIPMAP);
				} else {
					diffLayer.setTextureFilter(TextureFilter.LINEAR);
				}
			}

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(bounds);
					support.quad2D(bounds, new double[][] { {0,0},{1,0},{1,1},{0,1} });
				}
			};

			support.useFramebuffer(operation, 0, buffer, diffLayer);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			//diffLayer.setTextureFilter(filter);
		}
	}

	private IShaderProgram getProgram(boolean matteOnly) {
		String programName = DifferenceMatte.class.getName()
								+ (matteOnly ? ".MATTE_ONLY" : ".DIFFERENCE_MATTE");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createProgramSource(matteOnly);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, null, source);
		}
		return program;
	}

	private String[] createProgramSource(boolean matteOnly) {
		boolean mo = matteOnly;
		return new String[] {
		   mo ? "#define MATTE_ONLY" : "",
				"",
				"uniform sampler2D texture1;",
				"uniform sampler2D texture2;",
				"uniform float t;",
				"uniform float s;",
				"",
				"#ifndef MATTE_ONLY",
				"	uniform sampler2D source;",
				"#endif",
				"",
				"void main(void)",
				"{",
				"	vec2 tc1 = gl_TexCoord[0].st;",
				"	vec4 color1 = texture2D(texture1, tc1);",
				"	color1.rgb = (color1.a > 0.0) ? color1.rgb/color1.a : vec3(0.0);",
				"",
				"	vec2 tc2 = gl_TexCoord[1].st;",
				"	vec4 color2 = texture2D(texture2, tc2);",
				"	color2.rgb = (color2.a > 0.0) ? color2.rgb/color2.a : vec3(0.0);",
				"",
				"	float d = distance(color1.rgb, color2.rgb);",
				"	d = clamp((d-t)/s, 0.0, 1.0);",
				"",
				"#ifdef MATTE_ONLY",
				"	gl_FragColor = vec4(d);",
				"#else",
				"	gl_FragColor = texture2D(source, tc1) * d;",
				"#endif",
				"}"
		};
	}

}
