/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.noiseGrain;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.vecmath.Matrix3d;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.FractalNoise", category=Categories.NOISE_AND_GRAIN)
public class FractalNoise {

	@ShaderSource(program=false)
	public static final String noise = "noise.frag";

	@ShaderSource(attach="noise")
	public static final String[] FRACTAL_NOISE = {
		"uniform int fractalType;",
		"uniform float contrast;",
		"uniform float brightness;",
		"uniform int overflow;",
		"uniform int complexity;",
		"uniform float influences[20];",
		"uniform mat3 matrices[20];",
		"uniform float evolution;",
		"uniform float cycle;",
		"",
		"float noise(vec3 P);",
		"",
		"void main(void)",
		"{",
		"	float sum = 0.0;",
		"	vec3 pt = vec3(gl_FragCoord.st/128.0, 1.0);",
		"",
		"	if (fractalType == 0) {",
		"		if (cycle == 0.0) {",
		"			for (int i = 0; i < complexity; ++i) {",
		"				sum += influences[i]*noise(vec3((matrices[i]*pt).xy, evolution));",
		"			}",
		"		} else {",
		"			for (int i = 0; i < complexity; ++i) {",
		"				sum += influences[i]/cycle*(",
		"						(cycle-evolution)*noise(vec3((matrices[i]*pt).xy, evolution))",
		"					  + evolution*noise(vec3((matrices[i]*pt).xy, evolution-cycle)));",
		"			}",
		"		}",
		"		sum *= 2.0;",
		"	} else if (fractalType == 1) {",
		"		if (cycle == 0.0) {",
		"			for (int i = 0; i < complexity; ++i) {",
		"				sum += influences[i]*abs(noise(vec3((matrices[i]*pt).xy, evolution)));",
		"			}",
		"		} else {",
		"			for (int i = 0; i < complexity; ++i) {",
		"				sum += influences[i]/cycle*(",
		"						(cycle-evolution)*abs(noise(vec3((matrices[i]*pt).xy, evolution)))",
		"					  + evolution*abs(noise(vec3((matrices[i]*pt).xy, evolution-cycle))));",
		"			}",
		"		}",
		"		sum = sum * 2.0 - 1.0;",
		"	}",
		"",
		"	float t = sum * contrast + brightness;",
		"",
		"	if (overflow == 0) {",									// CLIP
		"		t = clamp(0.5+0.5*t, 0.0, 1.0);",
		"	} else if (overflow == 1) {",							// SOFT_CLAMP
		"		t = 0.5+0.3183099*atan(t);",
		"	} else if (overflow == 2) {",							// WRAP_BACK
		"		t = abs(0.5+0.5*t);",
		"		t = 1.0 - abs(t - 2.0*float(int(t*0.5)) - 1.0);",
		"	} else {",												// HDR
		"		t = 0.5+0.5*t;",
		"	}",
		"	",
		"	gl_FragColor = vec4(t, t, t, 1.0);",
		"}",
	};


	public enum FractalType { BASIC, TURBULENT }

	public enum Overflow { CLIP, SOFT_CLAMP, WRAP_BACK, HDR }


	@Property
	private IAnimatableEnum<FractalType> fractalType;

//	@Property
//	private IAnimatableEnum<NoiseType> noiseType;

	@Property
	private IAnimatableBoolean invert;

	@Property(value="100", min="0", max="10000")
	private IAnimatableDouble contrast;

	@Property(value="0", min="-10000", max="10000")
	private IAnimatableDouble brightness;

	@Property("HDR")
	private IAnimatableEnum<Overflow> overflow;

	@Property
	private IAnimatableDouble rotation;

	@Property(value="100", min="1", max="10000")
	private IAnimatableVec2d scale;

	@Property
	private IAnimatableVec2d offset;

	@Property(value="6", min="1", max="20")
	private IAnimatableDouble complexity;

	@Property(value="70", min="0", max="10000")
	private IAnimatableDouble subInfluence;

	@Property(value="56", min="10", max="10000")
	private IAnimatableDouble subScale;

	@Property
	private IAnimatableDouble subRotation;

	@Property
	private IAnimatableVec2d subOffset;

	@Property
	private IAnimatableBoolean centerSubscale;

	@Property
	private IAnimatableDouble evolution;

	@Property(value="0", min="0", max="100")
	private IAnimatableInteger cycle;

	@Property(value="0", min="0", max="100000")
	private IAnimatableInteger randomSeed;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property("NORMAL")
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	private final IBlendSupport blendSupport;

	@Inject
	public FractalNoise(
			IVideoEffectContext context, IVideoRenderSupport support,
			IShaderRegistry shaders, IBlendSupport blendSupport) {

		this.context = context;
		this.support = support;

		program = shaders.getProgram(FractalNoise.class, "FRACTAL_NOISE");

		this.blendSupport = blendSupport;
		blendSupport.setProgramsClass(blendSupport.getPremultAndMatteClass(true));
		blendSupport.replace(BlendMode.HUE, shaders.getProgram(FractalNoise.class, "HUE"));
		blendSupport.replace(BlendMode.SATURATION, shaders.getProgram(FractalNoise.class, "SATURATION"));
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer original = context.doPreviousEffect();
		VideoBounds bounds = original.getBounds();
		if (bounds.isEmpty()) {
			return original;
		}

		IVideoBuffer fractal = null;
		try {
			fractal = doFractalNoise(bounds);

			BlendMode blendMode = context.value(this.blendMode);
			double opacity = context.value(this.opacity) / 100;
			return blendSupport.blend(fractal, original, null, blendMode, opacity, context);

		} finally {
			if (original != null) original.dispose();
			if (fractal != null) fractal.dispose();
		}
	}

	private IVideoBuffer doFractalNoise(VideoBounds bounds) {
		Resolution resolution = context.getVideoResolution();

		FractalType fractalType = context.value(this.fractalType);
		boolean invert = context.value(this.invert);
		double contrast = context.value(this.contrast) / (invert ? -100 : 100);
		double brightness = context.value(this.brightness) / 100;
		Overflow overflow = context.value(this.overflow);
		double rotation = Math.toRadians(context.value(this.rotation));
		Vec2d scale = resolution.scale(context.value(this.scale));
		Vec2d offset = resolution.scale(context.value(this.offset));
		double complexity = context.value(this.complexity);
		double subInfluence = context.value(this.subInfluence) / 100;
		double subScale = context.value(this.subScale);
		double subRotation = Math.toRadians(context.value(this.subRotation));
		Vec2d subOffset = context.value(this.subOffset);
		boolean centerSubscale = context.value(this.centerSubscale);
		double evolution = context.value(this.evolution);
		int cycle = context.value(this.cycle);
		int randomSeed = context.value(this.randomSeed);

		offset = new Vec2d(offset.x - bounds.x, offset.y - bounds.y);

		evolution += randomSeed*367;
		evolution /= 180;
		if (cycle != 0) {
			cycle *= 2;
			evolution %= cycle;
			if (evolution < 0) {
				evolution += cycle;
			}
		}


		double cosRot = Math.cos(rotation);
		double sinRot = Math.sin(rotation);
		Matrix3d transformMat = new Matrix3d(100/scale.x, 0.0, 0.0, 0.0, 100/scale.y, 0.0, 0.0, 0.0, 1.0);	// scale
		transformMat.mul(new Matrix3d(cosRot, sinRot, 0.0, -sinRot, cosRot, 0.0, 0.0, 0.0, 1.0));			// rotation
		transformMat.mul(new Matrix3d(1.0, 0.0, -offset.x/128, 0.0, 1.0, -offset.y/128, 0.0, 0.0, 1.0));	// offset

		double cosSubRot = Math.cos(subRotation);
		double sinSubRot = Math.sin(subRotation);
		Matrix3d subScaleMat1 = new Matrix3d(100/subScale, 0.0, 0.0, 0.0, 100/subScale, 0.0, 0.0, 0.0, 1.0);			// subScale  1回分
		Matrix3d subRotMat1 = new Matrix3d(cosSubRot, sinSubRot, 0.0, -sinSubRot, cosSubRot, 0.0, 0.0, 0.0, 1.0);		// subRotation  1回分
		Matrix3d subOffsetMat1 = new Matrix3d(1.0, 0.0, -subOffset.x/128, 0.0, 1.0, -subOffset.y/128, 0.0, 0.0, 1.0);	// subOffset  1回分

		Matrix3d subScaleMat = new Matrix3d();		// subScaleMat1 の累積
		subScaleMat.setIdentity();
		Matrix3d subRotMat = new Matrix3d();		// subRotMat1 の累積
		subRotMat.setIdentity();
		Matrix3d subOffsetMat = new Matrix3d();		// subOffsetMat1 の累積
		if (centerSubscale) {
			subOffsetMat.set(subOffsetMat1);
		} else {
			subOffsetMat.setIdentity();
		}

		int ceiledComplexity = (int)Math.ceil(complexity); 
		float[] influences = new float[ceiledComplexity];
		float infTotal = 0;
		float[] matrices = new float[9*ceiledComplexity];

		for (int i = 0; i < ceiledComplexity; ++i) {
			influences[i] = (float)Math.pow(subInfluence, i);
			if (i+1 > complexity) {
				influences[i] *= complexity-i;
			}

			// 複雑度をアニメーションさせたとき、より滑らかに変化する計算式の例
			//double t = complexity - i;
			//influences[i] = (float)(Math.pow(subInfluence, i)
			//		*(Math.atan(Math.PI/2*(t-1))+Math.atan(Math.PI/2))/(Math.PI/2+Math.atan(Math.PI/2)));

			infTotal += influences[i];

			Matrix3d mat = new Matrix3d(subOffsetMat);
			mat.mul(subScaleMat);
			mat.mul(subRotMat);
			mat.mul(transformMat);

			for (int j = 0; j < 9; ++j) {
				matrices[i*9+j] = (float)mat.getElement(j%3, j/3);
												// (j/3, j%3) ではないのは transpose しながら取得しているため。
			}

			subScaleMat.mul(subScaleMat1);
			subRotMat.mul(subRotMat1);

			if (!centerSubscale) {
				subOffsetMat.mul(new Matrix3d(
						1.0, 0.0, 179,		// TODO ここの値はこれでいいのか？
						0.0, 1.0, 179,		//      179は256以下の適当な素数。256がちょうど1周期分？
						0.0, 0.0, 1.0));
				subOffsetMat.mul(subOffsetMat1);
			}
		}
		for (int i = 0; i < ceiledComplexity; ++i) {
			influences[i] /= infTotal;
		}


		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("permTexture", 0));
		//uniforms.add(new GLUniformData("gradTexture", 1));
		uniforms.add(new GLUniformData("fractalType", fractalType.ordinal()));
		uniforms.add(new GLUniformData("contrast", (float)contrast));
		uniforms.add(new GLUniformData("brightness", (float)brightness));
		uniforms.add(new GLUniformData("overflow", overflow.ordinal()));
		uniforms.add(new GLUniformData("complexity", ceiledComplexity));
		uniforms.add(new GLUniformData("influences[0]", 1, FloatBuffer.wrap(influences)));
		uniforms.add(new GLUniformData("matrices[0]", 3, 3, FloatBuffer.wrap(matrices)));
		uniforms.add(new GLUniformData("evolution", (float)evolution));
		uniforms.add(new GLUniformData("cycle", (float)cycle));


		IVideoBuffer buffer = null;
		try {
			final IVideoBuffer output = buffer = context.createVideoBuffer(bounds);

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					int permTexture = NoiseTexture.getPermTexture(gl);

					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_2D, permTexture);
					//gl.glActiveTexture(GL2.GL_TEXTURE1);
					//gl.glBindTexture(GL2.GL_TEXTURE_2D, gradTexture);

					support.ortho2D(output);
					support.quad2D(output);
				}
			};

			support.useShaderProgram(program, uniforms, operation, GL2.GL_TEXTURE_BIT, output);

			buffer = null;
			return output;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	@ShaderSource(program=false)
	public static final String[] hsl2rgb = {
		"float hue2rgb(float t1, float t2, float hue)",
		"{",
		"	if (hue < 0.0) hue += 1.0;",
		"	else if (hue > 1.0) hue -= 1.0;",
		"",
		"	return (hue*6.0 < 1.0) ? t1 + (t2-t1)*6.0*hue :",
		"		   (hue*2.0 < 1.0) ? t2 :",
		"		   (hue*3.0 < 2.0) ? t1 + (t2-t1)*(2.0/3.0-hue)*6.0 :",
		"		                     t1;",
		"}",
		"",
		"vec3 hsl2rgb(vec3 hsl)",
		"{",
		"	float hue = hsl.x;",
		"	float sat = hsl.y;",
		"	float luma = hsl.z;",
		"	vec3 rgb;",
		"",
		"	if (sat == 0.0) {",
		"		rgb = vec3(luma);",
		"	} else {",
		"		float t2 = (luma < 0.5) ? luma*(1.0+sat) : luma+sat-luma*sat;",
		"		float t1 = luma*2.0 - t2;",
		"",
		"		rgb = vec3(",
		"				hue2rgb(t1, t2, hue+1.0/3.0),",
		"				hue2rgb(t1, t2, hue),",
		"				hue2rgb(t1, t2, hue-1.0/3.0));",
		"	}",
		"	return rgb;",
		"}",
	};

	private static String[] createHueOrSaturationBlendSource(boolean hue) {
		return new String[] {
				"uniform sampler2D texDst;",	// original
				"uniform sampler2D texSrc;",	// noise
				"uniform float opacity;",
				"",
				"const vec3 lumaVec = vec3(0.299, 0.587, 0.114);",
				"",
		  hue ? "vec3 hsl2rgb(vec3 hsl);" : "",
		  hue ? "float sat(vec3 color);" : "",
				"vec3 set_sat(vec3 color, float s);",
				"vec3 set_lum(vec3 color, float l);",
				"",
				"void main(void)",
				"{",
				"	vec2 texCoord = gl_TexCoord[0].st;",
				"	vec4 pDst = texture2D(texDst, texCoord);",
				"	vec4 pSrc = texture2D(texSrc, texCoord);",
				"",
				"	float da = pDst.a;",
				"	float sa = pSrc.a*opacity;",
//				"	float cda = 1.0 - da;",
				"	float csa = 1.0 - sa;",
				"	float a = sa + da*csa;",
				"",
				"	vec3 pDst3 = pDst.rgb;",
				"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
				"",
		  hue ? "	vec3 newDst3 = set_lum(set_sat(hsl2rgb(vec3(pSrc.r, 1.0, 0.5)), sat(uDst3)), dot(uDst3, lumaVec));"
			  : "	vec3 newDst3 = set_lum(set_sat(uDst3, pSrc.r), dot(uDst3, lumaVec));",
				"",
				"	vec4 blend = vec4(newDst3*da*sa + pDst3*csa, a);",
				"	if (blend.a != 0.0) {",
				"		gl_FragColor = blend/blend.a * da;",
				"	} else {",
				"		gl_FragColor = vec4(0.0);",
				"	}",
				"}"
		};
	};

	@ShaderSource(attach={"ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions", "hsl2rgb"})
	public static final String[] HUE = createHueOrSaturationBlendSource(true);

	@ShaderSource(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] SATURATION = createHueOrSaturationBlendSource(false);

}
